package physx.extensions;


/**
 * a joint that maintains an upper or lower bound (or both) on the distance between two points on different objects
 * @see PxJoint
 */
public class PxDistanceJoint extends PxJoint {

    protected PxDistanceJoint() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxDistanceJoint wrapPointer(long address) {
        return address != 0L ? new PxDistanceJoint(address) : null;
    }
    
    public static PxDistanceJoint arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxDistanceJoint(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Return the current distance of the joint
     */
    public float getDistance() {
        checkNotNull();
        return _getDistance(address);
    }
    private static native float _getDistance(long address);

    /**
     * Set the allowed minimum distance for the joint.
     * <p>
     * The minimum distance must be no more than the maximum distance
     * <p>
     * <b>Default</b> 0.0f
     * <b>Range</b> [0, PX_MAX_F32)
     * @param distance the minimum distance
     * @see #getMinDistance
     */
    public void setMinDistance(float distance) {
        checkNotNull();
        _setMinDistance(address, distance);
    }
    private static native void _setMinDistance(long address, float distance);

    /**
     * Get the allowed minimum distance for the joint.
     * @return the allowed minimum distance
     * @see #setMinDistance
     */
    public float getMinDistance() {
        checkNotNull();
        return _getMinDistance(address);
    }
    private static native float _getMinDistance(long address);

    /**
     * Set the allowed maximum distance for the joint.
     * <p>
     * The maximum distance must be no less than the minimum distance. 
     * <p>
     * <b>Default</b> 0.0f
     * <b>Range</b> [0, PX_MAX_F32)
     * @param distance the maximum distance
     * @see #getMinDistance
     */
    public void setMaxDistance(float distance) {
        checkNotNull();
        _setMaxDistance(address, distance);
    }
    private static native void _setMaxDistance(long address, float distance);

    /**
     * Get the allowed maximum distance for the joint.
     * @return the allowed maximum distance
     * @see #setMaxDistance
     */
    public float getMaxDistance() {
        checkNotNull();
        return _getMaxDistance(address);
    }
    private static native float _getMaxDistance(long address);

    /**
     * Set the error tolerance of the joint.
     * @param tolerance the distance beyond the allowed range at which the joint becomes active
     * @see #getTolerance
     */
    public void setTolerance(float tolerance) {
        checkNotNull();
        _setTolerance(address, tolerance);
    }
    private static native void _setTolerance(long address, float tolerance);

    /**
     * Get the error tolerance of the joint.
     * <p>
     * the distance beyond the joint's [min, max] range before the joint becomes active.
     * <p>
     * <b>Default</b> 0.25f * PxTolerancesScale::length
     * <b>Range</b> (0, PX_MAX_F32)
     * <p>
     * This value should be used to ensure that if the minimum distance is zero and the 
     * spring function is in use, the rest length of the spring is non-zero. 
     * @see #setTolerance
     */
    public float getTolerance() {
        checkNotNull();
        return _getTolerance(address);
    }
    private static native float _getTolerance(long address);

    /**
     * Set the strength of the joint spring.
     * <p>
     * The spring is used if enabled, and the distance exceeds the range [min-error, max+error].
     * <p>
     * <b>Default</b> 0.0f
     * <b>Range</b> [0, PX_MAX_F32)
     * @param stiffness the spring strength of the joint
     * @see #getStiffness
     */
    public void setStiffness(float stiffness) {
        checkNotNull();
        _setStiffness(address, stiffness);
    }
    private static native void _setStiffness(long address, float stiffness);

    /**
     * Get the strength of the joint spring.
     * @return stiffness the spring strength of the joint
     * @see #setStiffness
     */
    public float getStiffness() {
        checkNotNull();
        return _getStiffness(address);
    }
    private static native float _getStiffness(long address);

    /**
     * Set the damping of the joint spring.
     * <p>
     * The spring is used if enabled, and the distance exceeds the range [min-error, max+error].
     * <p>
     * <b>Default</b> 0.0f
     * <b>Range</b> [0, PX_MAX_F32)
     * @param damping the degree of damping of the joint spring of the joint
     * @see #setDamping
     */
    public void setDamping(float damping) {
        checkNotNull();
        _setDamping(address, damping);
    }
    private static native void _setDamping(long address, float damping);

    /**
     * Get the damping of the joint spring.
     * @return the degree of damping of the joint spring of the joint
     * @see #setDamping
     */
    public float getDamping() {
        checkNotNull();
        return _getDamping(address);
    }
    private static native float _getDamping(long address);

    /**
     * Set the flags specific to the Distance Joint.
     * <p>
     * <b>Default</b> PxDistanceJointFlag::eMAX_DISTANCE_ENABLED
     * @param flags The joint flags.
     */
    public void setDistanceJointFlags(PxDistanceJointFlags flags) {
        checkNotNull();
        _setDistanceJointFlags(address, flags.getAddress());
    }
    private static native void _setDistanceJointFlags(long address, long flags);

    /**
     * Set a single flag specific to a Distance Joint to true or false.
     * @param flag The flag to set or clear.
     * @param value the value to which to set the flag
     */
    public void setDistanceJointFlag(PxDistanceJointFlagEnum flag, boolean value) {
        checkNotNull();
        _setDistanceJointFlag(address, flag.value, value);
    }
    private static native void _setDistanceJointFlag(long address, int flag, boolean value);

    /**
     * Get the flags specific to the Distance Joint.
     * @return the joint flags
     */
    public PxDistanceJointFlags getDistanceJointFlags() {
        checkNotNull();
        return PxDistanceJointFlags.wrapPointer(_getDistanceJointFlags(address));
    }
    private static native long _getDistanceJointFlags(long address);

}
