package physx.common;

import physx.NativeObject;
import physx.PlatformChecks;

/**
 * * Manages thread locks, and task scheduling for a CUDA context
 * *
 * * A PxCudaContextManager manages access to a single CUDA context, allowing it to
 * * be shared between multiple scenes.
 * * The context must be acquired from the manager before using any CUDA APIs unless stated differently.
 * *
 * * The PxCudaContextManager is based on the CUDA driver API and explicitly does not
 * * support the CUDA runtime API (aka, CUDART).
 */
public class PxCudaContextManager extends NativeObject {

    static {
        PlatformChecks.requirePlatform(3, "physx.common.PxCudaContextManager");
    }

    protected PxCudaContextManager() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxCudaContextManager wrapPointer(long address) {
        return address != 0L ? new PxCudaContextManager(address) : null;
    }
    
    public static PxCudaContextManager arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxCudaContextManager(long address) {
        super(address);
    }

    // Functions

    /**
     * * Acquire the CUDA context for the current thread
     * *
     * * Acquisitions are allowed to be recursive within a single thread.
     * * You can acquire the context multiple times so long as you release
     * * it the same count.
     * *
     * * The context must be acquired before using most CUDA functions.
     */
    public void acquireContext() {
        checkNotNull();
        _acquireContext(address);
    }
    private static native void _acquireContext(long address);

    /**
     * * Release the CUDA context from the current thread
     * *
     * * The CUDA context should be released as soon as practically
     * * possible, to allow other CPU threads to work efficiently.
     */
    public void releaseContext() {
        checkNotNull();
        _releaseContext(address);
    }
    private static native void _releaseContext(long address);

    /**
     * * Return the CUcontext
     */
    public CUcontext getContext() {
        checkNotNull();
        return CUcontext.wrapPointer(_getContext(address));
    }
    private static native long _getContext(long address);

    /**
     * * Return the CudaContext
     */
    public PxCudaContext getCudaContext() {
        checkNotNull();
        return PxCudaContext.wrapPointer(_getCudaContext(address));
    }
    private static native long _getCudaContext(long address);

    /**
     * * Context manager has a valid CUDA context
     * *
     * * This method should be called after creating a PxCudaContextManager,
     * * especially if the manager was responsible for allocating its own
     * * CUDA context (desc.ctx == NULL).
     */
    public boolean contextIsValid() {
        checkNotNull();
        return _contextIsValid(address);
    }
    private static native boolean _contextIsValid(long address);

    /**
     * G80
     */
    public boolean supportsArchSM10() {
        checkNotNull();
        return _supportsArchSM10(address);
    }
    private static native boolean _supportsArchSM10(long address);

    /**
     * G92
     */
    public boolean supportsArchSM11() {
        checkNotNull();
        return _supportsArchSM11(address);
    }
    private static native boolean _supportsArchSM11(long address);

    /**
     * GT200
     */
    public boolean supportsArchSM12() {
        checkNotNull();
        return _supportsArchSM12(address);
    }
    private static native boolean _supportsArchSM12(long address);

    /**
     * GT260
     */
    public boolean supportsArchSM13() {
        checkNotNull();
        return _supportsArchSM13(address);
    }
    private static native boolean _supportsArchSM13(long address);

    /**
     * GF100
     */
    public boolean supportsArchSM20() {
        checkNotNull();
        return _supportsArchSM20(address);
    }
    private static native boolean _supportsArchSM20(long address);

    /**
     * GK100
     */
    public boolean supportsArchSM30() {
        checkNotNull();
        return _supportsArchSM30(address);
    }
    private static native boolean _supportsArchSM30(long address);

    /**
     * GK110
     */
    public boolean supportsArchSM35() {
        checkNotNull();
        return _supportsArchSM35(address);
    }
    private static native boolean _supportsArchSM35(long address);

    /**
     * GM100
     */
    public boolean supportsArchSM50() {
        checkNotNull();
        return _supportsArchSM50(address);
    }
    private static native boolean _supportsArchSM50(long address);

    /**
     * GM200
     */
    public boolean supportsArchSM52() {
        checkNotNull();
        return _supportsArchSM52(address);
    }
    private static native boolean _supportsArchSM52(long address);

    /**
     * GP100
     */
    public boolean supportsArchSM60() {
        checkNotNull();
        return _supportsArchSM60(address);
    }
    private static native boolean _supportsArchSM60(long address);

    /**
     * true if GPU is an integrated (MCP) part
     */
    public boolean isIntegrated() {
        checkNotNull();
        return _isIntegrated(address);
    }
    private static native boolean _isIntegrated(long address);

    /**
     * true if GPU map host memory to GPU (0-copy)
     */
    public boolean canMapHostMemory() {
        checkNotNull();
        return _canMapHostMemory(address);
    }
    private static native boolean _canMapHostMemory(long address);

    /**
     * returns cached value of cuGetDriverVersion()
     */
    public int getDriverVersion() {
        checkNotNull();
        return _getDriverVersion(address);
    }
    private static native int _getDriverVersion(long address);

    /**
     * returns cached value of device memory size
     */
    public long getDeviceTotalMemBytes() {
        checkNotNull();
        return _getDeviceTotalMemBytes(address);
    }
    private static native long _getDeviceTotalMemBytes(long address);

    /**
     * returns cache value of SM unit count
     */
    public int getMultiprocessorCount() {
        checkNotNull();
        return _getMultiprocessorCount(address);
    }
    private static native int _getMultiprocessorCount(long address);

    /**
     * returns cached value of SM clock frequency
     */
    public int getClockRate() {
        checkNotNull();
        return _getClockRate(address);
    }
    private static native int _getClockRate(long address);

    /**
     * returns total amount of shared memory available per block in bytes
     */
    public int getSharedMemPerBlock() {
        checkNotNull();
        return _getSharedMemPerBlock(address);
    }
    private static native int _getSharedMemPerBlock(long address);

    /**
     * returns total amount of shared memory available per multiprocessor in bytes
     */
    public int getSharedMemPerMultiprocessor() {
        checkNotNull();
        return _getSharedMemPerMultiprocessor(address);
    }
    private static native int _getSharedMemPerMultiprocessor(long address);

    /**
     * returns the maximum number of threads per block
     */
    public int getMaxThreadsPerBlock() {
        checkNotNull();
        return _getMaxThreadsPerBlock(address);
    }
    private static native int _getMaxThreadsPerBlock(long address);

    /**
     * @return WebIDL type: DOMString [Const]
     */
    public String getDeviceName() {
        checkNotNull();
        return _getDeviceName(address);
    }
    private static native String _getDeviceName(long address);

    /**
     * returns device handle retrieved from driver
     */
    public CUdevice getDevice() {
        checkNotNull();
        return CUdevice.wrapPointer(_getDevice(address));
    }
    private static native long _getDevice(long address);

    /**
     * interop mode the context was created with
     */
    public PxCudaInteropModeEnum getInteropMode() {
        checkNotNull();
        return PxCudaInteropModeEnum.forValue(_getInteropMode(address));
    }
    private static native int _getInteropMode(long address);

    /**
     * turn on/off using concurrent streams for GPU work
     */
    public void setUsingConcurrentStreams(boolean flag) {
        checkNotNull();
        _setUsingConcurrentStreams(address, flag);
    }
    private static native void _setUsingConcurrentStreams(long address, boolean flag);

    /**
     * true if GPU work can run in concurrent streams
     */
    public boolean getUsingConcurrentStreams() {
        checkNotNull();
        return _getUsingConcurrentStreams(address);
    }
    private static native boolean _getUsingConcurrentStreams(long address);

    /**
     * * Determine if the user has configured a dedicated PhysX GPU in the NV Control Panel
     * * <b>Note:</b> If using CUDA Interop, this will always return false
     * * \returns 1 if there is a dedicated GPU
     * *   0 if there is NOT a dedicated GPU
     * *   -1 if the routine is not implemented
     */
    public int usingDedicatedGPU() {
        checkNotNull();
        return _usingDedicatedGPU(address);
    }
    private static native int _usingDedicatedGPU(long address);

    /**
     * * Get the cuda modules that have been loaded into this context on construction
     * * @return Pointer to the cuda modules
     */
    public CUmodule getCuModules() {
        checkNotNull();
        return CUmodule.wrapPointer(_getCuModules(address));
    }
    private static native long _getCuModules(long address);

    /**
     *    * Release the PxCudaContextManager
     *    *
     *    * If the PxCudaContextManager created the CUDA context it was 
     * * responsible for, it also frees that context.
     *    *
     *    * Do not release the PxCudaContextManager if there are any scenes
     *    * using it.  Those scenes must be released first.
     *    *
     */
    public void release() {
        checkNotNull();
        _release(address);
    }
    private static native void _release(long address);

}
