package physx.physics;

/**
 * Collection of flags describing the behavior of a rigid body.
 * <p>
 * <b>See also:</b> PxRigidBody.setRigidBodyFlag(), PxRigidBody.getRigidBodyFlags()
 */
public enum PxRigidBodyFlagEnum {

    /**
     * Enables kinematic mode for the actor.
     * <p>
     * Kinematic actors are special dynamic actors that are not 
     * influenced by forces (such as gravity), and have no momentum. They are considered to have infinite
     * mass and can be moved around the world using the setKinematicTarget() method. They will push 
     * regular dynamic actors out of the way. Kinematics will not collide with static or other kinematic objects.
     * <p>
     * Kinematic actors are great for moving platforms or characters, where direct motion control is desired.
     * <p>
     * You can not connect Reduced joints to kinematic actors. Lagrange joints work ok if the platform
     * is moving with a relatively low, uniform velocity.
     * <p>
     * <b>Sleeping:</b>
     * \li Setting this flag on a dynamic actor will put the actor to sleep and set the velocities to 0.
     * \li If this flag gets cleared, the current sleep state of the actor will be kept.
     * <p>
     * <b>Note:</b> kinematic actors are incompatible with CCD so raising this flag will automatically clear eENABLE_CCD
     * <p>
     * <b>See also:</b> PxRigidDynamic.setKinematicTarget()
     */
    eKINEMATIC(geteKINEMATIC()),
    /**
     * Use the kinematic target transform for scene queries.
     * <p>
     * If this flag is raised, then scene queries will treat the kinematic target transform as the current pose
     * of the body (instead of using the actual pose). Without this flag, the kinematic target will only take 
     * effect with respect to scene queries after a simulation step.
     * <p>
     * <b>See also:</b> PxRigidDynamic.setKinematicTarget()
     */
    eUSE_KINEMATIC_TARGET_FOR_SCENE_QUERIES(geteUSE_KINEMATIC_TARGET_FOR_SCENE_QUERIES()),
    /**
     * Enables swept integration for the actor.
     * <p>
     * If this flag is raised and CCD is enabled on the scene, then this body will be simulated by the CCD system to ensure that collisions are not missed due to 
     * high-speed motion. Note individual shape pairs still need to enable PxPairFlag::eDETECT_CCD_CONTACT in the collision filtering to enable the CCD to respond to 
     * individual interactions. 
     * <p>
     * <b>Note:</b> kinematic actors are incompatible with CCD so this flag will be cleared automatically when raised on a kinematic actor
     */
    eENABLE_CCD(geteENABLE_CCD()),
    /**
     * Enabled CCD in swept integration for the actor.
     * <p>
     * If this flag is raised and CCD is enabled, CCD interactions will simulate friction. By default, friction is disabled in CCD interactions because 
     * CCD friction has been observed to introduce some simulation artifacts. CCD friction was enabled in previous versions of the SDK. Raising this flag will result in behavior 
     * that is a closer match for previous versions of the SDK.
     * <p>
     * <b>Note:</b> This flag requires PxRigidBodyFlag::eENABLE_CCD to be raised to have any effect.
     */
    eENABLE_CCD_FRICTION(geteENABLE_CCD_FRICTION()),
    /**
     * Register a rigid body for reporting pose changes by the simulation at an early stage.
     * <p>
     * Sometimes it might be advantageous to get access to the new pose of a rigid body as early as possible and
     * not wait until the call to fetchResults() returns. Setting this flag will schedule the rigid body to get reported
     * in #PxSimulationEventCallback::onAdvance(). Please refer to the documentation of that callback to understand
     * the behavior and limitations of this functionality.
     * <p>
     * <b>See also:</b> PxSimulationEventCallback::onAdvance()
     */
    eENABLE_POSE_INTEGRATION_PREVIEW(geteENABLE_POSE_INTEGRATION_PREVIEW()),
    /**
     * Register a rigid body to dynamically adjust contact offset based on velocity. This can be used to achieve a CCD effect.
     * <p>
     * If both eENABLE_CCD and eENABLE_SPECULATIVE_CCD are set on the same body, then angular motions are handled by speculative
     * contacts (eENABLE_SPECULATIVE_CCD) while linear motions are handled by sweeps (eENABLE_CCD).
     */
    eENABLE_SPECULATIVE_CCD(geteENABLE_SPECULATIVE_CCD()),
    /**
     * Permit CCD to limit maxContactImpulse. This is useful for use-cases like a destruction system but can cause visual artefacts so is not enabled by default.
     */
    eENABLE_CCD_MAX_CONTACT_IMPULSE(geteENABLE_CCD_MAX_CONTACT_IMPULSE()),
    /**
     * Carries over forces/accelerations between frames, rather than clearing them
     */
    eRETAIN_ACCELERATIONS(geteRETAIN_ACCELERATIONS());
    public final int value;
    
    PxRigidBodyFlagEnum(int value) {
        this.value = value;
    }

    private static native int _geteKINEMATIC();
    private static int geteKINEMATIC() {
        de.fabmax.physxjni.Loader.load();
        return _geteKINEMATIC();
    }

    private static native int _geteUSE_KINEMATIC_TARGET_FOR_SCENE_QUERIES();
    private static int geteUSE_KINEMATIC_TARGET_FOR_SCENE_QUERIES() {
        de.fabmax.physxjni.Loader.load();
        return _geteUSE_KINEMATIC_TARGET_FOR_SCENE_QUERIES();
    }

    private static native int _geteENABLE_CCD();
    private static int geteENABLE_CCD() {
        de.fabmax.physxjni.Loader.load();
        return _geteENABLE_CCD();
    }

    private static native int _geteENABLE_CCD_FRICTION();
    private static int geteENABLE_CCD_FRICTION() {
        de.fabmax.physxjni.Loader.load();
        return _geteENABLE_CCD_FRICTION();
    }

    private static native int _geteENABLE_POSE_INTEGRATION_PREVIEW();
    private static int geteENABLE_POSE_INTEGRATION_PREVIEW() {
        de.fabmax.physxjni.Loader.load();
        return _geteENABLE_POSE_INTEGRATION_PREVIEW();
    }

    private static native int _geteENABLE_SPECULATIVE_CCD();
    private static int geteENABLE_SPECULATIVE_CCD() {
        de.fabmax.physxjni.Loader.load();
        return _geteENABLE_SPECULATIVE_CCD();
    }

    private static native int _geteENABLE_CCD_MAX_CONTACT_IMPULSE();
    private static int geteENABLE_CCD_MAX_CONTACT_IMPULSE() {
        de.fabmax.physxjni.Loader.load();
        return _geteENABLE_CCD_MAX_CONTACT_IMPULSE();
    }

    private static native int _geteRETAIN_ACCELERATIONS();
    private static int geteRETAIN_ACCELERATIONS() {
        de.fabmax.physxjni.Loader.load();
        return _geteRETAIN_ACCELERATIONS();
    }

    public static PxRigidBodyFlagEnum forValue(int value) {
        for (int i = 0; i < values().length; i++) {
            if (values()[i].value == value) {
                return values()[i];
            }
        }
        throw new IllegalArgumentException("Unknown value for enum PxRigidBodyFlagEnum: " + value);
    }

}
