package physx.physics;

import physx.common.PxTransform;

/**
 * PxRigidActor represents a base class shared between dynamic and static rigid bodies in the physics SDK.
 * <p>
 * PxRigidActor objects specify the geometry of the object by defining a set of attached shapes (see #PxShape).
 * @see PxActor
 */
public class PxRigidActor extends PxActor {

    protected PxRigidActor() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxRigidActor wrapPointer(long address) {
        return address != 0L ? new PxRigidActor(address) : null;
    }
    
    public static PxRigidActor arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxRigidActor(long address) {
        super(address);
    }

    // Functions

    /**
     * Retrieves the actors world space transform.
     * <p>
     * The getGlobalPose() method retrieves the actor's current actor space to world space transformation.
     * <p>
     * <b>Note:</b> It is not allowed to use this method while the simulation is running (except during PxScene::collide(),
     * in PxContactModifyCallback or in contact report callbacks).
     * @return Global pose of object.
     */
    public PxTransform getGlobalPose() {
        checkNotNull();
        return PxTransform.wrapPointer(_getGlobalPose(address));
    }
    private static native long _getGlobalPose(long address);

    /**
     * Method for setting an actor's pose in the world.
     * <p>
     * This method instantaneously changes the actor space to world space transformation.
     * <p>
     * This method is mainly for dynamic rigid bodies (see #PxRigidDynamic). Calling this method on static actors is
     * likely to result in a performance penalty, since internal optimization structures for static actors may need to be
     * recomputed. In addition, moving static actors will not interact correctly with dynamic actors or joints.
     * <p>
     * To directly control an actor's position and have it correctly interact with dynamic bodies and joints, create a dynamic
     * body with the PxRigidBodyFlag::eKINEMATIC flag, then use the setKinematicTarget() commands to define its path.
     * <p>
     * Even when moving dynamic actors, exercise restraint in making use of this method. Where possible, avoid:
     * <p>
     * \li moving actors into other actors, thus causing overlap (an invalid physical state)
     * <p>
     * \li moving an actor that is connected by a joint to another away from the other (thus causing joint error)
     * <p>
     * <b>Sleeping:</b> This call wakes dynamic actors if they are sleeping and the autowake parameter is true (default).
     * @param pose Transformation from the actors local frame to the global frame. <b>Range:</b> rigid body transform.
     * @see #getGlobalPose
     */
    public void setGlobalPose(PxTransform pose) {
        checkNotNull();
        _setGlobalPose(address, pose.getAddress());
    }
    private static native void _setGlobalPose(long address, long pose);

    /**
     * Method for setting an actor's pose in the world.
     * <p>
     * This method instantaneously changes the actor space to world space transformation.
     * <p>
     * This method is mainly for dynamic rigid bodies (see #PxRigidDynamic). Calling this method on static actors is
     * likely to result in a performance penalty, since internal optimization structures for static actors may need to be
     * recomputed. In addition, moving static actors will not interact correctly with dynamic actors or joints.
     * <p>
     * To directly control an actor's position and have it correctly interact with dynamic bodies and joints, create a dynamic
     * body with the PxRigidBodyFlag::eKINEMATIC flag, then use the setKinematicTarget() commands to define its path.
     * <p>
     * Even when moving dynamic actors, exercise restraint in making use of this method. Where possible, avoid:
     * <p>
     * \li moving actors into other actors, thus causing overlap (an invalid physical state)
     * <p>
     * \li moving an actor that is connected by a joint to another away from the other (thus causing joint error)
     * <p>
     * <b>Sleeping:</b> This call wakes dynamic actors if they are sleeping and the autowake parameter is true (default).
     * @param pose Transformation from the actors local frame to the global frame. <b>Range:</b> rigid body transform.
     * @param autowake whether to wake the object if it is dynamic. This parameter has no effect for static or kinematic actors. If true and the current wake counter value is smaller than #PxSceneDesc::wakeCounterResetValue it will get increased to the reset value.
     * @see #getGlobalPose
     */
    public void setGlobalPose(PxTransform pose, boolean autowake) {
        checkNotNull();
        _setGlobalPose(address, pose.getAddress(), autowake);
    }
    private static native void _setGlobalPose(long address, long pose, boolean autowake);

    /**
     * Attach a shape to an actor 
     * <p>
     * This call will increment the reference count of the shape.
     * <p>
     * <b>Note:</b> Mass properties of dynamic rigid actors will not automatically be recomputed 
     * to reflect the new mass distribution implied by the shape. Follow this call with a call to 
     * the PhysX extensions method #PxRigidBodyExt::updateMassAndInertia() to do that.
     * <p>
     * Attaching a triangle mesh, heightfield or plane geometry shape configured as eSIMULATION_SHAPE is not supported for 
     * non-kinematic PxRigidDynamic instances.
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake the actor up automatically.
     * @param shape the shape to attach.
     * @return True if success.
     */
    public boolean attachShape(PxShape shape) {
        checkNotNull();
        return _attachShape(address, shape.getAddress());
    }
    private static native boolean _attachShape(long address, long shape);

    /**
     * Detach a shape from an actor. 
     * <p>
     * This will also decrement the reference count of the PxShape, and if the reference count is zero, will cause it to be deleted.
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake the actor up automatically.
     * @param shape the shape to detach.
     */
    public void detachShape(PxShape shape) {
        checkNotNull();
        _detachShape(address, shape.getAddress());
    }
    private static native void _detachShape(long address, long shape);

    /**
     * Detach a shape from an actor. 
     * <p>
     * This will also decrement the reference count of the PxShape, and if the reference count is zero, will cause it to be deleted.
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake the actor up automatically.
     * @param shape the shape to detach.
     * @param wakeOnLostTouch Specifies whether touching objects from the previous frame should get woken up in the next frame. Only applies to PxArticulationReducedCoordinate and PxRigidActor types.
     */
    public void detachShape(PxShape shape, boolean wakeOnLostTouch) {
        checkNotNull();
        _detachShape(address, shape.getAddress(), wakeOnLostTouch);
    }
    private static native void _detachShape(long address, long shape, boolean wakeOnLostTouch);

    /**
     * Returns the number of shapes assigned to the actor.
     * <p>
     * You can use #getShapes() to retrieve the shape pointers.
     * @return Number of shapes associated with this actor.
     * @see PxShape
     */
    public int getNbShapes() {
        checkNotNull();
        return _getNbShapes(address);
    }
    private static native int _getNbShapes(long address);

    /**
     * Returns the number of constraint shaders attached to the actor.
     * <p>
     * You can use #getConstraints() to retrieve the constraint shader pointers.
     * @return Number of constraint shaders attached to this actor.
     * @see PxConstraint
     */
    public int getNbConstraints() {
        checkNotNull();
        return _getNbConstraints(address);
    }
    private static native int _getNbConstraints(long address);

}
