package physx.physics;

import physx.NativeObject;

/**
 * Material class to represent a set of surface properties. 
 * @see PxPhysics#createMaterial
 */
public class PxMaterial extends PxBaseMaterial {

    protected PxMaterial() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxMaterial wrapPointer(long address) {
        return address != 0L ? new PxMaterial(address) : null;
    }
    
    public static PxMaterial arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxMaterial(long address) {
        super(address);
    }

    // Attributes

    /**
     * @return WebIDL type: VoidPtr
     */
    public NativeObject getUserData() {
        checkNotNull();
        return NativeObject.wrapPointer(_getUserData(address));
    }
    private static native long _getUserData(long address);

    /**
     * @param value WebIDL type: VoidPtr
     */
    public void setUserData(NativeObject value) {
        checkNotNull();
        _setUserData(address, value.getAddress());
    }
    private static native void _setUserData(long address, long value);

    // Functions

    /**
     * Sets the coefficient of dynamic friction.
     * <p>
     * The coefficient of dynamic friction should be in [0, PX_MAX_F32). If set to greater than staticFriction, the effective value of staticFriction will be increased to match.
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake any actors which may be affected.
     * @param coef Coefficient of dynamic friction. <b>Range:</b> [0, PX_MAX_F32)
     * @see #getDynamicFriction
     */
    public void setDynamicFriction(float coef) {
        checkNotNull();
        _setDynamicFriction(address, coef);
    }
    private static native void _setDynamicFriction(long address, float coef);

    /**
     * Retrieves the DynamicFriction value.
     * @return The coefficient of dynamic friction.
     * @see #setDynamicFriction
     */
    public float getDynamicFriction() {
        checkNotNull();
        return _getDynamicFriction(address);
    }
    private static native float _getDynamicFriction(long address);

    /**
     * Sets the coefficient of static friction
     * <p>
     * The coefficient of static friction should be in the range [0, PX_MAX_F32)
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake any actors which may be affected.
     * @param coef Coefficient of static friction. <b>Range:</b> [0, PX_MAX_F32)
     * @see #getStaticFriction
     */
    public void setStaticFriction(float coef) {
        checkNotNull();
        _setStaticFriction(address, coef);
    }
    private static native void _setStaticFriction(long address, float coef);

    /**
     * Retrieves the coefficient of static friction.
     * @return The coefficient of static friction.
     * @see #setStaticFriction
     */
    public float getStaticFriction() {
        checkNotNull();
        return _getStaticFriction(address);
    }
    private static native float _getStaticFriction(long address);

    /**
     * Sets the coefficient of restitution 
     * <p>
     * A coefficient of 0 makes the object bounce as little as possible, higher values up to 1.0 result in more bounce.
     * <p>
     * This property is overloaded when PxMaterialFlag::eCOMPLIANT_CONTACT flag is enabled. This permits negative values for restitution to be provided.
     * The negative values are converted into spring stiffness terms for an implicit spring simulated at the contact site, with the spring positional error defined by
     * the contact separation value. Higher stiffness terms produce stiffer springs that behave more like a rigid contact.
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake any actors which may be affected.
     * @see #getRestitution
     */
    public void setRestitution(float coef) {
        checkNotNull();
        _setRestitution(address, coef);
    }
    private static native void _setRestitution(long address, float coef);

    /**
     * Retrieves the coefficient of restitution.     
     * <p>
     * See #setRestitution.
     * @return The coefficient of restitution.
     * @see #setRestitution
     */
    public float getRestitution() {
        checkNotNull();
        return _getRestitution(address);
    }
    private static native float _getRestitution(long address);

    /**
     * Raises or clears a particular material flag.
     * <p>
     * See the list of flags #PxMaterialFlag
     * <p>
     * <b>Default:</b> eIMPROVED_PATCH_FRICTION
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake any actors which may be affected.
     * @param flag The PxMaterial flag to raise(set) or clear.
     * @param b  New state of the flag
     * @see #getFlags
     * @see #setFlags
     */
    public void setFlag(PxMaterialFlagEnum flag, boolean b) {
        checkNotNull();
        _setFlag(address, flag.value, b);
    }
    private static native void _setFlag(long address, int flag, boolean b);

    /**
     * sets all the material flags.
     * <p>
     * See the list of flags #PxMaterialFlag
     * <p>
     * <b>Default:</b> eIMPROVED_PATCH_FRICTION
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake any actors which may be affected.
     * @param flags All PxMaterial flags
     * @see #getFlags
     * @see #setFlag
     */
    public void setFlags(PxMaterialFlags flags) {
        checkNotNull();
        _setFlags(address, flags.getAddress());
    }
    private static native void _setFlags(long address, long flags);

    /**
     * Retrieves the flags. See #PxMaterialFlag.
     * @return The material flags.
     * @see #setFlags
     */
    public PxMaterialFlags getFlags() {
        checkNotNull();
        return PxMaterialFlags.wrapPointer(_getFlags(address));
    }
    private static native long _getFlags(long address);

    /**
     * Sets the friction combine mode.
     * <p>
     * See the enum ::PxCombineMode .
     * <p>
     * <b>Default:</b> PxCombineMode::eAVERAGE
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake any actors which may be affected.
     * @param combMode Friction combine mode to set for this material. See #PxCombineMode.
     * @see #getFrictionCombineMode
     * @see #setStaticFriction
     * @see #setDynamicFriction
     */
    public void setFrictionCombineMode(PxCombineModeEnum combMode) {
        checkNotNull();
        _setFrictionCombineMode(address, combMode.value);
    }
    private static native void _setFrictionCombineMode(long address, int combMode);

    /**
     * Retrieves the friction combine mode.
     * <p>
     * See #setFrictionCombineMode.
     * @return The friction combine mode for this material.
     * @see #setFrictionCombineMode
     */
    public PxCombineModeEnum getFrictionCombineMode() {
        checkNotNull();
        return PxCombineModeEnum.forValue(_getFrictionCombineMode(address));
    }
    private static native int _getFrictionCombineMode(long address);

    /**
     * Sets the restitution combine mode.
     * <p>
     * See the enum ::PxCombineMode .
     * <p>
     * <b>Default:</b> PxCombineMode::eAVERAGE
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake any actors which may be affected.
     * @param combMode Restitution combine mode for this material. See #PxCombineMode.
     * @see #getRestitutionCombineMode
     * @see #setRestitution
     */
    public void setRestitutionCombineMode(PxCombineModeEnum combMode) {
        checkNotNull();
        _setRestitutionCombineMode(address, combMode.value);
    }
    private static native void _setRestitutionCombineMode(long address, int combMode);

    /**
     * Retrieves the restitution combine mode.
     * <p>
     * See #setRestitutionCombineMode.
     * @return The coefficient of restitution combine mode for this material.
     * @see #setRestitutionCombineMode
     * @see #getRestitution
     */
    public PxCombineModeEnum getRestitutionCombineMode() {
        checkNotNull();
        return PxCombineModeEnum.forValue(_getRestitutionCombineMode(address));
    }
    private static native int _getRestitutionCombineMode(long address);

}
