package physx.physics;

import physx.common.PxBase;
import physx.common.PxVec3;

/**
 * A plugin class for implementing constraints
 */
public class PxConstraint extends PxBase {

    protected PxConstraint() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxConstraint wrapPointer(long address) {
        return address != 0L ? new PxConstraint(address) : null;
    }
    
    public static PxConstraint arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxConstraint(long address) {
        super(address);
    }

    // Functions

    /**
     * Retrieves the scene which this constraint belongs to.
     * @return Owner Scene. NULL if not part of a scene.
     * @see PxScene
     */
    public PxScene getScene() {
        checkNotNull();
        return PxScene.wrapPointer(_getScene(address));
    }
    private static native long _getScene(long address);

    /**
     * Sets the actors for this constraint.
     * @param actor0 a reference to the pointer for the first actor
     * @param actor1 a reference to the pointer for the second actor
     * @see PxActor
     */
    public void setActors(PxRigidActor actor0, PxRigidActor actor1) {
        checkNotNull();
        _setActors(address, actor0.getAddress(), actor1.getAddress());
    }
    private static native void _setActors(long address, long actor0, long actor1);

    /**
     * Notify the scene that the constraint shader data has been updated by the application
     */
    public void markDirty() {
        checkNotNull();
        _markDirty(address);
    }
    private static native void _markDirty(long address);

    /**
     * Set the flags for this constraint
     * @param flags the new constraint flags
     * <p>
     * default: PxConstraintFlag::eDRIVE_LIMITS_ARE_FORCES
     * @see PxConstraintFlags
     */
    public void setFlags(PxConstraintFlags flags) {
        checkNotNull();
        _setFlags(address, flags.getAddress());
    }
    private static native void _setFlags(long address, long flags);

    /**
     * Retrieve the flags for this constraint
     * @return the constraint flags
     * @see PxConstraintFlags
     */
    public PxConstraintFlags getFlags() {
        checkNotNull();
        return PxConstraintFlags.wrapPointer(_getFlags(address));
    }
    private static native long _getFlags(long address);

    /**
     * Set a flag for this constraint
     * @param flag the constraint flag
     * @param value the new value of the flag
     * @see PxConstraintFlags
     */
    public void setFlag(PxConstraintFlagEnum flag, boolean value) {
        checkNotNull();
        _setFlag(address, flag.value, value);
    }
    private static native void _setFlag(long address, int flag, boolean value);

    /**
     * Retrieve the constraint force most recently applied to maintain this constraint.
     * <p>
     * <b>Note:</b> It is not allowed to use this method while the simulation is running (except during PxScene::collide(),
     * in PxContactModifyCallback or in contact report callbacks).
     * @param linear the constraint force
     * @param angular the constraint torque
     */
    public void getForce(PxVec3 linear, PxVec3 angular) {
        checkNotNull();
        _getForce(address, linear.getAddress(), angular.getAddress());
    }
    private static native void _getForce(long address, long linear, long angular);

    /**
     * whether the constraint is valid. 
     * <p>
     * A constraint is valid if it has at least one dynamic rigid body or articulation link. A constraint that
     * is not valid may not be inserted into a scene, and therefore a static actor to which an invalid constraint
     * is attached may not be inserted into a scene.
     * <p>
     * Invalid constraints arise only when an actor to which the constraint is attached has been deleted.
     */
    public boolean isValid() {
        checkNotNull();
        return _isValid(address);
    }
    private static native boolean _isValid(long address);

    /**
     * Set the break force and torque thresholds for this constraint. 
     * <p>
     * If either the force or torque measured at the constraint exceed these thresholds the constraint will break.
     * @param linear the linear break threshold
     * @param angular the angular break threshold
     */
    public void setBreakForce(float linear, float angular) {
        checkNotNull();
        _setBreakForce(address, linear, angular);
    }
    private static native void _setBreakForce(long address, float linear, float angular);

    /**
     * Set the minimum response threshold for a constraint row 
     * <p>
     * When using mass modification for a joint or infinite inertia for a jointed body, very stiff solver constraints can be generated which 
     * can destabilize simulation. Setting this value to a small positive value (e.g. 1e-8) will cause constraint rows to be ignored if very 
     * large changes in impulses will generate only small changes in velocity. When setting this value, also set 
     * PxConstraintFlag::eDISABLE_PREPROCESSING. The solver accuracy for this joint may be reduced.
     * @param threshold the minimum response threshold
     */
    public void setMinResponseThreshold(float threshold) {
        checkNotNull();
        _setMinResponseThreshold(address, threshold);
    }
    private static native void _setMinResponseThreshold(long address, float threshold);

    /**
     * Retrieve the constraint break force and torque thresholds
     * @return the minimum response threshold for a constraint row
     */
    public float getMinResponseThreshold() {
        checkNotNull();
        return _getMinResponseThreshold(address);
    }
    private static native float _getMinResponseThreshold(long address);

}
