package physx.physics;

import physx.common.PxBase;

/**
 * Common API base class shared by PxArticulationSpatialTendon and PxArticulationFixedTendon.
 */
public class PxArticulationTendon extends PxBase {

    protected PxArticulationTendon() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxArticulationTendon wrapPointer(long address) {
        return address != 0L ? new PxArticulationTendon(address) : null;
    }
    
    public static PxArticulationTendon arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxArticulationTendon(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Sets the spring stiffness term acting on the tendon length.
     * @param stiffness The spring stiffness.
     * <b>Default:</b> 0
     * @see #getStiffness
     */
    public void setStiffness(float stiffness) {
        checkNotNull();
        _setStiffness(address, stiffness);
    }
    private static native void _setStiffness(long address, float stiffness);

    /**
     * Gets the spring stiffness of the tendon.
     * @return The spring stiffness.
     * @see #setStiffness
     */
    public float getStiffness() {
        checkNotNull();
        return _getStiffness(address);
    }
    private static native float _getStiffness(long address);

    /**
     * Sets the damping term acting both on the tendon length and tendon-length limits.
     * @param damping The damping term.
     * <b>Default:</b> 0
     * @see #getDamping
     */
    public void setDamping(float damping) {
        checkNotNull();
        _setDamping(address, damping);
    }
    private static native void _setDamping(long address, float damping);

    /**
     * Gets the damping term acting both on the tendon length and tendon-length limits.
     * @return The damping term.
     * @see #setDamping
     */
    public float getDamping() {
        checkNotNull();
        return _getDamping(address);
    }
    private static native float _getDamping(long address);

    /**
     * Sets the limit stiffness term acting on the tendon's length limits.
     * <p>
     * For spatial tendons, this parameter applies to all its leaf attachments / sub-tendons.
     * @param stiffness The limit stiffness term.
     * <b>Default:</b> 0
     * @see #getLimitStiffness
     */
    public void setLimitStiffness(float stiffness) {
        checkNotNull();
        _setLimitStiffness(address, stiffness);
    }
    private static native void _setLimitStiffness(long address, float stiffness);

    /**
     * Gets the limit stiffness term acting on the tendon's length limits.
     * <p>
     * For spatial tendons, this parameter applies to all its leaf attachments / sub-tendons.
     * @return The limit stiffness term.
     * @see #setLimitStiffness
     */
    public float getLimitStiffness() {
        checkNotNull();
        return _getLimitStiffness(address);
    }
    private static native float _getLimitStiffness(long address);

    /**
     * Sets the length offset term for the tendon.
     * <p>
     * An offset defines an amount to be added to the accumulated length computed for the tendon. It allows the
     * application to actuate the tendon by shortening or lengthening it.
     * @param offset The offset term. <b>Default:</b> 0
     * to #PxSceneDesc::wakeCounterResetValue if the counter value is below the reset value.
     * @see #getOffset
     */
    public void setOffset(float offset) {
        checkNotNull();
        _setOffset(address, offset);
    }
    private static native void _setOffset(long address, float offset);

    /**
     * Sets the length offset term for the tendon.
     * <p>
     * An offset defines an amount to be added to the accumulated length computed for the tendon. It allows the
     * application to actuate the tendon by shortening or lengthening it.
     * @param offset The offset term. <b>Default:</b> 0
     * @param autowake If true and the articulation is in a scene, the call wakes up the articulation and increases the wake counter
     * to #PxSceneDesc::wakeCounterResetValue if the counter value is below the reset value.
     * @see #getOffset
     */
    public void setOffset(float offset, boolean autowake) {
        checkNotNull();
        _setOffset(address, offset, autowake);
    }
    private static native void _setOffset(long address, float offset, boolean autowake);

    /**
     * Gets the length offset term for the tendon.
     * @return The offset term.
     * @see #setOffset
     */
    public float getOffset() {
        checkNotNull();
        return _getOffset(address);
    }
    private static native float _getOffset(long address);

    /**
     * Gets the articulation that the tendon is a part of.
     * @return The articulation.
     * @see PxArticulationReducedCoordinate
     */
    public PxArticulationReducedCoordinate getArticulation() {
        checkNotNull();
        return PxArticulationReducedCoordinate.wrapPointer(_getArticulation(address));
    }
    private static native long _getArticulation(long address);

}
