package physx.geometry;


/**
 * Triangle mesh geometry class.
 * <p>
 * This class unifies a mesh object with a scaling transform, and 
 * lets the combined object be used anywhere a PxGeometry is needed.
 * <p>
 * The scaling is a transform along arbitrary axes contained in the scale object.
 * The vertices of the mesh in geometry (or shape) space is the 
 * PxMeshScale::toMat33() transform, multiplied by the vertex space vertices 
 * in the PxTriangleMeshGeometry object.
 */
public class PxTriangleMeshGeometry extends PxGeometry {

    protected PxTriangleMeshGeometry() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxTriangleMeshGeometry wrapPointer(long address) {
        return address != 0L ? new PxTriangleMeshGeometry(address) : null;
    }
    
    public static PxTriangleMeshGeometry arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxTriangleMeshGeometry(long address) {
        super(address);
    }

    // Placed Constructors

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @param mesh    WebIDL type: {@link PxTriangleMesh}
     * @return Stack allocated object of PxTriangleMeshGeometry
     */
    public static PxTriangleMeshGeometry createAt(long address, PxTriangleMesh mesh) {
        __placement_new_PxTriangleMeshGeometry(address, mesh.getAddress());
        PxTriangleMeshGeometry createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @param mesh      WebIDL type: {@link PxTriangleMesh}
     * @return Stack allocated object of PxTriangleMeshGeometry
     */
    public static <T> PxTriangleMeshGeometry createAt(T allocator, Allocator<T> allocate, PxTriangleMesh mesh) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxTriangleMeshGeometry(address, mesh.getAddress());
        PxTriangleMeshGeometry createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxTriangleMeshGeometry(long address, long mesh);

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @param mesh    WebIDL type: {@link PxTriangleMesh}
     * @param scaling WebIDL type: {@link PxMeshScale} [Const, Ref]
     * @return Stack allocated object of PxTriangleMeshGeometry
     */
    public static PxTriangleMeshGeometry createAt(long address, PxTriangleMesh mesh, PxMeshScale scaling) {
        __placement_new_PxTriangleMeshGeometry(address, mesh.getAddress(), scaling.getAddress());
        PxTriangleMeshGeometry createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @param mesh      WebIDL type: {@link PxTriangleMesh}
     * @param scaling   WebIDL type: {@link PxMeshScale} [Const, Ref]
     * @return Stack allocated object of PxTriangleMeshGeometry
     */
    public static <T> PxTriangleMeshGeometry createAt(T allocator, Allocator<T> allocate, PxTriangleMesh mesh, PxMeshScale scaling) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxTriangleMeshGeometry(address, mesh.getAddress(), scaling.getAddress());
        PxTriangleMeshGeometry createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxTriangleMeshGeometry(long address, long mesh, long scaling);

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @param mesh    WebIDL type: {@link PxTriangleMesh}
     * @param scaling WebIDL type: {@link PxMeshScale} [Const, Ref]
     * @param flags   WebIDL type: {@link PxMeshGeometryFlags} [Ref]
     * @return Stack allocated object of PxTriangleMeshGeometry
     */
    public static PxTriangleMeshGeometry createAt(long address, PxTriangleMesh mesh, PxMeshScale scaling, PxMeshGeometryFlags flags) {
        __placement_new_PxTriangleMeshGeometry(address, mesh.getAddress(), scaling.getAddress(), flags.getAddress());
        PxTriangleMeshGeometry createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @param mesh      WebIDL type: {@link PxTriangleMesh}
     * @param scaling   WebIDL type: {@link PxMeshScale} [Const, Ref]
     * @param flags     WebIDL type: {@link PxMeshGeometryFlags} [Ref]
     * @return Stack allocated object of PxTriangleMeshGeometry
     */
    public static <T> PxTriangleMeshGeometry createAt(T allocator, Allocator<T> allocate, PxTriangleMesh mesh, PxMeshScale scaling, PxMeshGeometryFlags flags) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxTriangleMeshGeometry(address, mesh.getAddress(), scaling.getAddress(), flags.getAddress());
        PxTriangleMeshGeometry createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxTriangleMeshGeometry(long address, long mesh, long scaling, long flags);

    // Constructors

    /**
     * Constructor. By default creates an empty object with a NULL mesh and identity scale.
     * @param mesh  Mesh pointer. May be NULL, though this will not make the object valid for shape construction.
     */
    public PxTriangleMeshGeometry(PxTriangleMesh mesh) {
        address = _PxTriangleMeshGeometry(mesh.getAddress());
    }
    private static native long _PxTriangleMeshGeometry(long mesh);

    /**
     * Constructor. By default creates an empty object with a NULL mesh and identity scale.
     * @param mesh  Mesh pointer. May be NULL, though this will not make the object valid for shape construction.
     * @param scaling Scale factor.
     */
    public PxTriangleMeshGeometry(PxTriangleMesh mesh, PxMeshScale scaling) {
        address = _PxTriangleMeshGeometry(mesh.getAddress(), scaling.getAddress());
    }
    private static native long _PxTriangleMeshGeometry(long mesh, long scaling);

    /**
     * Constructor. By default creates an empty object with a NULL mesh and identity scale.
     * @param mesh  Mesh pointer. May be NULL, though this will not make the object valid for shape construction.
     * @param scaling Scale factor.
     * @param flags Mesh flags.
     */
    public PxTriangleMeshGeometry(PxTriangleMesh mesh, PxMeshScale scaling, PxMeshGeometryFlags flags) {
        address = _PxTriangleMeshGeometry(mesh.getAddress(), scaling.getAddress(), flags.getAddress());
    }
    private static native long _PxTriangleMeshGeometry(long mesh, long scaling, long flags);

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Returns true if the geometry is valid.
     * @return  True if the current settings are valid for shape creation.
     * <p>
     * <b>Note:</b> A valid triangle mesh has a positive scale value in each direction (scale.scale.x &gt; 0, scale.scale.y &gt; 0, scale.scale.z &gt; 0).
     * It is illegal to call PxRigidActor::createShape and PxPhysics::createShape with a triangle mesh that has zero extents in any direction.
     */
    public boolean isValid() {
        checkNotNull();
        return _isValid(address);
    }
    private static native boolean _isValid(long address);

}
