package physx.geometry;

import physx.NativeObject;
import physx.common.PxStridedData;

/**
 * Descriptor class for #PxHeightField.
 * <p>
 * <b>Note:</b> The heightfield data is *copied* when a PxHeightField object is created from this descriptor. After the call the
 * user may discard the height data.
 * @see PxHeightField
 * @see PxHeightFieldGeometry
 * @see physx.physics.PxShape
 * @see physx.cooking.PxCooking#createHeightField
 */
public class PxHeightFieldDesc extends NativeObject {

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxHeightFieldDesc wrapPointer(long address) {
        return address != 0L ? new PxHeightFieldDesc(address) : null;
    }
    
    public static PxHeightFieldDesc arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxHeightFieldDesc(long address) {
        super(address);
    }

    // Placed Constructors

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @return Stack allocated object of PxHeightFieldDesc
     */
    public static PxHeightFieldDesc createAt(long address) {
        __placement_new_PxHeightFieldDesc(address);
        PxHeightFieldDesc createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @return Stack allocated object of PxHeightFieldDesc
     */
    public static <T> PxHeightFieldDesc createAt(T allocator, Allocator<T> allocate) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxHeightFieldDesc(address);
        PxHeightFieldDesc createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxHeightFieldDesc(long address);

    // Constructors

    /**
     * Constructor sets to default.
     */
    public PxHeightFieldDesc() {
        address = _PxHeightFieldDesc();
    }
    private static native long _PxHeightFieldDesc();

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * Number of sample rows in the height field samples array.
     * <p>
     * <b>Note:</b> Local space X-axis corresponds to rows.
     * <p>
     * <b>Range:</b> &amp;gt;1<br>
     * <b>Default:</b> 0
     */
    public int getNbRows() {
        checkNotNull();
        return _getNbRows(address);
    }
    private static native int _getNbRows(long address);

    /**
     * Number of sample rows in the height field samples array.
     * <p>
     * <b>Note:</b> Local space X-axis corresponds to rows.
     * <p>
     * <b>Range:</b> &amp;gt;1<br>
     * <b>Default:</b> 0
     */
    public void setNbRows(int value) {
        checkNotNull();
        _setNbRows(address, value);
    }
    private static native void _setNbRows(long address, int value);

    /**
     * Number of sample columns in the height field samples array.
     * <p>
     * <b>Note:</b> Local space Z-axis corresponds to columns.
     * <p>
     * <b>Range:</b> &amp;gt;1<br>
     * <b>Default:</b> 0
     */
    public int getNbColumns() {
        checkNotNull();
        return _getNbColumns(address);
    }
    private static native int _getNbColumns(long address);

    /**
     * Number of sample columns in the height field samples array.
     * <p>
     * <b>Note:</b> Local space Z-axis corresponds to columns.
     * <p>
     * <b>Range:</b> &amp;gt;1<br>
     * <b>Default:</b> 0
     */
    public void setNbColumns(int value) {
        checkNotNull();
        _setNbColumns(address, value);
    }
    private static native void _setNbColumns(long address, int value);

    /**
     * Format of the sample data.
     * <p>
     * Currently the only supported format is PxHeightFieldFormat::eS16_TM:
     * <p>
     * <b>Default:</b> PxHeightFieldFormat::eS16_TM
     */
    public PxHeightFieldFormatEnum getFormat() {
        checkNotNull();
        return PxHeightFieldFormatEnum.forValue(_getFormat(address));
    }
    private static native int _getFormat(long address);

    /**
     * Format of the sample data.
     * <p>
     * Currently the only supported format is PxHeightFieldFormat::eS16_TM:
     * <p>
     * <b>Default:</b> PxHeightFieldFormat::eS16_TM
     */
    public void setFormat(PxHeightFieldFormatEnum value) {
        checkNotNull();
        _setFormat(address, value.value);
    }
    private static native void _setFormat(long address, int value);

    /**
     * The samples array.
     * <p>
     * It is copied to the SDK's storage at creation time.
     * <p>
     * There are nbRows * nbColumn samples in the array,
     * which define nbRows * nbColumn vertices and cells,
     * of which (nbRows - 1) * (nbColumns - 1) cells are actually used.
     * <p>
     * The array index of sample(row, column) = row * nbColumns + column.
     * The byte offset of sample(row, column) = sampleStride * (row * nbColumns + column).
     * The sample data follows at the offset and spans the number of bytes defined by the format.
     * Then there are zero or more unused bytes depending on sampleStride before the next sample.
     * <p>
     * <b>Default:</b> NULL
     */
    public PxStridedData getSamples() {
        checkNotNull();
        return PxStridedData.wrapPointer(_getSamples(address));
    }
    private static native long _getSamples(long address);

    /**
     * The samples array.
     * <p>
     * It is copied to the SDK's storage at creation time.
     * <p>
     * There are nbRows * nbColumn samples in the array,
     * which define nbRows * nbColumn vertices and cells,
     * of which (nbRows - 1) * (nbColumns - 1) cells are actually used.
     * <p>
     * The array index of sample(row, column) = row * nbColumns + column.
     * The byte offset of sample(row, column) = sampleStride * (row * nbColumns + column).
     * The sample data follows at the offset and spans the number of bytes defined by the format.
     * Then there are zero or more unused bytes depending on sampleStride before the next sample.
     * <p>
     * <b>Default:</b> NULL
     */
    public void setSamples(PxStridedData value) {
        checkNotNull();
        _setSamples(address, value.getAddress());
    }
    private static native void _setSamples(long address, long value);

    /**
     * This threshold is used by the collision detection to determine if a height field edge is convex
     * and can generate contact points.
     * Usually the convexity of an edge is determined from the angle (or cosine of the angle) between
     * the normals of the faces sharing that edge.
     * The height field allows a more efficient approach by comparing height values of neighboring vertices.
     * This parameter offsets the comparison. Smaller changes than 0.5 will not alter the set of convex edges.
     * The rule of thumb is that larger values will result in fewer edge contacts.
     * <p>
     * This parameter is ignored in contact generation with sphere and capsule primitives.
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 0
     */
    public float getConvexEdgeThreshold() {
        checkNotNull();
        return _getConvexEdgeThreshold(address);
    }
    private static native float _getConvexEdgeThreshold(long address);

    /**
     * This threshold is used by the collision detection to determine if a height field edge is convex
     * and can generate contact points.
     * Usually the convexity of an edge is determined from the angle (or cosine of the angle) between
     * the normals of the faces sharing that edge.
     * The height field allows a more efficient approach by comparing height values of neighboring vertices.
     * This parameter offsets the comparison. Smaller changes than 0.5 will not alter the set of convex edges.
     * The rule of thumb is that larger values will result in fewer edge contacts.
     * <p>
     * This parameter is ignored in contact generation with sphere and capsule primitives.
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 0
     */
    public void setConvexEdgeThreshold(float value) {
        checkNotNull();
        _setConvexEdgeThreshold(address, value);
    }
    private static native void _setConvexEdgeThreshold(long address, float value);

    /**
     * Flags bits, combined from values of the enum ::PxHeightFieldFlag.
     * <p>
     * <b>Default:</b> 0
     * @see PxHeightFieldFlags
     */
    public PxHeightFieldFlags getFlags() {
        checkNotNull();
        return PxHeightFieldFlags.wrapPointer(_getFlags(address));
    }
    private static native long _getFlags(long address);

    /**
     * Flags bits, combined from values of the enum ::PxHeightFieldFlag.
     * <p>
     * <b>Default:</b> 0
     * @see PxHeightFieldFlags
     */
    public void setFlags(PxHeightFieldFlags value) {
        checkNotNull();
        _setFlags(address, value.getAddress());
    }
    private static native void _setFlags(long address, long value);

    // Functions

    /**
     * (re)sets the structure to the default.
     */
    public void setToDefault() {
        checkNotNull();
        _setToDefault(address);
    }
    private static native void _setToDefault(long address);

    /**
     * Returns true if the descriptor is valid.
     * @return True if the current settings are valid.
     */
    public boolean isValid() {
        checkNotNull();
        return _isValid(address);
    }
    private static native boolean _isValid(long address);

}
