package physx.extensions;


/**
 * A joint which behaves in a similar way to a hinge or axle.
 * <p>
 *  A hinge joint removes all but a single rotational degree of freedom from two objects.
 *  The axis along which the two bodies may rotate is specified with a point and a direction
 *  vector.
 * <p>
 *  The position of the hinge on each body is specified by the origin of the body's joint frame.
 *  The axis of the hinge is specified as the direction of the x-axis in the body's joint frame.
 * <p>
 *  \image html revoluteJoint.png
 * <p>
 *  A revolute joint can be given a motor, so that it can apply a force to rotate the attached actors.
 *  It may also be given a limit, to restrict the revolute motion to within a certain range. In
 *  addition, the bodies may be projected together if the distance or angle between them exceeds
 *  a given threshold.
 * <p>
 *  Projection, drive and limits are activated by setting the appropriate flags on the joint.
 * @see PxJoint
 */
public class PxRevoluteJoint extends PxJoint {

    protected PxRevoluteJoint() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxRevoluteJoint wrapPointer(long address) {
        return address != 0L ? new PxRevoluteJoint(address) : null;
    }
    
    public static PxRevoluteJoint arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxRevoluteJoint(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * return the angle of the joint, in the range (-2*Pi, 2*Pi]
     */
    public float getAngle() {
        checkNotNull();
        return _getAngle(address);
    }
    private static native float _getAngle(long address);

    /**
     * return the velocity of the joint
     */
    public float getVelocity() {
        checkNotNull();
        return _getVelocity(address);
    }
    private static native float _getVelocity(long address);

    /**
     * set the joint limit parameters. 
     * <p>
     * The limit is activated using the flag PxRevoluteJointFlag::eLIMIT_ENABLED
     * <p>
     * The limit angle range is (-2*Pi, 2*Pi).
     * @param limits The joint limit parameters. 
     * @see PxJointAngularLimitPair
     */
    public void setLimit(PxJointAngularLimitPair limits) {
        checkNotNull();
        _setLimit(address, limits.getAddress());
    }
    private static native void _setLimit(long address, long limits);

    /**
     * set the target velocity for the drive model.
     * <p>
     * The motor will only be able to reach this velocity if the maxForce is sufficiently large.
     * If the joint is spinning faster than this velocity, the motor will actually try to brake
     * (see PxRevoluteJointFlag::eDRIVE_FREESPIN.)
     * <p>
     * The sign of this variable determines the rotation direction, with positive values going
     * the same way as positive joint angles. Setting a very large target velocity may cause
     * undesirable results.
     * @param velocity the drive target velocity
     * <p>
     * <b>Range:</b> (-PX_MAX_F32, PX_MAX_F32)<br>
     * <b>Default:</b> 0.0
     */
    public void setDriveVelocity(float velocity) {
        checkNotNull();
        _setDriveVelocity(address, velocity);
    }
    private static native void _setDriveVelocity(long address, float velocity);

    /**
     * set the target velocity for the drive model.
     * <p>
     * The motor will only be able to reach this velocity if the maxForce is sufficiently large.
     * If the joint is spinning faster than this velocity, the motor will actually try to brake
     * (see PxRevoluteJointFlag::eDRIVE_FREESPIN.)
     * <p>
     * The sign of this variable determines the rotation direction, with positive values going
     * the same way as positive joint angles. Setting a very large target velocity may cause
     * undesirable results.
     * @param velocity the drive target velocity
     * @param autowake Whether to wake up the joint rigids if they are asleep.
     * <p>
     * <b>Range:</b> (-PX_MAX_F32, PX_MAX_F32)<br>
     * <b>Default:</b> 0.0
     */
    public void setDriveVelocity(float velocity, boolean autowake) {
        checkNotNull();
        _setDriveVelocity(address, velocity, autowake);
    }
    private static native void _setDriveVelocity(long address, float velocity, boolean autowake);

    /**
     * gets the target velocity for the drive model.
     * @return the drive target velocity
     * @see #setDriveVelocity
     */
    public float getDriveVelocity() {
        checkNotNull();
        return _getDriveVelocity(address);
    }
    private static native float _getDriveVelocity(long address);

    /**
     * sets the maximum torque the drive can exert.
     * <p>
     * The value set here may be used either as an impulse limit or a force limit, depending on the flag PxConstraintFlag::eDRIVE_LIMITS_ARE_FORCES
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> PX_MAX_F32
     * @see #setDriveVelocity
     */
    public void setDriveForceLimit(float limit) {
        checkNotNull();
        _setDriveForceLimit(address, limit);
    }
    private static native void _setDriveForceLimit(long address, float limit);

    /**
     * gets the maximum torque the drive can exert.
     * @return the torque limit
     * @see #setDriveVelocity
     */
    public float getDriveForceLimit() {
        checkNotNull();
        return _getDriveForceLimit(address);
    }
    private static native float _getDriveForceLimit(long address);

    /**
     * sets the gear ratio for the drive.
     * <p>
     * When setting up the drive constraint, the velocity of the first actor is scaled by this value, and its response to drive torque is scaled down.
     * So if the drive target velocity is zero, the second actor will be driven to the velocity of the first scaled by the gear ratio
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 1.0
     * @param ratio the drive gear ratio
     * @see #getDriveGearRatio
     */
    public void setDriveGearRatio(float ratio) {
        checkNotNull();
        _setDriveGearRatio(address, ratio);
    }
    private static native void _setDriveGearRatio(long address, float ratio);

    /**
     * gets the gear ratio.
     * @return the drive gear ratio
     * @see #setDriveGearRatio
     */
    public float getDriveGearRatio() {
        checkNotNull();
        return _getDriveGearRatio(address);
    }
    private static native float _getDriveGearRatio(long address);

    /**
     * sets the flags specific to the Revolute Joint.
     * <p>
     * <b>Default</b> PxRevoluteJointFlags(0)
     * @param flags The joint flags.
     */
    public void setRevoluteJointFlags(PxRevoluteJointFlags flags) {
        checkNotNull();
        _setRevoluteJointFlags(address, flags.getAddress());
    }
    private static native void _setRevoluteJointFlags(long address, long flags);

    /**
     * sets a single flag specific to a Revolute Joint.
     * @param flag The flag to set or clear.
     * @param value the value to which to set the flag
     */
    public void setRevoluteJointFlag(PxRevoluteJointFlagEnum flag, boolean value) {
        checkNotNull();
        _setRevoluteJointFlag(address, flag.value, value);
    }
    private static native void _setRevoluteJointFlag(long address, int flag, boolean value);

    /**
     * gets the flags specific to the Revolute Joint.
     * @return the joint flags
     */
    public PxRevoluteJointFlags getRevoluteJointFlags() {
        checkNotNull();
        return PxRevoluteJointFlags.wrapPointer(_getRevoluteJointFlags(address));
    }
    private static native long _getRevoluteJointFlags(long address);

    /**
     * Set the linear tolerance threshold for projection. Projection is enabled if PxConstraintFlag::ePROJECTION
     * is set for the joint.
     * <p>
     * If the joint separates by more than this distance along its locked degrees of freedom, the solver 
     * will move the bodies to close the distance.
     * <p>
     * Setting a very small tolerance may result in simulation jitter or other artifacts.
     * <p>
     * Sometimes it is not possible to project (for example when the joints form a cycle).
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 1e10f
     * @param tolerance the linear tolerance threshold
     * @see #getProjectionLinearTolerance
     * @deprecated
     */
    @Deprecated
    public void setProjectionLinearTolerance(float tolerance) {
        checkNotNull();
        _setProjectionLinearTolerance(address, tolerance);
    }
    private static native void _setProjectionLinearTolerance(long address, float tolerance);

    /**
     * Get the linear tolerance threshold for projection.
     * @return the linear tolerance threshold
     * @see #setProjectionLinearTolerance
     * @deprecated
     */
    @Deprecated
    public float getProjectionLinearTolerance() {
        checkNotNull();
        return _getProjectionLinearTolerance(address);
    }
    private static native float _getProjectionLinearTolerance(long address);

    /**
     * Set the angular tolerance threshold for projection. Projection is enabled if 
     * PxConstraintFlag::ePROJECTION is set for the joint.
     * <p>
     * If the joint deviates by more than this angle around its locked angular degrees of freedom, 
     * the solver will move the bodies to close the angle.
     * <p>
     * Setting a very small tolerance may result in simulation jitter or other artifacts.
     * <p>
     * Sometimes it is not possible to project (for example when the joints form a cycle).
     * <p>
     * <b>Range:</b> [0,Pi] <br>
     * <b>Default:</b> Pi
     * @param tolerance the angular tolerance threshold in radians
     * @see #getProjectionAngularTolerance
     * @deprecated
     */
    @Deprecated
    public void setProjectionAngularTolerance(float tolerance) {
        checkNotNull();
        _setProjectionAngularTolerance(address, tolerance);
    }
    private static native void _setProjectionAngularTolerance(long address, float tolerance);

    /**
     * gets the angular tolerance threshold for projection.
     * @return the angular tolerance threshold in radians
     * @see #setProjectionAngularTolerance
     * @deprecated
     */
    @Deprecated
    public float getProjectionAngularTolerance() {
        checkNotNull();
        return _getProjectionAngularTolerance(address);
    }
    private static native float _getProjectionAngularTolerance(long address);

}
