package physx.character;


/**
 * A capsule character controller.
 * <p>
 * The capsule is defined as a position, a vertical height, and a radius.
 * The height is the distance between the two sphere centers at the end of the capsule.
 * In other words:
 * <p>
 * p = pos (returned by controller)<br>
 * h = height<br>
 * r = radius<br>
 * <p>
 * p = center of capsule<br>
 * top sphere center = p.y + h*0.5<br>
 * bottom sphere center = p.y - h*0.5<br>
 * top capsule point = p.y + h*0.5 + r<br>
 * bottom capsule point = p.y - h*0.5 - r<br>
 */
public class PxCapsuleController extends PxController {

    protected PxCapsuleController() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxCapsuleController wrapPointer(long address) {
        return address != 0L ? new PxCapsuleController(address) : null;
    }
    
    public static PxCapsuleController arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxCapsuleController(long address) {
        super(address);
    }

    // Functions

    /**
     * Gets controller's radius.
     * @return The radius of the controller.
     * @see #setRadius
     */
    public float getRadius() {
        checkNotNull();
        return _getRadius(address);
    }
    private static native float _getRadius(long address);

    /**
     * Sets controller's radius.
     * <p>
     * \warning this doesn't check for collisions.
     * @param radius The new radius for the controller.
     * @return Currently always true.
     * @see #getRadius
     */
    public boolean setRadius(float radius) {
        checkNotNull();
        return _setRadius(address, radius);
    }
    private static native boolean _setRadius(long address, float radius);

    /**
     * Gets controller's height.
     * @return The height of the capsule controller.
     * @see #setHeight
     */
    public float getHeight() {
        checkNotNull();
        return _getHeight(address);
    }
    private static native float _getHeight(long address);

    /**
     * Resets controller's height.
     * <p>
     * \warning this doesn't check for collisions.
     * @param height The new height for the controller.
     * @return Currently always true.
     * @see #getHeight
     */
    public boolean setHeight(float height) {
        checkNotNull();
        return _setHeight(address, height);
    }
    private static native boolean _setHeight(long address, float height);

    /**
     * Gets controller's climbing mode.
     * @return The capsule controller's climbing mode.
     * @see #setClimbingMode
     */
    public PxCapsuleClimbingModeEnum getClimbingMode() {
        checkNotNull();
        return PxCapsuleClimbingModeEnum.forValue(_getClimbingMode(address));
    }
    private static native int _getClimbingMode(long address);

    /**
     * Sets controller's climbing mode.
     * @param mode The capsule controller's climbing mode.
     * @see #getClimbingMode
     */
    public boolean setClimbingMode(PxCapsuleClimbingModeEnum mode) {
        checkNotNull();
        return _setClimbingMode(address, mode.value);
    }
    private static native boolean _setClimbingMode(long address, int mode);

}
