package physx.common;

import physx.NativeObject;

public class PxBoundedData extends NativeObject {

    public static PxBoundedData wrapPointer(long address) {
        return address != 0L ? new PxBoundedData(address) : null;
    }
    
    protected PxBoundedData(long address) {
        super(address);
    }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;

    // Placed Constructors

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @return Stack allocated object of PxBoundedData
     */
    public static PxBoundedData createAt(long address) {
        __placement_new_PxBoundedData(address);
        PxBoundedData createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @return Stack allocated object of PxBoundedData
     */
    public static <T> PxBoundedData createAt(T allocator, Allocator<T> allocate) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxBoundedData(address);
        PxBoundedData createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxBoundedData(long address);

    // Constructors

    public PxBoundedData() {
        address = _PxBoundedData();
    }
    private static native long _PxBoundedData();

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * @return WebIDL type: unsigned long
     */
    public int getCount() {
        if (address == 0L) {
            throw new NullPointerException("Native address of " + this + " is 0");
        }
        return _getCount(address);
    }
    private static native int _getCount(long address);

    /**
     * @param value WebIDL type: unsigned long
     */
    public void setCount(int value) {
        if (address == 0L) {
            throw new NullPointerException("Native address of " + this + " is 0");
        }
        _setCount(address, value);
    }
    private static native void _setCount(long address, int value);

    /**
     * @return WebIDL type: unsigned long
     */
    public int getStride() {
        if (address == 0L) {
            throw new NullPointerException("Native address of " + this + " is 0");
        }
        return _getStride(address);
    }
    private static native int _getStride(long address);

    /**
     * @param value WebIDL type: unsigned long
     */
    public void setStride(int value) {
        if (address == 0L) {
            throw new NullPointerException("Native address of " + this + " is 0");
        }
        _setStride(address, value);
    }
    private static native void _setStride(long address, int value);

    /**
     * @return WebIDL type: VoidPtr [Const]
     */
    public NativeObject getData() {
        if (address == 0L) {
            throw new NullPointerException("Native address of " + this + " is 0");
        }
        return NativeObject.wrapPointer(_getData(address));
    }
    private static native long _getData(long address);

    /**
     * @param value WebIDL type: VoidPtr [Const]
     */
    public void setData(NativeObject value) {
        if (address == 0L) {
            throw new NullPointerException("Native address of " + this + " is 0");
        }
        _setData(address, value.getAddress());
    }
    private static native void _setData(long address, long value);

}
