/*
 * Decompiled with CFR 0.152.
 */
package de.entwicklertraining.api.base;

import de.entwicklertraining.api.base.ApiCallCaptureInput;
import de.entwicklertraining.api.base.ApiClientSettings;
import de.entwicklertraining.api.base.ApiRequest;
import de.entwicklertraining.api.base.ApiRequestExecutionContext;
import de.entwicklertraining.api.base.ApiResponse;
import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class ApiClient {
    private static final Logger logger = LoggerFactory.getLogger((String)ApiClient.class.getName());
    private static final ExecutorService HTTP_CLIENT_EXECUTOR = Executors.newVirtualThreadPerTaskExecutor();
    private static final ExecutorService CANCEL_WATCHER_EXECUTOR = Executors.newVirtualThreadPerTaskExecutor();
    protected final HttpClient httpClient;
    protected ApiClientSettings settings;
    private Optional<String> baseUrl = Optional.empty();
    private boolean statusCodeExceptionsWarningLogged = false;
    private final Map<Integer, StatusCodeExceptionRegistration> statusCodeExceptions = new HashMap<Integer, StatusCodeExceptionRegistration>();

    protected String getBaseUrl() {
        return this.baseUrl.orElseThrow(() -> new IllegalStateException("Base URL has not been set. Call setBaseUrl() in the constructor of your client implementation."));
    }

    protected final void setBaseUrl(String baseUrl) {
        if (baseUrl == null || baseUrl.trim().isEmpty()) {
            throw new IllegalArgumentException("Base URL cannot be null or empty");
        }
        this.baseUrl = Optional.of(baseUrl.endsWith("/") ? baseUrl.substring(0, baseUrl.length() - 1) : baseUrl);
    }

    protected ApiClient(ApiClientSettings settings) {
        this.httpClient = HttpClient.newBuilder().executor(HTTP_CLIENT_EXECUTOR).build();
        this.settings = settings;
    }

    protected final void registerStatusCodeException(int statusCode, Class<? extends RuntimeException> exceptionClass, String message, boolean retry) {
        this.statusCodeExceptions.put(statusCode, new StatusCodeExceptionRegistration(exceptionClass, message, retry));
    }

    public <T extends ApiRequest<U>, U extends ApiResponse<T>> U sendRequestWithExponentialBackoff(T request) {
        if (this.settings.getBeforeSendAction() != null) {
            this.settings.getBeforeSendAction().accept(request);
        }
        long startMillis = System.currentTimeMillis();
        Instant startInstant = Instant.ofEpochMilli(startMillis);
        boolean success = false;
        RuntimeException finalException = null;
        ApiResponse finalResponse = null;
        ApiRequestExecutionContext context = new ApiRequestExecutionContext();
        try {
            finalResponse = this.executeWithRetry(() -> this.runRequest(request, context), request);
            success = true;
            ApiResponse apiResponse = finalResponse;
            return (U)apiResponse;
        }
        catch (RuntimeException ex) {
            finalException = ex;
            throw ex;
        }
        finally {
            Instant endInstant = Instant.ofEpochMilli(System.currentTimeMillis());
            if (success && request.hasCaptureOnSuccess()) {
                this.storeCaptureData(request, request.getCaptureOnSuccess(), finalResponse, context, finalException, startInstant, endInstant, success);
            } else if (!success && request.hasCaptureOnError()) {
                this.storeCaptureData(request, request.getCaptureOnError(), finalResponse, context, finalException, startInstant, endInstant, success);
            }
        }
    }

    public <T extends ApiRequest<U>, U extends ApiResponse<T>> U sendRequest(T request) {
        if (this.settings.getBeforeSendAction() != null) {
            this.settings.getBeforeSendAction().accept(request);
        }
        long startMillis = System.currentTimeMillis();
        Instant startInstant = Instant.ofEpochMilli(startMillis);
        boolean success = false;
        RuntimeException finalException = null;
        ApiResponse finalResponse = null;
        int maxDurationSeconds = request.getMaxExecutionTimeInSeconds();
        long maxDurationMs = (long)maxDurationSeconds * 1000L;
        ApiRequestExecutionContext context = new ApiRequestExecutionContext();
        try {
            finalResponse = this.executeWithTimeout(() -> this.runRequest(request, context), maxDurationMs);
            success = true;
            ApiResponse apiResponse = finalResponse;
            return (U)apiResponse;
        }
        catch (RuntimeException ex) {
            finalException = ex;
            throw ex;
        }
        finally {
            Instant endInstant = Instant.ofEpochMilli(System.currentTimeMillis());
            if (success && request.hasCaptureOnSuccess()) {
                this.storeCaptureData(request, request.getCaptureOnSuccess(), finalResponse, context, finalException, startInstant, endInstant, success);
            } else if (!success && request.hasCaptureOnError()) {
                this.storeCaptureData(request, request.getCaptureOnError(), finalResponse, context, finalException, startInstant, endInstant, success);
            }
        }
    }

    protected <T extends ApiRequest<U>, U extends ApiResponse<T>> U runRequest(T request, ApiRequestExecutionContext<T, U> context) {
        String method;
        if (this.statusCodeExceptions.isEmpty() && !this.statusCodeExceptionsWarningLogged) {
            logger.warn("No status code exceptions registered. It's recommended to register appropriate status code exceptions in the constructor of your client implementation using registerStatusCodeException().");
            this.statusCodeExceptionsWarningLogged = true;
        }
        String fullUrl = this.getBaseUrl() + request.getRelativeUrl();
        HttpRequest.Builder builder = HttpRequest.newBuilder().uri(URI.create(fullUrl)).header("Content-Type", request.getContentType());
        if (this.settings.getBearerAuthenticationKey().isPresent()) {
            builder = builder.header("Authorization", "Bearer " + this.settings.getBearerAuthenticationKey().get());
        }
        for (Map.Entry<String, String> entry : request.getAdditionalHeaders().entrySet()) {
            builder.header(entry.getKey(), entry.getValue());
        }
        switch (method = request.getHttpMethod().toUpperCase()) {
            case "POST": {
                if (request.getContentType().startsWith("multipart/form-data")) {
                    byte[] bodyBytes = request.getBodyBytes();
                    builder.POST(HttpRequest.BodyPublishers.ofByteArray(bodyBytes));
                    break;
                }
                builder.POST(HttpRequest.BodyPublishers.ofString(request.getBody()));
                break;
            }
            case "GET": {
                builder.GET();
                break;
            }
            case "DELETE": {
                builder.DELETE();
                break;
            }
            default: {
                throw new ApiClientException("Unsupported HTTP method: " + method);
            }
        }
        HttpRequest httpRequest = builder.build();
        CompletionStage future = request.isBinaryResponse() ? this.httpClient.sendAsync(httpRequest, HttpResponse.BodyHandlers.ofByteArray()).thenApply(response -> response) : this.httpClient.sendAsync(httpRequest, HttpResponse.BodyHandlers.ofString()).thenApply(response -> response);
        CompletableFuture<Void> cancelWatcher = CompletableFuture.runAsync(() -> ApiClient.lambda$runRequest$2((CompletableFuture)future, request), CANCEL_WATCHER_EXECUTOR);
        try {
            if (Thread.currentThread().isInterrupted()) {
                ((CompletableFuture)future).cancel(true);
                throw new ApiTimeoutException("Thread was interrupted before sending request");
            }
            HttpResponse httpResponse = request.getMaxExecutionTimeInSeconds() > 0 ? (HttpResponse)((CompletableFuture)future).get(request.getMaxExecutionTimeInSeconds(), TimeUnit.SECONDS) : (HttpResponse)((CompletableFuture)future).get();
            int statusCode = httpResponse.statusCode();
            if (statusCode == 200) {
                if (request.isBinaryResponse()) {
                    byte[] bodyBytes = (byte[])httpResponse.body();
                    context.setResponseBytes(bodyBytes);
                    U u = request.createResponse(bodyBytes);
                    return u;
                }
                String bodyString = (String)httpResponse.body();
                context.setResponseBody(bodyString);
                U u = request.createResponse(bodyString);
                return u;
            }
            try {
                String bodySnippet;
                if (request.isBinaryResponse()) {
                    byte[] respBytes = (byte[])httpResponse.body();
                    context.setResponseBytes(respBytes);
                    bodySnippet = new String(respBytes);
                } else {
                    String respString = (String)httpResponse.body();
                    context.setResponseBody(respString);
                    bodySnippet = respString;
                }
                StatusCodeExceptionRegistration reg = this.statusCodeExceptions.get(statusCode);
                if (reg != null) {
                    throw this.createException(reg, bodySnippet);
                }
                throw new ApiClientException("Unexpected HTTP status " + statusCode + " - " + bodySnippet);
            }
            catch (CancellationException cex) {
                throw new ApiTimeoutException("Request was canceled", cex);
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                ((CompletableFuture)future).cancel(true);
                throw new ApiClientException("Request interrupted", ie);
            }
            catch (ExecutionException ex) {
                ((CompletableFuture)future).cancel(true);
                if (ex.getCause() != null) {
                    throw new ApiClientException("Request failed: " + ex.getCause().getMessage(), ex.getCause());
                }
                throw new ApiClientException("Execution failed", ex);
            }
            catch (TimeoutException e) {
                throw new ApiTimeoutException("Maximum execution timeout of " + request.getMaxExecutionTimeInSeconds() + " seconds has been reached!", e);
            }
        }
        finally {
            cancelWatcher.cancel(true);
        }
    }

    private RuntimeException createException(StatusCodeExceptionRegistration reg, String bodySnippet) {
        String errorMsg = reg.message + ": " + bodySnippet;
        try {
            return reg.exceptionClass.getConstructor(String.class).newInstance(errorMsg);
        }
        catch (IllegalAccessException | InstantiationException | NoSuchMethodException | InvocationTargetException e) {
            return new ApiClientException(errorMsg, e);
        }
    }

    protected <U extends ApiResponse<?>> U executeWithRetry(Supplier<U> operation, ApiRequest<?> request) {
        long startTimeMs = System.currentTimeMillis();
        int maxDurationSeconds = request.getMaxExecutionTimeInSeconds();
        long maxDurationMs = (long)maxDurationSeconds * 1000L;
        String latestReason = null;
        Throwable latestException = null;
        for (int attempt = 1; attempt <= this.settings.getMaxRetries(); ++attempt) {
            long remainingMs;
            long elapsedMs;
            try {
                elapsedMs = System.currentTimeMillis() - startTimeMs;
                remainingMs = maxDurationMs - elapsedMs;
                if (maxDurationMs > 0L && remainingMs <= 0L) {
                    throw this.createTimeoutException(latestReason, maxDurationSeconds, latestException);
                }
                return (U)((ApiResponse)this.executeWithTimeout(operation, remainingMs));
            }
            catch (Throwable ex) {
                if (!this.isRetryableException(ex)) {
                    throw ex;
                }
                latestReason = "Retriable error: " + ex.getMessage();
                latestException = ex;
                if (attempt == this.settings.getMaxRetries()) {
                    throw this.createRetriesExhaustedException(latestReason, latestException);
                }
                elapsedMs = System.currentTimeMillis() - startTimeMs;
                remainingMs = maxDurationMs - elapsedMs;
                if (maxDurationMs > 0L && remainingMs <= 0L) {
                    throw this.createTimeoutException(latestReason, maxDurationSeconds, latestException);
                }
                long durationOfNextSleep = this.calculateNextSleep(this.settings.getInitialDelayMs());
                durationOfNextSleep = this.maybeAdjustSleepForFinalRetry(maxDurationSeconds, durationOfNextSleep, remainingMs);
                this.applySleep(durationOfNextSleep, remainingMs);
                continue;
            }
        }
        throw new ApiClientException("Exponential backoff logic exhausted unexpectedly.");
    }

    private boolean isRetryableException(Throwable ex) {
        Class<?> exClass = ex.getClass();
        for (StatusCodeExceptionRegistration reg : this.statusCodeExceptions.values()) {
            if (!reg.exceptionClass.isAssignableFrom(exClass) || !reg.retry) continue;
            return true;
        }
        return false;
    }

    protected <U> U executeWithTimeout(Supplier<U> operation, long remainingMs) {
        CompletableFuture<U> future = CompletableFuture.supplyAsync(operation);
        try {
            if (remainingMs > 0L) {
                return future.get(remainingMs, TimeUnit.MILLISECONDS);
            }
            return future.get();
        }
        catch (TimeoutException ex) {
            future.cancel(true);
            throw new ApiTimeoutException("Request timed out during execution", ex);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            future.cancel(true);
            throw new ApiClientException("Request interrupted", ex);
        }
        catch (ExecutionException ex) {
            future.cancel(true);
            if (ex.getCause() != null) {
                throw (RuntimeException)ex.getCause();
            }
            throw new ApiClientException("Execution failed", ex);
        }
    }

    protected long calculateNextSleep(long currentDelay) {
        double factor = this.settings.getExponentialBase();
        if (this.settings.isUseJitter()) {
            factor *= 1.0 + Math.random();
        }
        return (long)((double)currentDelay * factor);
    }

    protected long maybeAdjustSleepForFinalRetry(int maxDurationSeconds, long delayMs, long remainingMs) {
        boolean isTheRemainingTimeLongEnoughToApplyAFinalRetry;
        if (maxDurationSeconds == 0) {
            return delayMs;
        }
        boolean isTheNextSleepPhaseLongerThanTheRemainingTime = delayMs > remainingMs;
        long minSleepMs = TimeUnit.SECONDS.toMillis(this.settings.getMinSleepDurationForFinalRetryInSeconds());
        boolean bl = isTheRemainingTimeLongEnoughToApplyAFinalRetry = minSleepMs >= remainingMs;
        if (isTheNextSleepPhaseLongerThanTheRemainingTime && isTheRemainingTimeLongEnoughToApplyAFinalRetry) {
            long cutoffExpectedExecutionTimeForFinalRetry = TimeUnit.SECONDS.toMillis(this.settings.getMaxExecutionTimeForFinalRetryInSeconds());
            return remainingMs - cutoffExpectedExecutionTimeForFinalRetry;
        }
        return delayMs;
    }

    protected void applySleep(long delayMs, long remainingMs) {
        if (remainingMs > 0L && delayMs > remainingMs) {
            throw new ApiTimeoutException("Skipping sleep to prevent timeout (remaining: " + TimeUnit.MILLISECONDS.toSeconds(remainingMs) + "s)");
        }
        try {
            Thread.sleep(delayMs);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new ApiClientException("Interrupted during backoff", e);
        }
    }

    protected ApiTimeoutException createTimeoutException(String reason, int maxDurationSeconds, Throwable cause) {
        String msg = "Maximum execution time of " + maxDurationSeconds + "s reached!";
        if (reason != null) {
            msg = msg + " " + reason;
        }
        return new ApiTimeoutException(msg, cause);
    }

    protected ApiTimeoutException createRetriesExhaustedException(String reason, Throwable cause) {
        String msg = "Maximum retries of " + this.settings.getMaxRetries() + " exhausted!";
        if (reason != null) {
            msg = msg + " " + reason;
        }
        return new ApiTimeoutException(msg, cause);
    }

    private <T extends ApiRequest<U>, U extends ApiResponse<T>> void storeCaptureData(T request, Consumer<ApiCallCaptureInput> captureConsumer, U finalResponse, ApiRequestExecutionContext<T, U> context, RuntimeException finalException, Instant start, Instant end, boolean success) {
        ApiTimeoutException ate;
        Throwable storeEx = finalException;
        if (storeEx instanceof ApiTimeoutException && (ate = (ApiTimeoutException)storeEx).getCause() != null) {
            storeEx = ate.getCause();
        }
        String exceptionClass = null;
        String exceptionMessage = null;
        String exceptionStacktrace = null;
        if (!success && storeEx != null) {
            exceptionClass = storeEx.getClass().getName();
            exceptionMessage = storeEx.getMessage();
            exceptionStacktrace = this.getStackTraceAsString(storeEx);
        }
        Object inputData = request.isBinaryResponse() && request.getBodyBytes() != null ? "binary body (size=" + request.getBodyBytes().length + ")" : request.getBody();
        Object outputData = null;
        if (success) {
            if (context.getResponseBody() != null) {
                outputData = context.getResponseBody();
            } else if (context.getResponseBytes() != null) {
                outputData = "binary response (size=" + context.getResponseBytes().length + ")";
            }
        } else if (context.getResponseBody() != null) {
            outputData = context.getResponseBody();
        } else if (context.getResponseBytes() != null) {
            outputData = "binary response (size=" + context.getResponseBytes().length + ")";
        }
        captureConsumer.accept(new ApiCallCaptureInput(start, end, success, exceptionClass, exceptionMessage, exceptionStacktrace, (String)inputData, (String)outputData));
    }

    private String getStackTraceAsString(Throwable throwable) {
        StringBuilder sb = new StringBuilder();
        sb.append(throwable.toString()).append("\n");
        for (StackTraceElement elem : throwable.getStackTrace()) {
            sb.append("\tat ").append(elem.toString()).append("\n");
        }
        return sb.toString();
    }

    private static /* synthetic */ void lambda$runRequest$2(CompletableFuture future, ApiRequest request) {
        try {
            while (!Thread.currentThread().isInterrupted() && !future.isDone()) {
                if (request.getIsCanceledSupplier().get().booleanValue()) {
                    future.cancel(true);
                    break;
                }
                Thread.sleep(100L);
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    protected static final class StatusCodeExceptionRegistration {
        final Class<? extends RuntimeException> exceptionClass;
        final String message;
        final boolean retry;

        StatusCodeExceptionRegistration(Class<? extends RuntimeException> exceptionClass, String message, boolean retry) {
            this.exceptionClass = exceptionClass;
            this.message = message;
            this.retry = retry;
        }
    }

    public static class ApiClientException
    extends RuntimeException {
        public ApiClientException(String message) {
            super(message);
        }

        public ApiClientException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class ApiTimeoutException
    extends ApiClientException {
        public ApiTimeoutException(String message) {
            super(message);
        }

        public ApiTimeoutException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class ApiResponseUnusableException
    extends ApiClientException {
        public ApiResponseUnusableException(String message) {
            super(message);
        }

        public ApiResponseUnusableException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class HTTP_504_ServerTimeoutException
    extends RuntimeException {
        public HTTP_504_ServerTimeoutException(String message) {
            super(message);
        }

        public HTTP_504_ServerTimeoutException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class HTTP_503_ServerUnavailableException
    extends RuntimeException {
        public HTTP_503_ServerUnavailableException(String message) {
            super(message);
        }

        public HTTP_503_ServerUnavailableException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class HTTP_500_ServerErrorException
    extends RuntimeException {
        public HTTP_500_ServerErrorException(String message) {
            super(message);
        }

        public HTTP_500_ServerErrorException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class HTTP_422_UnprocessableEntityException
    extends RuntimeException {
        public HTTP_422_UnprocessableEntityException(String message) {
            super(message);
        }

        public HTTP_422_UnprocessableEntityException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class HTTP_404_NotFoundException
    extends RuntimeException {
        public HTTP_404_NotFoundException(String message) {
            super(message);
        }

        public HTTP_404_NotFoundException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class HTTP_403_PermissionDeniedException
    extends RuntimeException {
        public HTTP_403_PermissionDeniedException(String message) {
            super(message);
        }

        public HTTP_403_PermissionDeniedException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class HTTP_402_PaymentRequiredException
    extends RuntimeException {
        public HTTP_402_PaymentRequiredException(String message) {
            super(message);
        }

        public HTTP_402_PaymentRequiredException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class HTTP_401_AuthorizationException
    extends RuntimeException {
        public HTTP_401_AuthorizationException(String message) {
            super(message);
        }

        public HTTP_401_AuthorizationException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class HTTP_400_RequestRejectedException
    extends RuntimeException {
        public HTTP_400_RequestRejectedException(String message) {
            super(message);
        }

        public HTTP_400_RequestRejectedException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class HTTP_429_RateLimitOrQuotaException
    extends RuntimeException {
        private ExceptionType type;

        public HTTP_429_RateLimitOrQuotaException(String message) {
            super(message);
            this.type = ExceptionType.Unknown;
        }

        public HTTP_429_RateLimitOrQuotaException(String message, Throwable cause) {
            super(message, cause);
            this.type = ExceptionType.RateLimit;
        }

        public void setType(ExceptionType type) {
            this.type = type;
        }

        public ExceptionType getType() {
            return this.type;
        }

        public static enum ExceptionType {
            Unknown,
            RateLimit,
            Quota;

        }
    }
}

