/*
 * Copyright (c) 2015, vseifert
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * * Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package de.dimensionv.java.libraries.common.utilities.enums;

import de.dimensionv.java.libraries.common.exceptions.InvalidEnumNameException;
import de.dimensionv.java.libraries.common.exceptions.InvalidEnumValueException;
import java.lang.reflect.Method;

/**
 *
 * @author vseifert
 */
public class EnumHelper {

  /**
   * This method takes an arbitrary {@code int} {@code ordinalValue} and returns the appropriate {@code enum}. In case
   * the {@code int} {@code ordinalValue} is not a valid value of the {@code enum}, an {@code InvalidEnumValueException}
   * will be thrown.
   *
   * @param <T> the {@code enum} to be worked with
   * @param ordinalValue The {@code int} value to convert into the {@code enum} (also called "ordinal").
   * @param enumValues An array containing all possible values of the {@code enum}.
   *
   * @return the {@code enum}-object associated with the given ordinal.
   *
   * @throws InvalidEnumValueException if {@code ordinalValue} is not a valid ordinal value for the {@code enum}.
   */
  public static <T> T fromOrdinal(int ordinalValue, final T[] enumValues) {
    try {
      return enumValues[ordinalValue];
    } catch (ArrayIndexOutOfBoundsException ex) {
      throw new InvalidEnumValueException(ordinalValue, enumValues.length);
    }
  }

  /**
   * <p>
   * This method takes an arbitrary {@code String} {@code name} and returns the appropriate {@code enum}.</p>
   * <p>
   * In case the {@code String} {@code name} is not a valid name of one of the {@code enum}s values, an
   * {@code InvalidEnumNameException} will be thrown.</p>
   * <p>
   * <b>Please Note:</b> This method does a case-sensitive comparison of the name. For a non-case-sensitive approach,
   * see {@link #fromNameIgnoreCase(String, Object[])}.
   *
   * @param <T> the {@code enum} to be worked with
   * @param name The name of one of the {@code enum}s values.
   * @param enumValues An array containing all possible values of the {@code enum}.
   *
   * @return the {@code enum}-object associated with the given ordinal.
   *
   * @throws InvalidEnumNameException if name is not a valid name of one of the {@code enum}s values.
   * @see #fromNameIgnoreCase(String, Object[])
   */
  public static <T> T fromName(String name, final T[] enumValues) {
    try {
      Method nameMethod = enumValues[0].getClass().getSuperclass().getDeclaredMethod("name");
      for (T value : enumValues) {
        if (nameMethod.invoke(value).equals(name)) {
          return value;
        }
      }
    } catch (Exception ex) {
      throw new IllegalStateException("An error occurred.", ex);
    }
    throw new InvalidEnumNameException(name);
  }

  /**
   * <p>
   * This method takes an arbitrary {@code String} {@code name} and returns the appropriate {@code enum}.</p>
   * <p>
   * In case the {@code String} {@code name} is not a valid name of one of the {@code enum}s values, an
   * {@code InvalidEnumNameException} will be thrown.</p>
   * <p>
   * <b>Please Note:</b> This method does a non-case-sensitive comparison of the name. For a case-sensitive approach,
   * see {@link #fromName(String, Object[]) }.
   *
   * @param <T> the {@code enum} to be worked with
   * @param name The name of one of the {@code enum}s values.
   * @param enumValues An array containing all possible values of the {@code enum}.
   *
   * @return the {@code enum}-object associated with the given ordinal.
   *
   * @throws InvalidEnumNameException if name is not a valid name of one of the {@code enum}s values.
   * @see #fromName(String, Object[])
   */
  public static <T> T fromNameIgnoreCase(String name, final T[] enumValues) {
    try {
      Method nameMethod = enumValues[0].getClass().getSuperclass().getDeclaredMethod("name");
      for (T value : enumValues) {
        if (((String) nameMethod.invoke(value)).equalsIgnoreCase(name)) {
          return value;
        }
      }
    } catch (Exception ex) {
      throw new IllegalStateException("An error occurred.", ex);
    }
    throw new InvalidEnumNameException(name);
  }
}
