/*
 * Decompiled with CFR 0.152.
 */
package de.digitalcollections.cudami.server.backend.impl.file.identifiable.resource;

import de.digitalcollections.cudami.server.backend.api.repository.identifiable.resource.FileResourceBinaryRepository;
import de.digitalcollections.model.exception.ResourceNotFoundException;
import de.digitalcollections.model.exception.TechnicalException;
import de.digitalcollections.model.file.MimeType;
import de.digitalcollections.model.identifiable.resource.FileResource;
import de.digitalcollections.model.identifiable.resource.ImageFileResource;
import de.digitalcollections.model.identifiable.resource.VideoFileResource;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringReader;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Iterator;
import java.util.Objects;
import java.util.UUID;
import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.stream.ImageInputStream;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.ReaderInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Repository;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

@Repository
public class FileResourceBinaryRepositoryImpl
implements FileResourceBinaryRepository {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileResourceBinaryRepositoryImpl.class);
    public static final String DEFAULT_FILENAME_WITHOUT_EXTENSION = "resource";
    private final URL iiifImageBaseUrl;
    private final URL mediaVideoBaseUrl;
    private final String repositoryFolderPath;
    private final ResourceLoader resourceLoader;

    @Autowired
    public FileResourceBinaryRepositoryImpl(@Value(value="${cudami.repositoryFolderPath}") String folderPath, @Value(value="${iiif.image.baseUrl:#{null}}") URL iiifImageBaseUrl, @Value(value="${media.video.baseUrl:#{null}}") URL mediaVideoBaseUrl, ResourceLoader resourceLoader) {
        this.iiifImageBaseUrl = iiifImageBaseUrl;
        this.mediaVideoBaseUrl = mediaVideoBaseUrl;
        this.repositoryFolderPath = folderPath.replace("~", System.getProperty("user.home"));
        this.resourceLoader = resourceLoader;
    }

    public void assertReadability(FileResource resource) throws TechnicalException, ResourceNotFoundException {
        try (InputStream is = this.getInputStream(resource);){
            if (is.available() <= 0) {
                throw new TechnicalException("Cannot read " + resource.getFilename() + ": Empty file");
            }
        }
        catch (TechnicalException e) {
            throw new TechnicalException("Cannot read " + resource.getFilename() + ": Empty file");
        }
        catch (ResourceNotFoundException e) {
            throw e;
        }
        catch (Exception e) {
            throw new TechnicalException("Cannot read " + resource.getFilename() + ": " + e.getMessage());
        }
    }

    protected URI createUri(@NonNull UUID uuid, MimeType mimeType) {
        Objects.requireNonNull(uuid, "uuid must not be null");
        String extension = "undefined";
        if (mimeType == null) {
            mimeType = MimeType.MIME_APPLICATION_OCTET_STREAM;
        }
        if (!mimeType.getExtensions().isEmpty()) {
            extension = (String)mimeType.getExtensions().get(0);
        }
        String primaryType = mimeType.getPrimaryType();
        String uuidStr = uuid.toString();
        String uuidPath = this.getSplittedUuidPath(uuidStr);
        Path path = Paths.get(this.repositoryFolderPath, primaryType, extension, uuidPath, DEFAULT_FILENAME_WITHOUT_EXTENSION);
        String location = "file://" + path.toString();
        if (!extension.isBlank() && !"undefined".equals(extension)) {
            location = location + "." + extension;
        }
        return URI.create(location);
    }

    public FileResource find(String uuidStr, MimeType mimeType) throws TechnicalException, ResourceNotFoundException {
        FileResource resource = new FileResource();
        UUID uuid = UUID.fromString(uuidStr);
        resource.setUuid(uuid);
        URI uri = this.createUri(uuid, mimeType);
        if (!this.resourceLoader.getResource(uri.toString()).isReadable()) {
            throw new TechnicalException("File resource at uri " + uri + " is not readable");
        }
        resource.setUri(uri);
        String filename = uri.toString().substring(uri.toString().lastIndexOf("/"));
        resource.setFilename(filename);
        resource.setMimeType(MimeType.fromFilename((String)filename));
        Resource springResource = this.resourceLoader.getResource(uri.toString());
        long lastModified = this.getLastModified(springResource);
        if (lastModified != 0L) {
            resource.setLastModified(Instant.ofEpochMilli(lastModified).atOffset(ZoneOffset.UTC).toLocalDateTime());
        } else {
            resource.setLastModified(LocalDateTime.ofEpochSecond(0L, 0, ZoneOffset.UTC));
        }
        long length = this.getSize(springResource);
        if (length > -1L) {
            resource.setSizeInBytes(length);
        }
        return resource;
    }

    public byte[] getAsBytes(FileResource resource) throws TechnicalException, ResourceNotFoundException {
        try {
            this.assertReadability(resource);
            return IOUtils.toByteArray((InputStream)this.getInputStream(resource));
        }
        catch (IOException ex) {
            String msg = "Could not read bytes from resource: " + resource;
            LOGGER.error(msg, (Throwable)ex);
            throw new TechnicalException(msg, (Throwable)ex);
        }
    }

    public Document getAsDocument(FileResource resource) throws TechnicalException, ResourceNotFoundException {
        Document doc = null;
        try {
            try (InputStream is = this.getInputStream(resource);){
                DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
                dbf.setFeature("http://javax.xml.XMLConstants/feature/secure-processing", true);
                dbf.setNamespaceAware(true);
                DocumentBuilder db = dbf.newDocumentBuilder();
                doc = db.parse(is);
            }
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Got document: " + doc);
            }
        }
        catch (IOException | ParserConfigurationException | SAXException ex) {
            throw new TechnicalException("Cannot read document from resolved resource '" + resource.getUri().toString() + "'", (Throwable)ex);
        }
        return doc;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void setImageProperties(ImageFileResource fileResource) throws IOException {
        block9: {
            try (ImageInputStream in = ImageIO.createImageInputStream(new File(fileResource.getUri()));){
                Iterator<ImageReader> readers = ImageIO.getImageReaders(in);
                if (!readers.hasNext()) break block9;
                ImageReader reader = readers.next();
                try {
                    reader.setInput(in);
                    fileResource.setWidth(reader.getWidth(0));
                    fileResource.setHeight(reader.getHeight(0));
                }
                finally {
                    reader.dispose();
                }
            }
        }
    }

    public InputStream getInputStream(URI resourceUri) throws TechnicalException, ResourceNotFoundException {
        try {
            String location = resourceUri.toString();
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Getting inputstream for location '{}'.", (Object)location);
            }
            Resource resource = this.resourceLoader.getResource(location);
            if (!resourceUri.getScheme().startsWith("http") && !resource.exists()) {
                throw new ResourceNotFoundException("Resource not found at location '" + location + "'");
            }
            return resource.getInputStream();
        }
        catch (IOException e) {
            throw new TechnicalException((Throwable)e);
        }
    }

    public InputStream getInputStream(FileResource resource) throws TechnicalException, ResourceNotFoundException {
        return this.getInputStream(resource.getUri());
    }

    protected long getLastModified(Resource springResource) {
        try {
            return springResource.lastModified();
        }
        catch (FileNotFoundException e) {
            LOGGER.warn("Resource " + springResource.toString() + " does not exist.");
        }
        catch (IOException ex) {
            LOGGER.warn("Can not get lastModified for resource " + springResource.toString(), (Throwable)ex);
        }
        return -1L;
    }

    public Reader getReader(FileResource resource, Charset charset) throws TechnicalException, ResourceNotFoundException {
        return new InputStreamReader(this.getInputStream(resource), charset);
    }

    protected long getSize(Resource springResource) {
        try {
            long length = springResource.contentLength();
            return length;
        }
        catch (IOException ex) {
            LOGGER.warn("Can not get size for resource " + springResource.toString(), (Throwable)ex);
            return -1L;
        }
    }

    protected String getSplittedUuidPath(String uuid) {
        String uuidWithoutDashes = uuid.replaceAll("-", "");
        CharSequence[] pathParts = this.splitEqually(uuidWithoutDashes, 4);
        String splittedUuidPath = String.join((CharSequence)File.separator, pathParts);
        return splittedUuidPath;
    }

    public FileResource save(FileResource fileResource, InputStream binaryData) throws TechnicalException {
        Assert.notNull((Object)fileResource, (String)"fileResource must not be null");
        Assert.notNull((Object)binaryData, (String)"binaryData must not be null");
        try {
            if (fileResource.isReadonly()) {
                throw new TechnicalException("fileResource is read only, does not support write-operations.");
            }
            URI uri = this.createUri(fileResource.getUuid(), fileResource.getMimeType());
            fileResource.setUri(uri);
            Path parentDirectory = Paths.get(uri).getParent();
            if (parentDirectory == null) {
                throw new TechnicalException("No parent directory defined for uri: " + uri);
            }
            Files.createDirectories(parentDirectory, new FileAttribute[0]);
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Writing: " + uri);
            }
            long size = IOUtils.copyLarge((InputStream)binaryData, (OutputStream)new FileOutputStream(Paths.get(uri).toFile()));
            fileResource.setSizeInBytes(size);
            if (fileResource instanceof ImageFileResource) {
                ImageFileResource imageFileResource = (ImageFileResource)fileResource;
                this.setImageProperties(imageFileResource);
                this.setIiifProperties(imageFileResource);
            } else if (fileResource instanceof VideoFileResource) {
                VideoFileResource videoFileResource = (VideoFileResource)fileResource;
                this.setVideoProperties(videoFileResource);
            }
        }
        catch (IOException ex) {
            String msg = "Error writing binary data of fileresource " + fileResource.getUuid().toString();
            throw new TechnicalException(msg, (Throwable)ex);
        }
        return fileResource;
    }

    public FileResource save(FileResource resource, String input, Charset charset) throws TechnicalException {
        FileResource fileResource;
        ReaderInputStream in = new ReaderInputStream((Reader)new StringReader(input), charset);
        try {
            fileResource = this.save(resource, (InputStream)in);
        }
        catch (Throwable throwable) {
            try {
                try {
                    in.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException ex) {
                String msg = "Could not write data to uri " + String.valueOf(resource.getUri());
                LOGGER.error(msg, (Throwable)ex);
                throw new TechnicalException(msg, (Throwable)ex);
            }
        }
        in.close();
        return fileResource;
    }

    private String[] splitEqually(String text, int partLength) {
        if (!StringUtils.hasText((String)text) || partLength == 0) {
            return new String[]{text};
        }
        int textLength = text.length();
        int numberOfParts = (textLength + partLength - 1) / partLength;
        String[] parts = new String[numberOfParts];
        int offset = 0;
        for (int i = 0; i < numberOfParts; ++i) {
            parts[i] = text.substring(offset, Math.min(offset + partLength, textLength));
            offset += partLength;
        }
        return parts;
    }

    private void setIiifProperties(ImageFileResource imageFileResource) {
        if (this.iiifImageBaseUrl != null) {
            try {
                Object iiifUrl = this.iiifImageBaseUrl.toString();
                if (!((String)iiifUrl).endsWith("/")) {
                    iiifUrl = (String)iiifUrl + "/";
                }
                iiifUrl = (String)iiifUrl + imageFileResource.getUuid().toString();
                imageFileResource.setHttpBaseUrl(URI.create((String)iiifUrl).toURL());
            }
            catch (MalformedURLException ex) {
                throw new IllegalStateException("Creating a valid iiif url failed! Check configuration!", ex);
            }
        }
    }

    private void setVideoProperties(VideoFileResource videoFileResource) {
        if (this.mediaVideoBaseUrl != null) {
            try {
                Object videoUrl = this.mediaVideoBaseUrl.toString();
                if (!((String)videoUrl).endsWith("/")) {
                    videoUrl = (String)videoUrl + "/";
                }
                videoUrl = (String)videoUrl + videoFileResource.getUuid().toString();
                videoFileResource.setHttpBaseUrl(URI.create((String)videoUrl).toURL());
            }
            catch (MalformedURLException ex) {
                throw new IllegalStateException("Creating a valid video url failed! Check configuration!", ex);
            }
        }
    }
}

