package de.devland.recruiting.robot.parser;

import de.devland.recruiting.robot.commands.*;

import java.io.File;
import java.io.IOException;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class CommandParser {

    private Scanner scanner;
    private final Stack<List<Command>> commandLists = new Stack<>();
    private int level;

    public List<Command> getParsedCommands() {
        assert commandLists.size() == 1;
        return commandLists.peek();
    }

    public int getLevel() {
        return level;
    }

    public void parse(File file) throws ParserException, IOException {
        scanner = new Scanner(file);
        level = robot();
        braceOpen();
        commandLists.push(new ArrayList<>());
        commands();
        braceClosed();

        assert !scanner.hasNext();
    }

    private int robot() throws ParserException {
        Pattern robotPattern = Pattern.compile("robot\\((\\d+)\\)");
        Matcher matcher = robotPattern.matcher(scanner.next());
        if (!matcher.matches()) {
            throw new ParserException("robot command not recognized");
        }
        return Integer.parseInt(matcher.group(1));
    }

    private void braceOpen() throws ParserException {
        if (!Objects.equals(scanner.next(), "{")) {
            throw new ParserException("{ expected");
        }
    }

    private void braceClosed() throws ParserException {
        String next = scanner.next();
        if (!Objects.equals(next, "}")) {
            throw new ParserException("} expected");
        }
    }

    private void commands() throws ParserException {
        int repeat = repeat();
        if (repeat >= 0) {
            braceOpen();
            commandLists.push(new ArrayList<>());
            commands();
            braceClosed();
            RepeatCommand cmd = new RepeatCommand(repeat, commandLists.pop());
            commandLists.peek().add(cmd);
            commands();
        } else {
            Optional<Command> command = command();
            if (command.isPresent()) {
                commandLists.peek().add(command.get());
                commands();
            }
        }
    }

    private int repeat() {
        Pattern repeatPattern = Pattern.compile("repeat\\((\\d+)\\)");
        if (scanner.hasNext(repeatPattern)) {
            Matcher matcher = repeatPattern.matcher(scanner.next());
            if (matcher.matches()) {
                return Integer.parseInt(matcher.group(1));
            }
        }

        return -1;
    }

    private Optional<Command> command() {
        Pattern commandPattern = Pattern.compile("(walk|light|left|jump|right)\\(\\);");
        if (scanner.hasNext(commandPattern)) {
            Matcher matcher = commandPattern.matcher(scanner.next());
            if (matcher.matches()) {
                return switch (matcher.group(1)) {
                    case "walk" -> Optional.of(new WalkCommand());
                    case "light" -> Optional.of(new LightCommand());
                    case "left" -> Optional.of(new LeftCommand());
                    case "jump" -> Optional.of(new JumpCommand());
                    case "right" -> Optional.of(new RightCommand());
                    default -> Optional.empty();
                };
            }
        }

        return Optional.empty();
    }
}
