/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.milo.client.internal;

import java.io.StringWriter;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import javax.xml.stream.XMLStreamException;
import org.apache.camel.component.milo.NamespaceId;
import org.apache.camel.component.milo.PartialNodeId;
import org.apache.camel.component.milo.client.MiloClientConfiguration;
import org.eclipse.milo.opcua.sdk.client.OpcUaClient;
import org.eclipse.milo.opcua.sdk.client.api.config.OpcUaClientConfigBuilder;
import org.eclipse.milo.opcua.sdk.client.api.identity.AnonymousProvider;
import org.eclipse.milo.opcua.sdk.client.api.identity.CompositeProvider;
import org.eclipse.milo.opcua.sdk.client.api.identity.IdentityProvider;
import org.eclipse.milo.opcua.sdk.client.api.identity.UsernameProvider;
import org.eclipse.milo.opcua.sdk.client.api.subscriptions.UaMonitoredItem;
import org.eclipse.milo.opcua.sdk.client.api.subscriptions.UaSubscription;
import org.eclipse.milo.opcua.sdk.client.api.subscriptions.UaSubscriptionManager;
import org.eclipse.milo.opcua.stack.client.UaTcpStackClient;
import org.eclipse.milo.opcua.stack.core.AttributeId;
import org.eclipse.milo.opcua.stack.core.Identifiers;
import org.eclipse.milo.opcua.stack.core.UaException;
import org.eclipse.milo.opcua.stack.core.serialization.xml.XmlEncoder;
import org.eclipse.milo.opcua.stack.core.types.builtin.DataValue;
import org.eclipse.milo.opcua.stack.core.types.builtin.DateTime;
import org.eclipse.milo.opcua.stack.core.types.builtin.NodeId;
import org.eclipse.milo.opcua.stack.core.types.builtin.QualifiedName;
import org.eclipse.milo.opcua.stack.core.types.builtin.StatusCode;
import org.eclipse.milo.opcua.stack.core.types.builtin.unsigned.UInteger;
import org.eclipse.milo.opcua.stack.core.types.builtin.unsigned.UShort;
import org.eclipse.milo.opcua.stack.core.types.builtin.unsigned.Unsigned;
import org.eclipse.milo.opcua.stack.core.types.enumerated.MonitoringMode;
import org.eclipse.milo.opcua.stack.core.types.enumerated.TimestampsToReturn;
import org.eclipse.milo.opcua.stack.core.types.structured.EndpointDescription;
import org.eclipse.milo.opcua.stack.core.types.structured.MonitoredItemCreateRequest;
import org.eclipse.milo.opcua.stack.core.types.structured.MonitoringParameters;
import org.eclipse.milo.opcua.stack.core.types.structured.ReadValueId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SubscriptionManager {
    private static final Logger LOG = LoggerFactory.getLogger(SubscriptionManager.class);
    private final AtomicLong clientHandleCounter = new AtomicLong(0L);
    private final MiloClientConfiguration configuration;
    private final OpcUaClientConfigBuilder clientBuilder;
    private final ScheduledExecutorService executor;
    private final long reconnectTimeout;
    private Connected connected;
    private boolean disposed;
    private Future<?> reconnectJob;
    private final Map<UInteger, Subscription> subscriptions = new HashMap<UInteger, Subscription>();

    public SubscriptionManager(MiloClientConfiguration configuration, OpcUaClientConfigBuilder clientBuilder, ScheduledExecutorService executor, long reconnectTimeout) {
        this.configuration = configuration;
        this.clientBuilder = clientBuilder;
        this.executor = executor;
        this.reconnectTimeout = reconnectTimeout;
        this.connect();
    }

    private synchronized void handleConnectionFailue(Throwable e) {
        if (this.connected != null) {
            this.connected.dispose();
            this.connected = null;
        }
        LOG.info("Connection failed", e);
        this.triggerReconnect(true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void connect() {
        LOG.info("Starting connect");
        SubscriptionManager subscriptionManager = this;
        synchronized (subscriptionManager) {
            this.reconnectJob = null;
            if (this.disposed) {
                return;
            }
        }
        try {
            Connected connected = this.performConnect();
            LOG.debug("Connect call done");
            SubscriptionManager subscriptionManager2 = this;
            synchronized (subscriptionManager2) {
                if (this.disposed) {
                    return;
                }
                try {
                    LOG.debug("Setting subscriptions: {}", (Object)this.subscriptions.size());
                    connected.putSubscriptions(this.subscriptions);
                    LOG.debug("Update state : {} -> {}", (Object)this.connected, (Object)connected);
                    Connected oldConnected = this.connected;
                    this.connected = connected;
                    if (oldConnected != null) {
                        LOG.debug("Dispose old state");
                        oldConnected.dispose();
                    }
                }
                catch (Exception e) {
                    LOG.info("Failed to set subscriptions", (Throwable)e);
                    connected.dispose();
                    throw e;
                }
            }
        }
        catch (Exception e) {
            LOG.info("Failed to connect", (Throwable)e);
            this.triggerReconnect(false);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void dispose() {
        Connected connected;
        SubscriptionManager subscriptionManager = this;
        synchronized (subscriptionManager) {
            if (this.disposed) {
                return;
            }
            this.disposed = true;
            connected = this.connected;
        }
        if (connected != null) {
            connected.dispose();
        }
    }

    private synchronized void triggerReconnect(boolean immediate) {
        LOG.info("Trigger re-connect (immediate: {})", (Object)immediate);
        if (this.reconnectJob != null) {
            LOG.info("Re-connect already scheduled");
            return;
        }
        this.reconnectJob = immediate ? this.executor.submit(this::connect) : this.executor.schedule(this::connect, this.reconnectTimeout, TimeUnit.MILLISECONDS);
    }

    private Connected performConnect() throws Exception {
        EndpointDescription endpoint = (EndpointDescription)((CompletableFuture)UaTcpStackClient.getEndpoints(this.configuration.getEndpointUri()).thenApply(endpoints -> {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Found enpoints:");
                for (EndpointDescription ep : endpoints) {
                    LOG.debug(this.toString(ep));
                }
            }
            return this.findEndpoint((EndpointDescription[])endpoints);
        })).get();
        LOG.debug("Selected endpoint: {}", (Object)this.toString(endpoint));
        URI uri = URI.create(this.configuration.getEndpointUri());
        LinkedList<IdentityProvider> providers = new LinkedList<IdentityProvider>();
        String user = uri.getUserInfo();
        if (user != null && !user.isEmpty()) {
            String[] creds = user.split(":", 2);
            if (creds != null && creds.length == 2) {
                LOG.debug("Enable username/password provider: {}", (Object)creds[0]);
            }
            providers.add(new UsernameProvider(creds[0], creds[1]));
        }
        OpcUaClientConfigBuilder cfg = this.clientBuilder;
        providers.add(new AnonymousProvider());
        cfg.setIdentityProvider(new CompositeProvider(providers));
        cfg.setEndpoint(endpoint);
        OpcUaClient client = new OpcUaClient(cfg.build());
        try {
            UaSubscription manager = client.getSubscriptionManager().createSubscription(1000.0).get();
            client.getSubscriptionManager().addSubscriptionListener(new SubscriptionListenerImpl());
            return new Connected(client, manager);
        }
        catch (Throwable e) {
            if (client != null) {
                client.disconnect();
            }
            throw e;
        }
    }

    private EndpointDescription findEndpoint(EndpointDescription[] endpoints) {
        EndpointDescription best = null;
        for (EndpointDescription ep : endpoints) {
            if (best != null && ep.getSecurityLevel().compareTo(best.getSecurityLevel()) <= 0) continue;
            best = ep;
        }
        return best;
    }

    private String toString(EndpointDescription ep) {
        StringWriter sw = new StringWriter();
        try {
            EndpointDescription.encode(ep, new XmlEncoder(sw));
        }
        catch (XMLStreamException e) {
            return ep.toString();
        }
        return sw.toString();
    }

    protected synchronized void whenConnected(Worker<Connected> worker) {
        if (this.connected != null) {
            try {
                worker.work(this.connected);
            }
            catch (Exception e) {
                this.handleConnectionFailue(e);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public UInteger registerItem(NamespaceId namespaceId, PartialNodeId partialNodeId, Double samplingInterval, Consumer<DataValue> valueConsumer) {
        UInteger clientHandle = Unsigned.uint(this.clientHandleCounter.incrementAndGet());
        Subscription subscription = new Subscription(namespaceId, partialNodeId, samplingInterval, valueConsumer);
        SubscriptionManager subscriptionManager = this;
        synchronized (subscriptionManager) {
            this.subscriptions.put(clientHandle, subscription);
            this.whenConnected(connected -> connected.activate(clientHandle, subscription));
        }
        return clientHandle;
    }

    public synchronized void unregisterItem(UInteger clientHandle) {
        if (this.subscriptions.remove(clientHandle) != null) {
            this.whenConnected(connected -> connected.deactivate(clientHandle));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void write(NamespaceId namespaceId, PartialNodeId partialNodeId, DataValue value, boolean await) {
        CompletionStage future = null;
        SubscriptionManager subscriptionManager = this;
        synchronized (subscriptionManager) {
            if (this.connected != null) {
                future = this.connected.write(namespaceId, partialNodeId, value).handleAsync((status, e) -> {
                    if (e != null) {
                        this.handleConnectionFailue((Throwable)e);
                    }
                    return null;
                }, (Executor)this.executor);
            }
        }
        if (await && future != null) {
            try {
                future.get();
            }
            catch (InterruptedException | ExecutionException e2) {
                LOG.warn("Failed to wait for completion", (Throwable)e2);
            }
        }
    }

    private class Connected {
        private OpcUaClient client;
        private final UaSubscription manager;
        private final Map<UInteger, Subscription> badSubscriptions = new HashMap<UInteger, Subscription>();
        private final Map<UInteger, UaMonitoredItem> goodSubscriptions = new HashMap<UInteger, UaMonitoredItem>();
        private final Map<String, UShort> namespaceCache = new ConcurrentHashMap<String, UShort>();

        public Connected(OpcUaClient client, UaSubscription manager) {
            this.client = client;
            this.manager = manager;
        }

        public void putSubscriptions(Map<UInteger, Subscription> subscriptions) throws Exception {
            if (subscriptions.isEmpty()) {
                return;
            }
            ArrayList<MonitoredItemCreateRequest> items = new ArrayList<MonitoredItemCreateRequest>(subscriptions.size());
            for (Map.Entry<UInteger, Subscription> entry : subscriptions.entrySet()) {
                Subscription s = entry.getValue();
                UShort namespaceIndex = s.getNamespaceId().isNumeric() ? s.getNamespaceId().getNumeric() : this.lookupNamespace(s.getNamespaceId().getUri());
                if (namespaceIndex == null) {
                    this.handleSubscriptionError(new StatusCode(2158690304L), entry.getKey(), s);
                    continue;
                }
                NodeId nodeId = s.getPartialNodeId().toNodeId(namespaceIndex);
                ReadValueId itemId = new ReadValueId(nodeId, AttributeId.Value.uid(), null, QualifiedName.NULL_VALUE);
                MonitoringParameters parameters = new MonitoringParameters(entry.getKey(), s.getSamplingInterval(), null, null, null);
                items.add(new MonitoredItemCreateRequest(itemId, MonitoringMode.Reporting, parameters));
            }
            if (!items.isEmpty()) {
                this.manager.createMonitoredItems(TimestampsToReturn.Both, items, (item, idx) -> {
                    Subscription s = (Subscription)subscriptions.get(item.getClientHandle());
                    if (item.getStatusCode().isBad()) {
                        this.handleSubscriptionError(item.getStatusCode(), item.getClientHandle(), s);
                    } else {
                        this.goodSubscriptions.put(item.getClientHandle(), (UaMonitoredItem)item);
                        item.setValueConsumer(s.getValueConsumer());
                    }
                }).get();
            }
            if (!this.badSubscriptions.isEmpty()) {
                SubscriptionManager.this.executor.schedule(this::resubscribe, SubscriptionManager.this.reconnectTimeout, TimeUnit.MILLISECONDS);
            }
        }

        private void handleSubscriptionError(StatusCode statusCode, UInteger clientHandle, Subscription s) {
            this.badSubscriptions.put(clientHandle, s);
            s.getValueConsumer().accept(new DataValue(statusCode));
        }

        private void resubscribe() {
            HashMap<UInteger, Subscription> subscriptions = new HashMap<UInteger, Subscription>(this.badSubscriptions);
            this.badSubscriptions.clear();
            try {
                this.putSubscriptions(subscriptions);
            }
            catch (Exception e) {
                SubscriptionManager.this.handleConnectionFailue(e);
            }
        }

        public void activate(UInteger clientHandle, Subscription subscription) throws Exception {
            this.putSubscriptions(Collections.singletonMap(clientHandle, subscription));
        }

        public void deactivate(UInteger clientHandle) throws Exception {
            UaMonitoredItem item = this.goodSubscriptions.remove(clientHandle);
            if (item != null) {
                this.manager.deleteMonitoredItems(Collections.singletonList(item)).get();
            } else {
                this.badSubscriptions.remove(clientHandle);
            }
        }

        private UShort lookupNamespace(String namespaceUri) throws Exception {
            return this.lookupNamespaceIndex(namespaceUri).get();
        }

        private CompletableFuture<UShort> lookupNamespaceIndex(String namespaceUri) {
            LOG.trace("Looking up namespace: {}", (Object)namespaceUri);
            UShort result = this.namespaceCache.get(namespaceUri);
            if (result != null) {
                LOG.trace("Found namespace in cache: {} -> {}", (Object)namespaceUri, (Object)result);
                return CompletableFuture.completedFuture(result);
            }
            LOG.debug("Looking up namespace on server: {}", (Object)namespaceUri);
            CompletableFuture<DataValue> future = this.client.readValue(0.0, TimestampsToReturn.Neither, Identifiers.Server_NamespaceArray);
            return future.thenApply(value -> {
                Object rawValue = value.getValue().getValue();
                if (rawValue instanceof String[]) {
                    String[] namespaces = (String[])rawValue;
                    for (int i = 0; i < namespaces.length; ++i) {
                        if (!namespaces[i].equals(namespaceUri)) continue;
                        UShort result = Unsigned.ushort(i);
                        this.namespaceCache.putIfAbsent(namespaceUri, result);
                        return result;
                    }
                }
                return null;
            });
        }

        public void dispose() {
            if (this.client != null) {
                this.client.disconnect();
                this.client = null;
            }
        }

        public CompletableFuture<StatusCode> write(NamespaceId namespaceId, PartialNodeId partialNodeId, DataValue value) {
            CompletableFuture<UShort> future;
            LOG.trace("Namespace: {}", (Object)namespaceId);
            if (namespaceId.isNumeric()) {
                LOG.trace("Using provided index: {}", (Object)namespaceId.getNumeric());
                future = CompletableFuture.completedFuture(namespaceId.getNumeric());
            } else {
                LOG.trace("Looking up namespace: {}", (Object)namespaceId.getUri());
                future = this.lookupNamespaceIndex(namespaceId.getUri());
            }
            return future.thenCompose(index -> {
                NodeId nodeId = partialNodeId.toNodeId((UShort)index);
                LOG.debug("Node - partial: {}, full: {}", (Object)partialNodeId, (Object)nodeId);
                return this.client.writeValue(nodeId, value).whenComplete((status, error) -> {
                    if (status != null) {
                        LOG.debug("Write to ns={}/{}, id={} = {} -> {}", new Object[]{namespaceId, index, nodeId, value, status});
                    } else {
                        LOG.debug("Failed to write", error);
                    }
                });
            });
        }
    }

    private static class Subscription {
        private final NamespaceId namespaceId;
        private final PartialNodeId partialNodeId;
        private final Double samplingInterval;
        private final Consumer<DataValue> valueConsumer;

        public Subscription(NamespaceId namespaceId, PartialNodeId partialNodeId, Double samplingInterval, Consumer<DataValue> valueConsumer) {
            this.namespaceId = namespaceId;
            this.partialNodeId = partialNodeId;
            this.samplingInterval = samplingInterval;
            this.valueConsumer = valueConsumer;
        }

        public NamespaceId getNamespaceId() {
            return this.namespaceId;
        }

        public PartialNodeId getPartialNodeId() {
            return this.partialNodeId;
        }

        public Double getSamplingInterval() {
            return this.samplingInterval;
        }

        public Consumer<DataValue> getValueConsumer() {
            return this.valueConsumer;
        }
    }

    public static interface Worker<T> {
        public void work(T var1) throws Exception;
    }

    private final class SubscriptionListenerImpl
    implements UaSubscriptionManager.SubscriptionListener {
        private SubscriptionListenerImpl() {
        }

        @Override
        public void onSubscriptionTransferFailed(UaSubscription subscription, StatusCode statusCode) {
            LOG.info("Transfer failed {} : {}", (Object)subscription.getSubscriptionId(), (Object)statusCode);
            SubscriptionManager.this.handleConnectionFailue(new RuntimeException("Subscription failed to reconnect"));
        }

        @Override
        public void onStatusChanged(UaSubscription subscription, StatusCode status) {
            LOG.info("Subscription status changed {} : {}", (Object)subscription.getSubscriptionId(), (Object)status);
        }

        @Override
        public void onPublishFailure(UaException exception) {
        }

        @Override
        public void onNotificationDataLost(UaSubscription subscription) {
        }

        @Override
        public void onKeepAlive(UaSubscription subscription, DateTime publishTime) {
        }
    }
}

