/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.milo.opcua.stack.core.channel;

import io.netty.buffer.ByteBuf;
import java.nio.ByteBuffer;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;
import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.eclipse.milo.opcua.stack.core.UaException;
import org.eclipse.milo.opcua.stack.core.channel.ChannelParameters;
import org.eclipse.milo.opcua.stack.core.channel.ChannelSecurity;
import org.eclipse.milo.opcua.stack.core.channel.SecureChannel;
import org.eclipse.milo.opcua.stack.core.channel.headers.AsymmetricSecurityHeader;
import org.eclipse.milo.opcua.stack.core.channel.headers.SecureMessageHeader;
import org.eclipse.milo.opcua.stack.core.channel.headers.SequenceHeader;
import org.eclipse.milo.opcua.stack.core.channel.headers.SymmetricSecurityHeader;
import org.eclipse.milo.opcua.stack.core.channel.messages.MessageType;
import org.eclipse.milo.opcua.stack.core.security.SecurityAlgorithm;
import org.eclipse.milo.opcua.stack.core.util.BufferUtil;
import org.eclipse.milo.opcua.stack.core.util.LongSequence;
import org.eclipse.milo.opcua.stack.core.util.SignatureUtil;

public class ChunkEncoder {
    private final Delegate asymmetricDelegate = new AsymmetricDelegate();
    private final Delegate symmetricDelegate = new SymmetricDelegate();
    private final LongSequence sequenceNumber = new LongSequence(1L, 0xFFFFFBFFL);
    private volatile long lastRequestId = 1L;
    private final ChannelParameters parameters;

    public ChunkEncoder(ChannelParameters parameters) {
        this.parameters = parameters;
    }

    public List<ByteBuf> encodeAsymmetric(SecureChannel channel, MessageType messageType, ByteBuf messageBuffer, long requestId) throws UaException {
        return this.encode(this.asymmetricDelegate, channel, messageType, messageBuffer, requestId);
    }

    public List<ByteBuf> encodeSymmetric(SecureChannel channel, MessageType messageType, ByteBuf messageBuffer, long requestId) throws UaException {
        return this.encode(this.symmetricDelegate, channel, messageType, messageBuffer, requestId);
    }

    private List<ByteBuf> encode(Delegate delegate, SecureChannel channel, MessageType messageType, ByteBuf messageBuffer, long requestId) throws UaException {
        ArrayList<ByteBuf> chunks = new ArrayList<ByteBuf>();
        boolean encrypted = delegate.isEncryptionEnabled(channel);
        int securityHeaderSize = delegate.getSecurityHeaderSize(channel);
        int cipherTextBlockSize = delegate.getCipherTextBlockSize(channel);
        int plainTextBlockSize = delegate.getPlainTextBlockSize(channel);
        int signatureSize = delegate.getSignatureSize(channel);
        int maxChunkSize = this.parameters.getLocalSendBufferSize();
        int headerSizes = 12 + securityHeaderSize;
        int paddingOverhead = encrypted ? (cipherTextBlockSize > 256 ? 2 : 1) : 0;
        int maxBlockCount = (maxChunkSize - headerSizes - signatureSize - paddingOverhead) / cipherTextBlockSize;
        int maxBodySize = plainTextBlockSize * maxBlockCount - 8;
        while (messageBuffer.readableBytes() > 0) {
            int bodySize = Math.min(messageBuffer.readableBytes(), maxBodySize);
            int paddingSize = encrypted ? plainTextBlockSize - (8 + bodySize + signatureSize + paddingOverhead) % plainTextBlockSize : 0;
            int plainTextContentSize = 8 + bodySize + signatureSize + paddingSize + paddingOverhead;
            assert (plainTextContentSize % plainTextBlockSize == 0);
            int chunkSize = 12 + securityHeaderSize + plainTextContentSize / plainTextBlockSize * cipherTextBlockSize;
            ByteBuf chunkBuffer = BufferUtil.buffer(chunkSize);
            SecureMessageHeader messageHeader = new SecureMessageHeader(messageType, messageBuffer.readableBytes() > bodySize ? (char)'C' : 'F', chunkSize, channel.getChannelId());
            SecureMessageHeader.encode(messageHeader, chunkBuffer);
            delegate.encodeSecurityHeader(channel, chunkBuffer);
            SequenceHeader sequenceHeader = new SequenceHeader(this.sequenceNumber.getAndIncrement(), requestId);
            SequenceHeader.encode(sequenceHeader, chunkBuffer);
            chunkBuffer.writeBytes(messageBuffer, bodySize);
            if (encrypted) {
                this.writePadding(cipherTextBlockSize, paddingSize, chunkBuffer);
            }
            if (delegate.isSigningEnabled(channel)) {
                ByteBuffer chunkNioBuffer = chunkBuffer.nioBuffer(0, chunkBuffer.writerIndex());
                byte[] signature = delegate.signChunk(channel, chunkNioBuffer);
                chunkBuffer.writeBytes(signature);
            }
            if (encrypted) {
                chunkBuffer.readerIndex(12 + securityHeaderSize);
                assert (chunkBuffer.readableBytes() % plainTextBlockSize == 0);
                try {
                    int blockCount = chunkBuffer.readableBytes() / plainTextBlockSize;
                    ByteBuffer chunkNioBuffer = chunkBuffer.nioBuffer(chunkBuffer.readerIndex(), blockCount * cipherTextBlockSize);
                    ByteBuf copyBuffer = chunkBuffer.copy();
                    ByteBuffer plainTextNioBuffer = copyBuffer.nioBuffer();
                    Cipher cipher = delegate.getAndInitializeCipher(channel);
                    if (delegate instanceof AsymmetricDelegate) {
                        for (int blockNumber = 0; blockNumber < blockCount; ++blockNumber) {
                            int position = blockNumber * plainTextBlockSize;
                            int limit = (blockNumber + 1) * plainTextBlockSize;
                            plainTextNioBuffer.position(position).limit(limit);
                            int bytesWritten = cipher.doFinal(plainTextNioBuffer, chunkNioBuffer);
                            assert (bytesWritten == cipherTextBlockSize);
                        }
                    } else {
                        cipher.doFinal(plainTextNioBuffer, chunkNioBuffer);
                    }
                    copyBuffer.release();
                }
                catch (GeneralSecurityException e) {
                    throw new UaException(2148728832L, (Throwable)e);
                }
            }
            chunkBuffer.readerIndex(0).writerIndex(chunkSize);
            chunks.add(chunkBuffer);
        }
        this.lastRequestId = requestId;
        return chunks;
    }

    public long getLastRequestId() {
        return this.lastRequestId;
    }

    private void writePadding(int cipherTextBlockSize, int paddingSize, ByteBuf buffer) {
        if (cipherTextBlockSize > 256) {
            buffer.writeShort(paddingSize);
        } else {
            buffer.writeByte(paddingSize);
        }
        for (int i = 0; i < paddingSize; ++i) {
            buffer.writeByte(paddingSize);
        }
        if (cipherTextBlockSize > 256) {
            int paddingLengthMsb = paddingSize >> 8;
            buffer.writerIndex(buffer.writerIndex() - 1);
            buffer.writeByte(paddingLengthMsb);
        }
    }

    private static class SymmetricDelegate
    implements Delegate {
        private volatile ChannelSecurity.SecuritySecrets securitySecrets;

        private SymmetricDelegate() {
        }

        @Override
        public void encodeSecurityHeader(SecureChannel channel, ByteBuf buffer) {
            ChannelSecurity channelSecurity = channel.getChannelSecurity();
            long tokenId = channelSecurity != null ? channelSecurity.getCurrentToken().getTokenId().longValue() : 0L;
            SymmetricSecurityHeader.encode(new SymmetricSecurityHeader(tokenId), buffer);
            this.securitySecrets = channelSecurity != null ? channelSecurity.getCurrentKeys() : null;
        }

        @Override
        public byte[] signChunk(SecureChannel channel, ByteBuffer chunkNioBuffer) throws UaException {
            SecurityAlgorithm signatureAlgorithm = channel.getSecurityPolicy().getSymmetricSignatureAlgorithm();
            byte[] signatureKey = channel.getEncryptionKeys(this.securitySecrets).getSignatureKey();
            return SignatureUtil.hmac(signatureAlgorithm, signatureKey, chunkNioBuffer);
        }

        @Override
        public Cipher getAndInitializeCipher(SecureChannel channel) throws UaException {
            try {
                String transformation = channel.getSecurityPolicy().getSymmetricEncryptionAlgorithm().getTransformation();
                ChannelSecurity.SecretKeys secretKeys = channel.getEncryptionKeys(this.securitySecrets);
                SecretKeySpec keySpec = new SecretKeySpec(secretKeys.getEncryptionKey(), "AES");
                IvParameterSpec ivSpec = new IvParameterSpec(secretKeys.getInitializationVector());
                Cipher cipher = Cipher.getInstance(transformation);
                cipher.init(1, (Key)keySpec, ivSpec);
                assert (cipher.getBlockSize() == channel.getSymmetricCipherTextBlockSize());
                return cipher;
            }
            catch (GeneralSecurityException e) {
                throw new UaException(2148728832L, (Throwable)e);
            }
        }

        @Override
        public int getSecurityHeaderSize(SecureChannel channel) {
            return 4;
        }

        @Override
        public int getCipherTextBlockSize(SecureChannel channel) {
            return channel.getSymmetricCipherTextBlockSize();
        }

        @Override
        public int getPlainTextBlockSize(SecureChannel channel) {
            return channel.getSymmetricPlainTextBlockSize();
        }

        @Override
        public int getSignatureSize(SecureChannel channel) {
            return channel.getSymmetricSignatureSize();
        }

        @Override
        public boolean isEncryptionEnabled(SecureChannel channel) {
            return channel.isSymmetricEncryptionEnabled();
        }

        @Override
        public boolean isSigningEnabled(SecureChannel channel) {
            return channel.isSymmetricSigningEnabled();
        }
    }

    private static class AsymmetricDelegate
    implements Delegate {
        private AsymmetricDelegate() {
        }

        @Override
        public byte[] signChunk(SecureChannel channel, ByteBuffer chunkNioBuffer) throws UaException {
            return SignatureUtil.sign(channel.getSecurityPolicy().getAsymmetricSignatureAlgorithm(), channel.getKeyPair().getPrivate(), chunkNioBuffer);
        }

        @Override
        public Cipher getAndInitializeCipher(SecureChannel channel) throws UaException {
            X509Certificate remoteCertificate = channel.getRemoteCertificate();
            assert (remoteCertificate != null);
            try {
                String transformation = channel.getSecurityPolicy().getAsymmetricEncryptionAlgorithm().getTransformation();
                Cipher cipher = Cipher.getInstance(transformation);
                cipher.init(1, remoteCertificate.getPublicKey());
                return cipher;
            }
            catch (GeneralSecurityException e) {
                throw new UaException(2148728832L, (Throwable)e);
            }
        }

        @Override
        public void encodeSecurityHeader(SecureChannel channel, ByteBuf buffer) throws UaException {
            AsymmetricSecurityHeader header = new AsymmetricSecurityHeader(channel.getSecurityPolicy().getSecurityPolicyUri(), channel.getLocalCertificateBytes(), channel.getRemoteCertificateThumbprint());
            AsymmetricSecurityHeader.encode(header, buffer);
        }

        @Override
        public int getSecurityHeaderSize(SecureChannel channel) throws UaException {
            String securityPolicyUri = channel.getSecurityPolicy().getSecurityPolicyUri();
            byte[] localCertificateBytes = channel.getLocalCertificateBytes().bytes();
            byte[] remoteCertificateThumbprint = channel.getRemoteCertificateThumbprint().bytes();
            return 12 + securityPolicyUri.length() + (localCertificateBytes != null ? localCertificateBytes.length : 0) + (remoteCertificateThumbprint != null ? remoteCertificateThumbprint.length : 0);
        }

        @Override
        public int getCipherTextBlockSize(SecureChannel channel) {
            return channel.getRemoteAsymmetricCipherTextBlockSize();
        }

        @Override
        public int getPlainTextBlockSize(SecureChannel channel) {
            return channel.getRemoteAsymmetricPlainTextBlockSize();
        }

        @Override
        public int getSignatureSize(SecureChannel channel) {
            return channel.getLocalAsymmetricSignatureSize();
        }

        @Override
        public boolean isEncryptionEnabled(SecureChannel channel) {
            return channel.isAsymmetricEncryptionEnabled();
        }

        @Override
        public boolean isSigningEnabled(SecureChannel channel) {
            return channel.isAsymmetricSigningEnabled();
        }
    }

    private static interface Delegate {
        public byte[] signChunk(SecureChannel var1, ByteBuffer var2) throws UaException;

        public void encodeSecurityHeader(SecureChannel var1, ByteBuf var2) throws UaException;

        public Cipher getAndInitializeCipher(SecureChannel var1) throws UaException;

        public int getSecurityHeaderSize(SecureChannel var1) throws UaException;

        public int getCipherTextBlockSize(SecureChannel var1);

        public int getPlainTextBlockSize(SecureChannel var1);

        public int getSignatureSize(SecureChannel var1);

        public boolean isEncryptionEnabled(SecureChannel var1);

        public boolean isSigningEnabled(SecureChannel var1);
    }
}

