/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.milo.opcua.stack.core.channel;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.CompositeByteBuf;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.Signature;
import java.security.SignatureException;
import java.util.Arrays;
import java.util.List;
import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.eclipse.milo.opcua.stack.core.UaException;
import org.eclipse.milo.opcua.stack.core.channel.ChannelParameters;
import org.eclipse.milo.opcua.stack.core.channel.ChannelSecurity;
import org.eclipse.milo.opcua.stack.core.channel.MessageAbortedException;
import org.eclipse.milo.opcua.stack.core.channel.SecureChannel;
import org.eclipse.milo.opcua.stack.core.channel.headers.AsymmetricSecurityHeader;
import org.eclipse.milo.opcua.stack.core.channel.headers.SequenceHeader;
import org.eclipse.milo.opcua.stack.core.channel.headers.SymmetricSecurityHeader;
import org.eclipse.milo.opcua.stack.core.channel.messages.ErrorMessage;
import org.eclipse.milo.opcua.stack.core.security.SecurityAlgorithm;
import org.eclipse.milo.opcua.stack.core.util.BufferUtil;
import org.eclipse.milo.opcua.stack.core.util.SignatureUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChunkDecoder {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final Delegate asymmetricDelegate = new AsymmetricDelegate();
    private final Delegate symmetricDelegate = new SymmetricDelegate();
    private volatile long lastSequenceNumber = -1L;
    private volatile long lastRequestId;
    private final ChannelParameters parameters;

    public ChunkDecoder(ChannelParameters parameters) {
        this.parameters = parameters;
    }

    public ByteBuf decodeAsymmetric(SecureChannel channel, List<ByteBuf> chunkBuffers) throws UaException {
        return this.decode(this.asymmetricDelegate, channel, chunkBuffers);
    }

    public ByteBuf decodeSymmetric(SecureChannel channel, List<ByteBuf> chunkBuffers) throws UaException {
        return this.decode(this.symmetricDelegate, channel, chunkBuffers);
    }

    private ByteBuf decode(Delegate delegate, SecureChannel channel, List<ByteBuf> chunkBuffers) throws UaException {
        CompositeByteBuf composite = BufferUtil.compositeBuffer();
        int signatureSize = delegate.getSignatureSize(channel);
        int cipherTextBlockSize = delegate.getCipherTextBlockSize(channel);
        boolean encrypted = delegate.isEncryptionEnabled(channel);
        boolean signed = delegate.isSigningEnabled(channel);
        for (ByteBuf chunkBuffer : chunkBuffers) {
            char chunkType = (char)chunkBuffer.getByte(3);
            chunkBuffer.skipBytes(12);
            delegate.readSecurityHeader(channel, chunkBuffer);
            if (encrypted) {
                this.decryptChunk(delegate, channel, chunkBuffer);
            }
            int encryptedStart = chunkBuffer.readerIndex();
            chunkBuffer.readerIndex(0);
            if (signed) {
                delegate.verifyChunk(channel, chunkBuffer);
            }
            int paddingSize = encrypted ? this.getPaddingSize(cipherTextBlockSize, signatureSize, chunkBuffer) : 0;
            int bodyEnd = chunkBuffer.readableBytes() - signatureSize - paddingSize;
            chunkBuffer.readerIndex(encryptedStart);
            SequenceHeader sequenceHeader = SequenceHeader.decode(chunkBuffer);
            long sequenceNumber = sequenceHeader.getSequenceNumber();
            this.lastRequestId = sequenceHeader.getRequestId();
            if (this.lastSequenceNumber == -1L) {
                this.lastSequenceNumber = sequenceNumber;
            } else {
                if (this.lastSequenceNumber + 1L != sequenceNumber) {
                    String message = String.format("expected sequence number %s but received %s", this.lastSequenceNumber + 1L, sequenceNumber);
                    this.logger.error(message);
                    this.logger.error(ByteBufUtil.hexDump((ByteBuf)chunkBuffer, (int)0, (int)chunkBuffer.writerIndex()));
                    throw new UaException(2148728832L, message);
                }
                this.lastSequenceNumber = sequenceNumber;
            }
            ByteBuf bodyBuffer = chunkBuffer.readSlice(bodyEnd - chunkBuffer.readerIndex());
            if (chunkType == 'A') {
                ErrorMessage errorMessage = ErrorMessage.decode(bodyBuffer);
                throw new MessageAbortedException(errorMessage.getError(), errorMessage.getReason());
            }
            composite.addComponent(bodyBuffer);
            composite.writerIndex(composite.writerIndex() + bodyBuffer.readableBytes());
        }
        return composite.order(ByteOrder.LITTLE_ENDIAN);
    }

    public long getLastRequestId() {
        return this.lastRequestId;
    }

    private void decryptChunk(Delegate delegate, SecureChannel channel, ByteBuf chunkBuffer) throws UaException {
        int cipherTextBlockSize = delegate.getCipherTextBlockSize(channel);
        int blockCount = chunkBuffer.readableBytes() / cipherTextBlockSize;
        int plainTextBufferSize = cipherTextBlockSize * blockCount;
        ByteBuf plainTextBuffer = BufferUtil.buffer(plainTextBufferSize);
        ByteBuffer plainTextNioBuffer = plainTextBuffer.writerIndex(plainTextBufferSize).nioBuffer();
        ByteBuffer chunkNioBuffer = chunkBuffer.nioBuffer();
        try {
            Cipher cipher = delegate.getCipher(channel);
            assert (chunkBuffer.readableBytes() % cipherTextBlockSize == 0);
            if (delegate instanceof AsymmetricDelegate) {
                for (int blockNumber = 0; blockNumber < blockCount; ++blockNumber) {
                    chunkNioBuffer.limit(chunkNioBuffer.position() + cipherTextBlockSize);
                    cipher.doFinal(chunkNioBuffer, plainTextNioBuffer);
                }
            } else {
                cipher.doFinal(chunkNioBuffer, plainTextNioBuffer);
            }
        }
        catch (GeneralSecurityException e) {
            throw new UaException(2148728832L, (Throwable)e);
        }
        plainTextNioBuffer.flip();
        chunkBuffer.writerIndex(chunkBuffer.readerIndex());
        chunkBuffer.writeBytes(plainTextNioBuffer);
        plainTextBuffer.release();
    }

    private int getPaddingSize(int cipherTextBlockSize, int signatureSize, ByteBuf buffer) {
        int lastPaddingByteOffset = buffer.readableBytes() - signatureSize - 1;
        return cipherTextBlockSize <= 256 ? buffer.getUnsignedByte(lastPaddingByteOffset) + 1 : buffer.getUnsignedShort(lastPaddingByteOffset - 1) + 2;
    }

    private static class SymmetricDelegate
    implements Delegate {
        private final Logger logger = LoggerFactory.getLogger(this.getClass());
        private volatile ChannelSecurity.SecuritySecrets securitySecrets;

        private SymmetricDelegate() {
        }

        @Override
        public void readSecurityHeader(SecureChannel channel, ByteBuf chunkBuffer) throws UaException {
            long receivedTokenId = SymmetricSecurityHeader.decode(chunkBuffer).getTokenId();
            ChannelSecurity channelSecurity = channel.getChannelSecurity();
            if (channelSecurity == null) {
                if (receivedTokenId != 0L) {
                    throw new UaException(0x80870000L, "unknown secure channel token: " + receivedTokenId);
                }
            } else {
                long currentTokenId = channelSecurity.getCurrentToken().getTokenId().longValue();
                if (receivedTokenId == currentTokenId) {
                    this.securitySecrets = channelSecurity.getCurrentKeys();
                } else {
                    long previousTokenId = channelSecurity.getPreviousToken().map(t -> t.getTokenId().longValue()).orElse(-1L);
                    this.logger.debug("Attempting to use SecuritySecrets from previousTokenId={}", (Object)previousTokenId);
                    if (receivedTokenId != previousTokenId) {
                        this.logger.warn("receivedTokenId={} did not match previousTokenId={}", (Object)receivedTokenId, (Object)previousTokenId);
                        throw new UaException(0x80870000L, "unknown secure channel token: " + receivedTokenId);
                    }
                    if (channel.isSymmetricEncryptionEnabled() && channelSecurity.getPreviousKeys().isPresent()) {
                        this.securitySecrets = channelSecurity.getPreviousKeys().get();
                    }
                }
            }
        }

        @Override
        public Cipher getCipher(SecureChannel channel) throws UaException {
            try {
                String transformation = channel.getSecurityPolicy().getSymmetricEncryptionAlgorithm().getTransformation();
                ChannelSecurity.SecretKeys decryptionKeys = channel.getDecryptionKeys(this.securitySecrets);
                SecretKeySpec keySpec = new SecretKeySpec(decryptionKeys.getEncryptionKey(), "AES");
                IvParameterSpec ivSpec = new IvParameterSpec(decryptionKeys.getInitializationVector());
                Cipher cipher = Cipher.getInstance(transformation);
                cipher.init(2, (Key)keySpec, ivSpec);
                return cipher;
            }
            catch (GeneralSecurityException e) {
                throw new UaException(2148728832L, (Throwable)e);
            }
        }

        @Override
        public int getCipherTextBlockSize(SecureChannel channel) {
            return channel.getSymmetricCipherTextBlockSize();
        }

        @Override
        public int getSignatureSize(SecureChannel channel) {
            return channel.getSymmetricSignatureSize();
        }

        @Override
        public void verifyChunk(SecureChannel channel, ByteBuf chunkBuffer) throws UaException {
            SecurityAlgorithm securityAlgorithm = channel.getSecurityPolicy().getSymmetricSignatureAlgorithm();
            byte[] secretKey = channel.getDecryptionKeys(this.securitySecrets).getSignatureKey();
            int signatureSize = channel.getSymmetricSignatureSize();
            ByteBuffer chunkNioBuffer = chunkBuffer.nioBuffer(0, chunkBuffer.writerIndex());
            chunkNioBuffer.position(0).limit(chunkBuffer.writerIndex() - signatureSize);
            byte[] signature = SignatureUtil.hmac(securityAlgorithm, secretKey, chunkNioBuffer);
            byte[] signatureBytes = new byte[signatureSize];
            chunkNioBuffer.limit(chunkNioBuffer.position() + signatureSize);
            chunkNioBuffer.get(signatureBytes);
            if (!Arrays.equals(signature, signatureBytes)) {
                throw new UaException(2148728832L, "could not verify signature");
            }
        }

        @Override
        public boolean isEncryptionEnabled(SecureChannel channel) {
            return channel.isSymmetricEncryptionEnabled();
        }

        @Override
        public boolean isSigningEnabled(SecureChannel channel) {
            return channel.isSymmetricSigningEnabled();
        }
    }

    private static class AsymmetricDelegate
    implements Delegate {
        private AsymmetricDelegate() {
        }

        @Override
        public void readSecurityHeader(SecureChannel channel, ByteBuf chunkBuffer) {
            AsymmetricSecurityHeader.decode(chunkBuffer);
        }

        @Override
        public Cipher getCipher(SecureChannel channel) throws UaException {
            try {
                String transformation = channel.getSecurityPolicy().getAsymmetricEncryptionAlgorithm().getTransformation();
                Cipher cipher = Cipher.getInstance(transformation);
                cipher.init(2, channel.getKeyPair().getPrivate());
                return cipher;
            }
            catch (GeneralSecurityException e) {
                throw new UaException(2148728832L, (Throwable)e);
            }
        }

        @Override
        public int getCipherTextBlockSize(SecureChannel channel) {
            return channel.getLocalAsymmetricCipherTextBlockSize();
        }

        @Override
        public int getSignatureSize(SecureChannel channel) {
            return channel.getRemoteAsymmetricSignatureSize();
        }

        @Override
        public void verifyChunk(SecureChannel channel, ByteBuf chunkBuffer) throws UaException {
            String transformation = channel.getSecurityPolicy().getAsymmetricSignatureAlgorithm().getTransformation();
            int signatureSize = channel.getRemoteAsymmetricSignatureSize();
            ByteBuffer chunkNioBuffer = chunkBuffer.nioBuffer(0, chunkBuffer.writerIndex());
            chunkNioBuffer.position(0).limit(chunkBuffer.writerIndex() - signatureSize);
            try {
                Signature signature = Signature.getInstance(transformation);
                signature.initVerify(channel.getRemoteCertificate().getPublicKey());
                signature.update(chunkNioBuffer);
                byte[] signatureBytes = new byte[signatureSize];
                chunkNioBuffer.limit(chunkNioBuffer.position() + signatureSize);
                chunkNioBuffer.get(signatureBytes);
                if (!signature.verify(signatureBytes)) {
                    throw new UaException(2148728832L, "could not verify signature");
                }
            }
            catch (NoSuchAlgorithmException | SignatureException e) {
                throw new UaException(0x80020000L, (Throwable)e);
            }
            catch (InvalidKeyException e) {
                throw new UaException(2148663296L, (Throwable)e);
            }
        }

        @Override
        public boolean isEncryptionEnabled(SecureChannel channel) {
            return channel.isAsymmetricEncryptionEnabled();
        }

        @Override
        public boolean isSigningEnabled(SecureChannel channel) {
            return channel.isAsymmetricEncryptionEnabled();
        }
    }

    private static interface Delegate {
        public void readSecurityHeader(SecureChannel var1, ByteBuf var2) throws UaException;

        public Cipher getCipher(SecureChannel var1) throws UaException;

        public int getCipherTextBlockSize(SecureChannel var1);

        public int getSignatureSize(SecureChannel var1);

        public void verifyChunk(SecureChannel var1, ByteBuf var2) throws UaException;

        public boolean isEncryptionEnabled(SecureChannel var1);

        public boolean isSigningEnabled(SecureChannel var1);
    }
}

