/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.milo.opcua.stack.client;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import io.netty.bootstrap.Bootstrap;
import io.netty.buffer.PooledByteBufAllocator;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelInitializer;
import io.netty.channel.ChannelOption;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.socket.SocketChannel;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.util.HashedWheelTimer;
import io.netty.util.Timeout;
import java.net.URI;
import java.nio.channels.ClosedChannelException;
import java.security.KeyPair;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import org.eclipse.milo.opcua.stack.client.ClientChannelManager;
import org.eclipse.milo.opcua.stack.client.config.UaTcpStackClientConfig;
import org.eclipse.milo.opcua.stack.client.handlers.UaRequestFuture;
import org.eclipse.milo.opcua.stack.client.handlers.UaTcpClientAcknowledgeHandler;
import org.eclipse.milo.opcua.stack.core.UaException;
import org.eclipse.milo.opcua.stack.core.UaServiceFaultException;
import org.eclipse.milo.opcua.stack.core.application.UaStackClient;
import org.eclipse.milo.opcua.stack.core.channel.ChannelConfig;
import org.eclipse.milo.opcua.stack.core.channel.ClientSecureChannel;
import org.eclipse.milo.opcua.stack.core.serialization.UaRequestMessage;
import org.eclipse.milo.opcua.stack.core.serialization.UaResponseMessage;
import org.eclipse.milo.opcua.stack.core.types.builtin.DateTime;
import org.eclipse.milo.opcua.stack.core.types.builtin.unsigned.UInteger;
import org.eclipse.milo.opcua.stack.core.types.builtin.unsigned.Unsigned;
import org.eclipse.milo.opcua.stack.core.types.enumerated.ApplicationType;
import org.eclipse.milo.opcua.stack.core.types.structured.ApplicationDescription;
import org.eclipse.milo.opcua.stack.core.types.structured.EndpointDescription;
import org.eclipse.milo.opcua.stack.core.types.structured.FindServersRequest;
import org.eclipse.milo.opcua.stack.core.types.structured.FindServersResponse;
import org.eclipse.milo.opcua.stack.core.types.structured.GetEndpointsRequest;
import org.eclipse.milo.opcua.stack.core.types.structured.GetEndpointsResponse;
import org.eclipse.milo.opcua.stack.core.types.structured.RequestHeader;
import org.eclipse.milo.opcua.stack.core.types.structured.ResponseHeader;
import org.eclipse.milo.opcua.stack.core.types.structured.ServiceFault;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UaTcpStackClient
implements UaStackClient {
    private static final long DEFAULT_TIMEOUT_MS = 60000L;
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final Map<UInteger, CompletableFuture<UaResponseMessage>> pending = Maps.newConcurrentMap();
    private final Map<UInteger, Timeout> timeouts = Maps.newConcurrentMap();
    private final HashedWheelTimer wheelTimer;
    private final ApplicationDescription application;
    private final ClientChannelManager channelManager;
    private final UaTcpStackClientConfig config;

    public UaTcpStackClient(UaTcpStackClientConfig config) {
        this.config = config;
        this.wheelTimer = config.getWheelTimer();
        this.application = new ApplicationDescription(config.getApplicationUri(), config.getProductUri(), config.getApplicationName(), ApplicationType.Client, null, null, null);
        this.channelManager = new ClientChannelManager(this);
    }

    public UaTcpStackClientConfig getConfig() {
        return this.config;
    }

    @Override
    public CompletableFuture<UaStackClient> connect() {
        CompletableFuture<UaStackClient> future = new CompletableFuture<UaStackClient>();
        this.channelManager.getChannel().whenComplete((ch, ex) -> {
            if (ch != null) {
                future.complete(this);
            } else {
                future.completeExceptionally((Throwable)ex);
            }
        });
        return future;
    }

    @Override
    public CompletableFuture<UaStackClient> disconnect() {
        return this.channelManager.disconnect().thenApply(v -> this);
    }

    @Override
    public <T extends UaResponseMessage> CompletableFuture<T> sendRequest(UaRequestMessage request) {
        return this.channelManager.getChannel().thenCompose(sc -> this.sendRequest(request, (ClientSecureChannel)sc));
    }

    private <T extends UaResponseMessage> CompletionStage<T> sendRequest(UaRequestMessage request, ClientSecureChannel sc) {
        Channel channel = sc.getChannel();
        CompletableFuture future = new CompletableFuture();
        UaRequestFuture requestFuture = new UaRequestFuture(request);
        RequestHeader requestHeader = request.getRequestHeader();
        this.pending.put(requestHeader.getRequestHandle(), future);
        this.scheduleRequestTimeout(requestHeader);
        requestFuture.getFuture().whenComplete((r, x) -> {
            if (r != null) {
                this.receiveResponse((UaResponseMessage)r);
            } else {
                UInteger requestHandle = request.getRequestHeader().getRequestHandle();
                this.pending.remove(requestHandle);
                future.completeExceptionally((Throwable)x);
            }
        });
        channel.writeAndFlush((Object)requestFuture).addListener(f -> {
            if (!f.isSuccess()) {
                Throwable cause = f.cause();
                if (cause instanceof ClosedChannelException) {
                    this.logger.debug("Channel closed; retrying...");
                    this.sendRequest(request).whenComplete((r, ex) -> {
                        if (r != null) {
                            UaResponseMessage t = r;
                            future.complete(t);
                        } else {
                            future.completeExceptionally((Throwable)ex);
                        }
                    });
                } else {
                    UInteger requestHandle = request.getRequestHeader().getRequestHandle();
                    this.pending.remove(requestHandle);
                    future.completeExceptionally(f.cause());
                    this.logger.debug("Write failed, requestHandle={}", (Object)requestHandle, (Object)cause);
                }
            } else if (this.logger.isTraceEnabled()) {
                this.logger.trace("writeAndFlush succeeded for request={}, requestHandle={}", (Object)request.getClass().getSimpleName(), (Object)requestHeader.getRequestHandle());
            }
        });
        return future;
    }

    @Override
    public void sendRequests(List<? extends UaRequestMessage> requests, List<CompletableFuture<? extends UaResponseMessage>> futures) {
        Preconditions.checkArgument((requests.size() == futures.size() ? 1 : 0) != 0, (Object)"requests and futures parameters must be same size");
        this.channelManager.getChannel().whenComplete((sc, ex) -> {
            if (sc != null) {
                this.sendRequests(requests, futures, (ClientSecureChannel)sc);
            } else {
                futures.forEach(f -> f.completeExceptionally((Throwable)ex));
            }
        });
    }

    private void sendRequests(List<? extends UaRequestMessage> requests, List<CompletableFuture<? extends UaResponseMessage>> futures, ClientSecureChannel sc) {
        Channel channel = sc.getChannel();
        Iterator<? extends UaRequestMessage> requestIterator = requests.iterator();
        Iterator<CompletableFuture<? extends UaResponseMessage>> futureIterator = futures.iterator();
        ArrayList<UaRequestFuture> pendingRequests = new ArrayList<UaRequestFuture>(requests.size());
        while (requestIterator.hasNext() && futureIterator.hasNext()) {
            UaRequestMessage request = requestIterator.next();
            CompletableFuture<UaResponseMessage> future = futureIterator.next();
            UaRequestFuture pendingRequest = new UaRequestFuture(request, future);
            pendingRequests.add(pendingRequest);
            RequestHeader requestHeader = request.getRequestHeader();
            this.pending.put(requestHeader.getRequestHandle(), future);
            this.scheduleRequestTimeout(requestHeader);
            pendingRequest.getFuture().thenAccept(this::receiveResponse);
        }
        channel.eventLoop().execute(() -> {
            for (UaRequestFuture pendingRequest : pendingRequests) {
                channel.write((Object)pendingRequest).addListener(f -> {
                    if (!f.isSuccess()) {
                        UInteger requestHandle = pendingRequest.getRequest().getRequestHeader().getRequestHandle();
                        CompletableFuture<UaResponseMessage> future = this.pending.remove(requestHandle);
                        if (future != null) {
                            future.completeExceptionally(f.cause());
                        }
                        this.logger.debug("Write failed, requestHandle={}", (Object)requestHandle, (Object)f.cause());
                    }
                });
            }
            channel.flush();
        });
    }

    public CompletableFuture<ClientSecureChannel> getChannelFuture() {
        return this.channelManager.getChannel();
    }

    private void scheduleRequestTimeout(RequestHeader requestHeader) {
        UInteger requestHandle = requestHeader.getRequestHandle();
        long timeoutHint = requestHeader.getTimeoutHint() != null ? requestHeader.getTimeoutHint().longValue() : 60000L;
        Timeout timeout = this.wheelTimer.newTimeout(t -> {
            CompletableFuture<UaResponseMessage> f;
            if (this.timeouts.remove(requestHandle) != null && !t.isCancelled() && (f = this.pending.remove(requestHandle)) != null) {
                String message = "request timed out after " + timeoutHint + "ms";
                f.completeExceptionally(new UaException(0x800A0000L, message));
            }
        }, timeoutHint, TimeUnit.MILLISECONDS);
        this.timeouts.put(requestHandle, timeout);
    }

    private void receiveResponse(UaResponseMessage response) {
        ResponseHeader header = response.getResponseHeader();
        UInteger requestHandle = header.getRequestHandle();
        CompletableFuture<UaResponseMessage> future = this.pending.remove(requestHandle);
        if (future != null) {
            if (header.getServiceResult().isGood()) {
                future.complete(response);
            } else {
                ServiceFault serviceFault = response instanceof ServiceFault ? (ServiceFault)response : new ServiceFault(header);
                future.completeExceptionally(new UaServiceFaultException(serviceFault));
            }
            Timeout timeout = this.timeouts.remove(requestHandle);
            if (timeout != null) {
                timeout.cancel();
            }
        } else {
            this.logger.warn("Received {} for unknown requestHandle: {}", (Object)response.getClass().getSimpleName(), (Object)requestHandle);
        }
    }

    @Override
    public Optional<X509Certificate> getCertificate() {
        return this.config.getCertificate();
    }

    @Override
    public Optional<KeyPair> getKeyPair() {
        return this.config.getKeyPair();
    }

    @Override
    public ChannelConfig getChannelConfig() {
        return this.config.getChannelConfig();
    }

    @Override
    public UInteger getChannelLifetime() {
        return this.config.getChannelLifetime();
    }

    @Override
    public ApplicationDescription getApplication() {
        return this.application;
    }

    @Override
    public Optional<EndpointDescription> getEndpoint() {
        return this.config.getEndpoint();
    }

    @Override
    public String getEndpointUrl() {
        return this.config.getEndpoint().map(EndpointDescription::getEndpointUrl).orElse(this.config.getEndpointUrl().orElse(""));
    }

    @Override
    public ExecutorService getExecutorService() {
        return this.config.getExecutor();
    }

    public static CompletableFuture<ClientSecureChannel> bootstrap(final UaTcpStackClient client, final Optional<ClientSecureChannel> existingChannel) {
        final CompletableFuture<ClientSecureChannel> handshake = new CompletableFuture<ClientSecureChannel>();
        Bootstrap bootstrap = new Bootstrap();
        ((Bootstrap)((Bootstrap)((Bootstrap)((Bootstrap)((Bootstrap)bootstrap.group((EventLoopGroup)client.getConfig().getEventLoop())).channel(NioSocketChannel.class)).option(ChannelOption.ALLOCATOR, (Object)PooledByteBufAllocator.DEFAULT)).option(ChannelOption.CONNECT_TIMEOUT_MILLIS, (Object)5000)).option(ChannelOption.TCP_NODELAY, (Object)true)).handler((ChannelHandler)new ChannelInitializer<SocketChannel>(){

            protected void initChannel(SocketChannel channel) throws Exception {
                UaTcpClientAcknowledgeHandler acknowledgeHandler = new UaTcpClientAcknowledgeHandler(client, existingChannel, handshake);
                channel.pipeline().addLast(new ChannelHandler[]{acknowledgeHandler});
            }
        });
        try {
            URI uri = new URI(client.getEndpointUrl()).parseServerAuthority();
            bootstrap.connect(uri.getHost(), uri.getPort()).addListener(f -> {
                if (!f.isSuccess()) {
                    handshake.completeExceptionally(f.cause());
                }
            });
        }
        catch (Throwable e) {
            UaException failure = new UaException(0x80830000L, e);
            handshake.completeExceptionally(failure);
        }
        return handshake;
    }

    public static CompletableFuture<ApplicationDescription[]> findServers(String endpointUrl) {
        UaTcpStackClientConfig config = UaTcpStackClientConfig.builder().setEndpointUrl(endpointUrl).build();
        UaTcpStackClient client = new UaTcpStackClient(config);
        FindServersRequest request = new FindServersRequest(new RequestHeader(null, DateTime.now(), Unsigned.uint(1), Unsigned.uint(0), null, Unsigned.uint(5000), null), endpointUrl, null, null);
        return ((CompletableFuture)client.sendRequest(request).whenComplete((r, ex) -> client.disconnect())).thenApply(FindServersResponse::getServers);
    }

    public static CompletableFuture<EndpointDescription[]> getEndpoints(String endpointUrl) {
        UaTcpStackClientConfig config = UaTcpStackClientConfig.builder().setEndpointUrl(endpointUrl).build();
        UaTcpStackClient client = new UaTcpStackClient(config);
        GetEndpointsRequest request = new GetEndpointsRequest(new RequestHeader(null, DateTime.now(), Unsigned.uint(1), Unsigned.uint(0), null, Unsigned.uint(5000), null), endpointUrl, null, new String[]{"http://opcfoundation.org/UA-Profile/Transport/uatcp-uasc-uabinary"});
        return ((CompletableFuture)client.sendRequest(request).whenComplete((r, ex) -> client.disconnect())).thenApply(GetEndpointsResponse::getEndpoints);
    }
}

