/*
 * Copyright © 2025 CUI-OpenSource-Software (info@cuioss.de)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.cuioss.test.generator.junit.parameterized;

import de.cuioss.test.generator.TypedGenerator;
import org.junit.jupiter.params.provider.ArgumentsSource;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * {@code @TypeGeneratorMethodSource} is an {@link ArgumentsSource} that provides
 * access to values from a {@link TypedGenerator} returned by a factory method
 * for parameterized tests.
 * 
 * <p>
 * This annotation allows you to use a method that returns a configured
 * {@link TypedGenerator} instance for your parameterized tests. The method will
 * be invoked to obtain the generator, and the specified number of values will be
 * generated and passed to your test method.
 * </p>
 * 
 * <p>
 * The method referenced by this annotation must:
 * <ul>
 * <li>Be static if referenced from a different class</li>
 * <li>Return a {@link TypedGenerator} instance</li>
 * <li>Take no parameters</li>
 * </ul>
 * 
 * <h2>Example Usage</h2>
 * 
 * <pre>
 * {@code
 * @ParameterizedTest
 * @TypeGeneratorMethodSource("createStringGenerator")
 * void testWithCustomGenerator(String value) {
 *     assertNotNull(value);
 * }
 * 
 * static TypedGenerator<String> createStringGenerator() {
 *     return Generators.strings(5, 10);
 * }
 * 
 * @ParameterizedTest
 * @TypeGeneratorMethodSource(value = "createIntegerGenerator", count = 5)
 * void testWithMultipleIntegers(Integer value) {
 *     assertNotNull(value);
 * }
 * 
 * static TypedGenerator<Integer> createIntegerGenerator() {
 *     return Generators.integers(1, 100);
 * }
 * }
 * </pre>
 * 
 * <p>
 * You can also reference a method in another class:
 * </p>
 * 
 * <pre>
 * {@code
 * @ParameterizedTest
 * @TypeGeneratorMethodSource("de.cuioss.test.MyGeneratorFactory#createGenerator")
 * void testWithExternalGenerator(MyType value) {
 *     // test with value
 * }
 * }
 * </pre>
 * 
 * @author Oliver Wolff
 * @since 2.0
 * @see TypedGenerator
 * @see TypeGeneratorMethodArgumentsProvider
 */
@Target({ElementType.METHOD})
@Retention(RetentionPolicy.RUNTIME)
@Documented
@ArgumentsSource(TypeGeneratorMethodArgumentsProvider.class)
public @interface TypeGeneratorMethodSource {

    /**
     * The name of the method to invoke to get the TypedGenerator.
     * <p>
     * This can be:
     * <ul>
     * <li>A method name in the test class (e.g., "createGenerator")</li>
     * <li>A fully qualified method name in an external class (e.g.,
     * "com.example.MyGeneratorFactory#createGenerator")</li>
     * </ul>
     * 
     * @return the method name or reference
     */
    String value();

    /**
     * The number of values to generate from the TypedGenerator.
     * <p>
     * If set to a value less than 1, a default of 1 will be used.
     * </p>
     * 
     * @return the number of values to generate
     */
    int count() default 1;

}
