/*
 * Licensed to the author under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.cuioss.test.generator.internal.net.java.quickcheck.generator;

import de.cuioss.test.generator.internal.net.java.quickcheck.ExtendibleGenerator;
import de.cuioss.test.generator.internal.net.java.quickcheck.FrequencyGenerator;
import de.cuioss.test.generator.internal.net.java.quickcheck.Generator;
import de.cuioss.test.generator.internal.net.java.quickcheck.GeneratorException;
import de.cuioss.test.generator.internal.net.java.quickcheck.StatefulGenerator;
import de.cuioss.test.generator.internal.net.java.quickcheck.collection.Pair;
import de.cuioss.test.generator.internal.net.java.quickcheck.collection.Triple;
import de.cuioss.test.generator.internal.net.java.quickcheck.generator.distribution.Distribution;
import de.cuioss.test.generator.internal.net.java.quickcheck.generator.support.ByteArrayGenerator;

import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * {@link CombinedGenerators} can be used to create custom {@link Generator}s.
 */
public class CombinedGenerators {

    public static final int DEFAULT_COLLECTION_MAX_SIZE = Generators.DEFAULT_COLLECTION_MAX_SIZE;
    // TODO this could be a bit high
    // for runs = 200 this means 20000 tries for the worst case
    public static final int DEFAULT_MAX_TRIES = Generators.DEFAULT_MAX_TRIES;

    /**
     * <p>
     * Create a frequency generator. The frequency of {@link Generator} usage
     * depends on the generator weight.
     * </p>
     *
     * @param generator pairs of generators and their weights used to created the
     *                  values
     * @param <T>       type of values generated by the generators.
     */
    public static <T> FrequencyGenerator<T> frequency(Generator<T> generator, int weight) {
        return Generators.frequency(generator, weight);
    }

    /**
     * OneOf is a convenience method for
     * {@link CombinedGenerators#frequency(Generator, int)} when all generator share
     * the same weight.
     */
    public static <T> ExtendibleGenerator<T, T> oneOf(Generator<T> generator) {
        return Generators.oneOf(generator);
    }

    /**
     * Create a generator which will create vectors (here lists) of type T.
     *
     * @param <T>  Type of the list values.
     * @param size Number of element in the vector.
     */
    public static <T> Generator<List<T>> vectors(Generator<T> content, int size) {
        return Generators.vectors(content, size);
    }

    /**
     * Create a generator of pairs of type A for the left value and type B for the
     * right value.
     *
     * @param <A>    Type of left value.
     * @param <B>    Type of right value.
     * @param first  Generator for left values.
     * @param second Generator for right values.
     */
    public static <A, B> Generator<Pair<A, B>> pairs(Generator<A> first, Generator<B> second) {
        return Generators.pairs(first, second);
    }

    /**
     * Create a generator of pairs where first value &lt;= second value.
     *
     * @param <T>     Type of the pair values.
     * @param content Generator for content of the pair values.
     */
    public static <T extends Comparable<T>> Generator<Pair<T, T>> sortedPairs(Generator<T> content) {
        return Generators.sortedPairs(content);
    }

    /**
     * Create a generator of triples of the types A, B and C for first, second and
     * third value.
     *
     * @param <A>    Type of first value.
     * @param <B>    Type of second value.
     * @param <C>    Type of third value.
     * @param first  Generator for first values.
     * @param second Generator for second values.
     * @param third  Generator for third values.
     */
    public static <A, B, C> Generator<Triple<A, B, C>> triples(Generator<A> first, Generator<B> second,
            Generator<C> third) {
        return Generators.triples(first, second, third);
    }

    /**
     * Create a generator return the given values.
     *
     * @param <T> Type of the values generated.
     */
    @SafeVarargs
    public static <T> Generator<T> nullsAnd(T... values) {
        return Generators.nullsAnd(values);
    }

    /**
     * Create a generator of triples where first value &lt;= second value &lt;=
     * third value.
     *
     * @param <T>     Type of the triple values.
     * @param content Generator for content of the triple values.
     */
    public static <T extends Comparable<T>> Generator<Triple<T, T, T>> sortedTriple(Generator<T> content) {
        return Generators.sortedTriple(content);
    }

    /**
     * Create a generator as a combination of a null value generator and generator
     * of type T.
     *
     * @param <T> Type of the values generated.
     */
    public static <T> Generator<T> nullsAnd(Generator<T> generator) {
        return Generators.nullsAnd(generator);
    }

    /**
     * Create a generator as a combination of a null value generator and generator
     * of type T.
     *
     * @param <T>    Type of the values generated.
     * @param weight weight of the provided generator
     */
    public static <T> Generator<T> nullsAnd(Generator<T> generator, int weight) {
        return Generators.nullsAnd(generator, weight);
    }

    /**
     * Create a generator of sets with values from the content generator.
     *
     * @param <T>     type of set elements generated
     * @param content generator providing the content of sets generated
     */
    public static <T> Generator<Set<T>> sets(Generator<? extends T> content) {
        return Generators.sets(content);
    }

    /**
     * Create a generator of sets with values from the content generator.
     *
     * @param <T>     type of set elements generated
     * @param content generator providing the content of sets generated
     * @param size    size of the sets generated
     */
    public static <T> Generator<Set<T>> sets(Generator<? extends T> content, Generator<Integer> size) {
        return Generators.sets(content, size);
    }

    /**
     * Create a generator of sets with values from the content generator. Length is
     * between high and low.
     *
     * @param <T>     type of set elements generated
     * @param content generator providing the content of sets generated
     * @param low     minimal size
     * @param high    max size
     */
    public static <T> Generator<Set<T>> sets(Generator<? extends T> content, int low, int high) {
        return Generators.sets(content, low, high);
    }

    /**
     * Create a generator of sets that are not empty.
     *
     * @param <T>     type of set elements generated
     * @param content generator providing the content of sets generated
     */
    public static <T> Generator<Set<T>> nonEmptySets(Generator<? extends T> content) {
        return Generators.nonEmptySets(content);
    }

    @SafeVarargs
    public static <T> Generator<Set<T>> sets(T... superset) {
        return Generators.sets(superset);
    }

    /**
     * Create a generator of subsets from a given set.
     *
     * @param <T>      type of set elements generated
     * @param superset of the generated set
     */
    public static <T> Generator<Set<T>> sets(Iterable<T> superset) {
        return Generators.sets(superset);
    }

    /**
     * Create a generator of subsets from a given set.
     *
     * @param <T>      type of set elements generated
     * @param superset of the generated set
     * @param size     of the generated set
     */
    public static <T> Generator<Set<T>> sets(Iterable<T> superset, Generator<Integer> size) {
        return Generators.sets(superset, size);
    }

    /**
     * Create a generator that produces lists of duplicates.
     *
     * @return a list derived from the input values. At least one input value is
     *         more than once in the resulting list.
     */
    @SafeVarargs
    public static <T> Generator<List<T>> duplicates(T... input) {
        return Generators.duplicates(input);
    }

    /**
     * Create a generator that produces lists of duplicates.
     *
     * @return a list derived from the input values. At least one input value is
     *         more than once in the resulting list.
     */
    public static <T> Generator<List<T>> duplicates(Iterable<T> input) {
        return Generators.duplicates(input);
    }

    /**
     * Create a generator of iterators.
     *
     * <p>
     * Values of the elements will be taken from the content generator.
     * </p>
     *
     * @param <T>     type of iterator elements generated
     * @param content generator providing the content of iterators generated
     */
    public static <T> Generator<Iterator<T>> iterators(Generator<? extends T> content) {
        return Generators.iterators(content);
    }

    /**
     * Create a generator of iterators.
     *
     * <p>
     * Values of the elements will be taken from the content generator. The
     * generated iterator will have at least one element.
     * </p>
     *
     * @param <T>     type of iterator elements generated
     * @param content generator providing the content of iterators generated
     */
    public static <T> Generator<Iterator<T>> nonEmptyIterators(Generator<T> content) {
        return Generators.nonEmptyIterators(content);
    }

    /**
     * Create a generator of iterators.
     *
     * <p>
     * Values of the elements will be taken from the content generator. The length
     * of the iterators will be determined with the size generator.
     * </p>
     *
     * @param <T>     type of iterator elements generated
     * @param content generator providing the content of iterators generated
     * @param size    used to determine the number of elements of the iterator
     */
    public static <T> Generator<Iterator<T>> iterators(Generator<? extends T> content, Generator<Integer> size) {
        return Generators.iterators(content, size);
    }

    /**
     * Create a generator of lists with values from the content generator. Length
     * values of lists generated will be created with {@link Distribution#UNIFORM}.
     *
     * @param <T>     type of list elements generated
     * @param content generator providing the content of lists generated
     */
    public static <T> Generator<List<T>> lists(Generator<? extends T> content) {
        return Generators.lists(content);
    }

    /**
     * Create a generator of non-empty lists with values from the content generator.
     * Length values of lists generated will be created with
     * {@link Distribution#UNIFORM}.
     *
     * @param <T>     type of list elements generated
     * @param content generator providing the content of lists generated
     */
    public static <T> Generator<List<T>> nonEmptyLists(Generator<? extends T> content) {
        return Generators.nonEmptyLists(content);
    }

    /**
     * Create a generator of lists with values from the content generator. Length
     * values of lists generated will be created with size generator.
     *
     * @param <T>     type of list elements generated
     * @param content generator providing the content of lists generated
     * @param size    integer used to determine the list size
     */
    public static <T> Generator<List<T>> lists(Generator<? extends T> content, Generator<Integer> size) {
        return Generators.lists(content, size);
    }

    /**
     * Create a generator of lists with values from the content generator. Length is
     * between high and low.
     *
     * @param <T>     type of list elements generated
     * @param content generator providing the content of lists generated
     * @param low     minimal size
     * @param high    max size
     */
    public static <T> Generator<List<T>> lists(Generator<? extends T> content, int low, int high) {
        return Generators.lists(content, low, high);
    }

    /**
     * Create a generator of lists with values from the content generator. Length is
     * at least low.
     *
     * @param <T>     type of list elements generated
     * @param content generator providing the content of lists generated
     * @param low     minimal size. If low is larger than
     *                {@link CombinedGenerators#DEFAULT_COLLECTION_MAX_SIZE} then it
     *                is the upper size bound as well.
     */
    public static <T> Generator<List<T>> lists(Generator<? extends T> content, int low) {
        return Generators.lists(content, low);
    }

    /**
     * Create a generator of sorted lists with values from the content generator.
     *
     * @param <T>     type of list elements generated
     * @param content generator providing the content of lists generated
     */
    public static <T extends Comparable<T>> Generator<List<T>> sortedLists(Generator<T> content) {
        return Generators.sortedLists(content);

    }

    /**
     * Create a generator of sorted lists with values from the content generator.
     * Length is between high and low.
     *
     * @param <T>     type of list elements generated
     * @param content generator providing the content of lists generated
     * @param low     minimal size
     * @param high    max size
     */
    public static <T extends Comparable<T>> Generator<List<T>> sortedLists(Generator<T> content, int low, int high) {
        return Generators.sortedLists(content, low, high);

    }

    /**
     * Create a generator of sorted lists with values from the content generator.
     * Length is between high and low.
     *
     * @param <T>     type of list elements generated
     * @param content generator providing the content of lists generated
     * @param size    integer used to determine the list size
     */
    public static <T extends Comparable<T>> Generator<List<T>> sortedLists(Generator<T> content,
            Generator<Integer> size) {
        return Generators.sortedLists(content, size);
    }

    /**
     * Create a generator of arrays with values from the content generator. Length
     * values of array generated will be created with {@link Distribution#UNIFORM}.
     *
     * @param <T>     type of arrays elements generated
     * @param content generator providing the content of arrays generated
     * @param type    type of arrays generated
     */
    public static <T> Generator<T[]> arrays(Generator<? extends T> content, Class<T> type) {
        return Generators.arrays(content, type);
    }

    /**
     * Create a generator of arrays that are not empty.
     *
     * @param <T>     type of arrays elements generated
     * @param content generator providing the content of arrays generated
     * @param type    type of arrays generated
     */
    public static <T> Generator<T[]> nonEmptyArrays(Generator<? extends T> content, Class<T> type) {
        return Generators.nonEmptyArrays(content, type);
    }

    /**
     * Create a generator of arrays with values from the content generator. Length
     * values of arrays generated will be created with size generator.
     *
     * @param <T>     type of arrays elements generated
     * @param content generator providing the content of arrays generated
     * @param size    integer used to determine the array size
     * @param type    type of arrays generated
     */
    public static <T> Generator<T[]> arrays(Generator<? extends T> content, Generator<Integer> size, Class<T> type) {
        return Generators.arrays(content, size, type);
    }

    /**
     * Create a generator of byte arrays. The length of arrays generated will be
     * determined by the {@link ByteArrayGenerator#MIN_SIZE} and
     * {@link ByteArrayGenerator#MAX_SIZE} constants.
     *
     */
    public static Generator<byte[]> byteArrays() {
        return Generators.byteArrays();
    }

    /**
     * Create a generator of byte arrays. Length values of arrays generated will be
     * created with size generator.
     *
     * @param size integer used to determine the array size
     */
    public static Generator<byte[]> byteArrays(Generator<Integer> size) {
        return Generators.byteArrays(size);
    }

    /**
     * Create a generator of byte arrays. Length values of arrays generated will be
     * created with size generator.
     *
     * @param size    integer used to determine the array size
     * @param content generator for the byte array content
     */
    public static Generator<byte[]> byteArrays(Generator<Byte> content, Generator<Integer> size) {
        return Generators.byteArrays(content, size);
    }

    /**
     * Create a generator of integer arrays.
     *
     */
    public static Generator<int[]> intArrays() {
        return Generators.intArrays();
    }

    /**
     * Create a generator of integer arrays. Length values of arrays generated will
     * be created with size generator.
     *
     * @param size integer used to determine the array size
     */
    public static Generator<int[]> intArrays(Generator<Integer> size) {
        return Generators.intArrays(size);
    }

    /**
     * Create a generator of integer arrays. Length values of arrays generated will
     * be created with size generator.
     *
     * @param size    integer used to determine the array size
     * @param content generator for the integer array content
     */
    public static Generator<int[]> intArrays(Generator<Integer> content, Generator<Integer> size) {
        return Generators.intArrays(content, size);
    }

    /**
     * Create a generator of {@link Map maps}.
     *
     * <p>
     * This is a generator for simple maps where the values are not related to the
     * keys.
     * </p>
     *
     * @param keys   {@link Generator} for the keys of the map
     * @param values {@link Generator} for the values of the map
     */
    public static <K, V> Generator<Map<K, V>> maps(Generator<K> keys, Generator<V> values) {
        return Generators.maps(keys, values);
    }

    /**
     * Create a generator of {@link Map maps}.
     *
     * <p>
     * This is a generator for simple maps where the values are not related to the
     * keys.
     * </p>
     *
     * @param keys   {@link Generator} for the keys of the map
     * @param values {@link Generator} for the values of the map
     * @param size   integer used to determine the size of the generated map
     */
    public static <K, V> Generator<Map<K, V>> maps(Generator<K> keys, Generator<V> values, Generator<Integer> size) {
        return Generators.maps(keys, values, size);
    }

    /**
     * Create a generator of maps from a given map.
     *
     * <p>
     * The entry set of the generated maps are subsets of the given map's entry set.
     * </p>
     * 
     * @param supermap of the generated maps
     */
    public static <K, V> Generator<Map<K, V>> maps(Map<K, V> supermap) {
        return Generators.maps(supermap);
    }

    /**
     * Create a generator of maps from a given map.
     *
     * <p>
     * The entry set of the generated maps are subsets of the given map's entry set.
     * </p>
     *
     * @param supermap of the generated maps
     * @param sizes    of the generated maps
     */
    public static <K, V> Generator<Map<K, V>> maps(Map<K, V> supermap, Generator<Integer> sizes) {
        return Generators.maps(supermap, sizes);
    }

    /**
     * Create a deterministic generator which guarantees that all values from the
     * ensuredValues collection will be return {@link Generator#next()} are issued
     * (i.e. ensuredValues.size() &lt;= # of runs). The order of values is
     * undefined.
     *
     * @param <T> type of values return
     */
    public static <T> StatefulGenerator<T> ensureValues(Iterable<T> ensuredValues) {
        return Generators.ensureValues(ensuredValues);
    }

    /**
     * Create a deterministic generator which guarantees that all values from the
     * ensuredValues array will be return {@link Generator#next()} are issued (i.e.
     * ensuredValues.size() &lt;= # of runs). The order of values is undefined.
     *
     * @param <T> type of values return
     */
    @SafeVarargs
    public static <T> StatefulGenerator<T> ensureValues(T... content) {
        return Generators.ensureValues(content);
    }

    /**
     * <p>
     * Create a deterministic generator which guarantees that all values from the
     * ensuredValues collection will be return {@link Generator#next()} are issued
     * (i.e. ensuredValues.size() &lt;= # of runs). The order of values is
     * undefined.
     * </p>
     * <p>
     * If all values of ensuredValues are generated calls to
     * {@link Generator#next()} will return generator.
     * </p>
     *
     * @param <T> type of values return
     */
    public static <T> StatefulGenerator<T> ensureValues(Iterable<T> ensuredValues, Generator<T> otherValues) {
        return Generators.ensureValues(ensuredValues, otherValues);
    }

    /**
     * <p>
     * Create a generator which guarantees that all values from the ensuredValues
     * will be return {@link Generator#next()} are issued.
     * </p>
     * <p>
     * The order of values is undefined. All other values in the window and after
     * the window are taken from the {@link Generator generator otherValues}.
     * </p>
     *
     * @param <T>    type of values return
     * @param window After window number of calls to {@link Generator#next()} it is
     *               guaranteed that all ensured values were return
     */
    public static <T> StatefulGenerator<T> ensureValues(Iterable<T> ensuredValues, int window,
            Generator<T> otherValues) {
        return Generators.ensureValues(ensuredValues, window, otherValues);
    }

    /**
     * <p>
     * Create a generator that ensures unique values.
     * </p>
     * <p>
     * The actual values are created with an arbitrary generator.
     * </p>
     * <p>
     * Note: unique generator depends on valid implementation of equals and hashCode
     * method of the content type generated.
     * </p>
     *
     * @param <T>       type of values return
     * @param generator used to create the raw values. This generator can create
     *                  duplicate values
     * @param tries     Number of tries to create a new unique value. After this
     *                  number of tries is exceeded the generation aborts with a
     *                  {@link GeneratorException}.
     * @return unique generator instance
     */
    public static <T> StatefulGenerator<T> uniqueValues(Generator<T> generator, int tries) {
        return Generators.uniqueValues(generator, tries);
    }

    /**
     * <p>
     * Create a generator that ensures unique values.
     * </p>
     * <p>
     * The actual values are created with an arbitrary generator.
     * </p>
     * <p>
     * Unique generator depends on the {@link Comparator} implementation to decide
     * if two instances are the same (i.e. when the comparator return for
     * {@link Comparator#compare(Object, Object)}).
     * </p>
     *
     * @param <T>        type of values return
     * @param generator  used to create the raw values. This generator can create
     *                   duplicate values
     * @param comparator that decides if two values are of the same equivalence
     *                   class.
     * @param tries      Number of tries to create a new unique value. After this
     *                   number of tries is exceeded the generation aborts with a
     *                   {@link GeneratorException}.
     * @return unique generator instance
     */
    public static <T> StatefulGenerator<T> uniqueValues(Generator<T> generator, Comparator<? super T> comparator,
            int tries) {
        return Generators.uniqueValues(generator, comparator, tries);
    }

    /**
     * <p>
     * Create a generator that ensures unique values.
     * </p>
     * <p>
     * The actual values are created with an arbitrary generator.
     * </p>
     * <p>
     * Unique generator depends on the {@link Comparator} implementation to decide
     * if two instances are the same (i.e. when the comparator return for
     * {@link Comparator#compare(Object, Object)}).
     * </p>
     *
     * @param <T>        type of values return
     * @param generator  used to create the raw values. This generator can create
     *                   duplicate values
     * @param comparator that decides if two values are of the same equivalence
     *                   class.
     * @return unique generator instance
     */
    public static <T> StatefulGenerator<T> uniqueValues(Generator<T> generator, Comparator<? super T> comparator) {
        return Generators.uniqueValues(generator, comparator);
    }

    /**
     * <p>
     * Create a generator that ensures unique values
     * </p>
     * <p>
     * The actual values are created with an arbitrary generator.
     * </p>
     * <p>
     * Note: unique generator depends on valid implementation of equals and hashCode
     * method of the content type generated.
     * </p>
     *
     * @param <T>       type of values return
     * @param generator used to create the raw values. This generator can create
     *                  duplicate values
     * @return unique generator instance
     */
    public static <T> StatefulGenerator<T> uniqueValues(Generator<T> generator) {
        return Generators.uniqueValues(generator);
    }

    /**
     * Create a generator that omits a given set of values.
     *
     * @param generator used to create the raw values.
     * @param excluded  values. These values will not be return
     */
    @SafeVarargs
    public static <T> Generator<T> excludeValues(Generator<T> generator, T... excluded) {
        return Generators.excludeValues(generator, excluded);
    }

    /**
     * Create a generator that omits a given set of values.
     *
     * @param values   of generator
     * @param excluded values. These values will not be return
     */
    @SafeVarargs
    public static <T> Generator<T> excludeValues(Iterable<T> values, T... excluded) {
        return Generators.excludeValues(values, excluded);
    }

    /**
     * Create a generator that omits a given set of values.
     *
     * @param values   of generator
     * @param excluded values. These values will not be return
     */
    public static <T> Generator<T> excludeValues(Iterable<T> values, Iterable<T> excluded) {
        return Generators.excludeValues(values, excluded);
    }

    /**
     * Create a generator that omits a given set of values.
     *
     * @param generator used to create the raw values.
     * @param excluded  values. These values will not be return
     */
    public static <T> Generator<T> excludeValues(Generator<T> generator, Iterable<T> excluded) {
        return Generators.excludeValues(generator, excluded);
    }

    /**
     * A generator for a lists. The values in the lists are strictly increasing.
     * <p>
     * For every element x in the list: x(n) &lt; x(n+1).
     * </p>
     *
     * @param input values generator
     */
    public static <T extends Comparable<T>> Generator<List<T>> strictlyOrdered(Generator<T> input) {
        return Generators.strictlyOrdered(input);
    }

    /**
     * A generator for a lists. The values in the lists are strictly increasing.
     *
     * <p>
     * For every element x in the list: x(n) &lt; x(n+1).
     * </p>
     *
     * @param input values generator
     * @param low   minimum size of the lists
     * @param high  maximum size of the lists
     */
    public static <T extends Comparable<T>> Generator<List<T>> strictlyOrdered(Generator<T> input, int low, int high) {
        return Generators.strictlyOrdered(input, low, high);
    }

    /**
     * A generator for a lists. The values in the lists are strictly increasing.
     * <p>
     * For every element x in the list: x(n) &lt; x(n+1).
     * </p>
     * <p>
     * This {@link Generator} can be used to generate a list of strictly decreasing
     * values:
     * {@code CombinedGenerators.strictlyOrdered(ts, Collections.&lt;T&gt; reverseOrder());}
     * </p>
     *
     *
     * @param input      values generator
     * @param comparator that orders the values
     */
    public static <T> Generator<List<T>> strictlyOrdered(Generator<T> input, Comparator<T> comparator) {
        return Generators.strictlyOrdered(input, comparator);
    }

    /**
     * A generator for a lists. The values in the lists are strictly increasing.
     * <p>
     * For every element x in the list: x(n) &lt; x(n+1).
     * </p>
     *
     * @param input      values generator
     * @param comparator that orders the values
     * @param size       of the resulting lists
     */
    public static <T> Generator<List<T>> strictlyOrdered(Generator<T> input, Comparator<T> comparator,
            Generator<Integer> size) {
        return Generators.strictlyOrdered(input, comparator, size);
    }
}
