/*
 * Decompiled with CFR 0.152.
 */
package de.cuioss.http.client.retry;

import de.cuioss.http.client.HttpLogMessages;
import de.cuioss.http.client.result.HttpResult;
import de.cuioss.http.client.retry.HttpOperation;
import de.cuioss.http.client.retry.RetryContext;
import de.cuioss.http.client.retry.RetryMetrics;
import de.cuioss.http.client.retry.RetryStrategy;
import de.cuioss.tools.logging.CuiLogger;
import java.time.Duration;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

public class ExponentialBackoffRetryStrategy
implements RetryStrategy {
    private static final CuiLogger LOGGER = new CuiLogger(ExponentialBackoffRetryStrategy.class);
    private final int maxAttempts;
    private final Duration initialDelay;
    private final double backoffMultiplier;
    private final Duration maxDelay;
    private final double jitterFactor;
    private final RetryMetrics retryMetrics;

    ExponentialBackoffRetryStrategy(int maxAttempts, Duration initialDelay, double backoffMultiplier, Duration maxDelay, double jitterFactor, RetryMetrics retryMetrics) {
        this.maxAttempts = maxAttempts;
        this.initialDelay = Objects.requireNonNull(initialDelay, "initialDelay");
        this.backoffMultiplier = backoffMultiplier;
        this.maxDelay = Objects.requireNonNull(maxDelay, "maxDelay");
        this.jitterFactor = jitterFactor;
        this.retryMetrics = Objects.requireNonNull(retryMetrics, "retryMetrics");
    }

    @Override
    public <T> CompletableFuture<HttpResult<T>> execute(HttpOperation<T> operation, RetryContext context) {
        Objects.requireNonNull(operation, "operation");
        Objects.requireNonNull(context, "context");
        long totalStartTime = System.nanoTime();
        this.retryMetrics.recordRetryStart(context);
        return this.executeAttempt(operation, context, 1, totalStartTime);
    }

    private <T> CompletableFuture<HttpResult<T>> executeAttempt(HttpOperation<T> operation, RetryContext context, int attempt, long totalStartTime) {
        return CompletableFuture.supplyAsync(() -> {
            long attemptStartTime = System.nanoTime();
            LOGGER.debug("Starting retry attempt %s for operation %s", new Object[]{attempt, context.operationName()});
            HttpResult result = operation.execute();
            Duration attemptDuration = Duration.ofNanos(System.nanoTime() - attemptStartTime);
            boolean success = result.isSuccess();
            this.retryMetrics.recordRetryAttempt(context, attempt, attemptDuration, success);
            return new AttemptResult(result, attemptDuration, success);
        }, Executors.newVirtualThreadPerTaskExecutor()).thenCompose(attemptResult -> {
            HttpResult result = attemptResult.result();
            Duration attemptDuration = attemptResult.attemptDuration();
            boolean success = attemptResult.success();
            if (success) {
                Duration totalDuration = Duration.ofNanos(System.nanoTime() - totalStartTime);
                this.retryMetrics.recordRetryComplete(context, totalDuration, true, attempt);
                if (attempt > 1) {
                    LOGGER.info(HttpLogMessages.INFO.RETRY_OPERATION_SUCCEEDED_AFTER_ATTEMPTS, new Object[]{context.operationName(), attempt, this.maxAttempts});
                    return CompletableFuture.completedFuture(result);
                }
                return CompletableFuture.completedFuture(result);
            }
            if (attempt >= this.maxAttempts) {
                LOGGER.warn(HttpLogMessages.WARN.RETRY_MAX_ATTEMPTS_REACHED, new Object[]{context.operationName(), this.maxAttempts, "Final attempt failed"});
                Duration totalDuration = Duration.ofNanos(System.nanoTime() - totalStartTime);
                this.retryMetrics.recordRetryComplete(context, totalDuration, false, this.maxAttempts);
                LOGGER.warn(HttpLogMessages.WARN.RETRY_OPERATION_FAILED, new Object[]{context.operationName(), this.maxAttempts, totalDuration.toMillis()});
                return CompletableFuture.completedFuture(result);
            }
            if (!result.isRetryable()) {
                LOGGER.debug("Non-retryable error for operation %s (duration: %sms)", new Object[]{context.operationName(), attemptDuration.toMillis()});
                Duration totalDuration = Duration.ofNanos(System.nanoTime() - totalStartTime);
                this.retryMetrics.recordRetryComplete(context, totalDuration, false, attempt);
                return CompletableFuture.completedFuture(result);
            }
            LOGGER.debug("Retry attempt %s failed for operation %s (duration: %sms)", new Object[]{attempt, context.operationName(), attemptDuration.toMillis()});
            Duration delay = this.calculateDelay(attempt);
            int nextAttempt = attempt + 1;
            this.retryMetrics.recordRetryDelay(context, nextAttempt, delay, delay);
            Executor delayedExecutor = CompletableFuture.delayedExecutor(delay.toMillis(), TimeUnit.MILLISECONDS, Executors.newVirtualThreadPerTaskExecutor());
            return CompletableFuture.supplyAsync(() -> this.executeAttempt(operation, context, nextAttempt, totalStartTime), delayedExecutor).thenCompose(future -> future);
        });
    }

    private Duration calculateDelay(int attemptNumber) {
        double exponentialDelay = (double)this.initialDelay.toMillis() * Math.pow(this.backoffMultiplier, (double)attemptNumber - 1.0);
        double randomFactor = 2.0 * ThreadLocalRandom.current().nextDouble() - 1.0;
        double jitter = 1.0 + randomFactor * this.jitterFactor;
        long delayMs = Math.round(exponentialDelay * jitter);
        return Duration.ofMillis(Math.min(delayMs, this.maxDelay.toMillis()));
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private int maxAttempts = 5;
        private Duration initialDelay = Duration.ofSeconds(1L);
        private double backoffMultiplier = 2.0;
        private Duration maxDelay = Duration.ofMinutes(1L);
        private double jitterFactor = 0.1;
        private RetryMetrics retryMetrics = RetryMetrics.noOp();

        public Builder maxAttempts(int maxAttempts) {
            if (maxAttempts < 1) {
                throw new IllegalArgumentException("maxAttempts must be positive, got: " + maxAttempts);
            }
            this.maxAttempts = maxAttempts;
            return this;
        }

        public Builder initialDelay(Duration initialDelay) {
            this.initialDelay = Objects.requireNonNull(initialDelay, "initialDelay");
            if (initialDelay.isNegative()) {
                throw new IllegalArgumentException("initialDelay cannot be negative");
            }
            return this;
        }

        public Builder backoffMultiplier(double backoffMultiplier) {
            if (backoffMultiplier < 1.0) {
                throw new IllegalArgumentException("backoffMultiplier must be >= 1.0, got: " + backoffMultiplier);
            }
            this.backoffMultiplier = backoffMultiplier;
            return this;
        }

        public Builder maxDelay(Duration maxDelay) {
            this.maxDelay = Objects.requireNonNull(maxDelay, "maxDelay");
            if (maxDelay.isNegative()) {
                throw new IllegalArgumentException("maxDelay cannot be negative");
            }
            return this;
        }

        public Builder jitterFactor(double jitterFactor) {
            if (jitterFactor < 0.0 || jitterFactor > 1.0) {
                throw new IllegalArgumentException("jitterFactor must be between 0.0 and 1.0, got: " + jitterFactor);
            }
            this.jitterFactor = jitterFactor;
            return this;
        }

        public Builder retryMetrics(RetryMetrics retryMetrics) {
            this.retryMetrics = Objects.requireNonNull(retryMetrics, "retryMetrics");
            return this;
        }

        public ExponentialBackoffRetryStrategy build() {
            return new ExponentialBackoffRetryStrategy(this.maxAttempts, this.initialDelay, this.backoffMultiplier, this.maxDelay, this.jitterFactor, this.retryMetrics);
        }
    }

    private record AttemptResult<T>(HttpResult<T> result, Duration attemptDuration, boolean success) {
    }
}

