/*
 * Decompiled with CFR 0.152.
 */
package de.csdev.ebus.utils;

import de.csdev.ebus.command.EBusCommandRegistry;
import de.csdev.ebus.command.EBusCommandUtils;
import de.csdev.ebus.command.IEBusCommandCollection;
import de.csdev.ebus.command.IEBusCommandMethod;
import de.csdev.ebus.command.datatypes.EBusTypeException;
import de.csdev.ebus.command.datatypes.EBusTypeRegistry;
import de.csdev.ebus.command.datatypes.IEBusType;
import de.csdev.ebus.command.datatypes.std.EBusTypeBCD;
import de.csdev.ebus.command.datatypes.std.EBusTypeData1c;
import de.csdev.ebus.command.datatypes.std.EBusTypeData2b;
import de.csdev.ebus.command.datatypes.std.EBusTypeData2c;
import de.csdev.ebus.command.datatypes.std.EBusTypeInteger;
import de.csdev.ebus.command.datatypes.std.EBusTypeWord;
import de.csdev.ebus.core.EBusDataException;
import de.csdev.ebus.service.device.EBusDevice;
import de.csdev.ebus.service.device.EBusDeviceTable;
import de.csdev.ebus.service.metrics.EBusMetricsService;
import de.csdev.ebus.utils.EBusUtils;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.apache.commons.lang.StringUtils;
import org.eclipse.jdt.annotation.NonNull;
import org.eclipse.jdt.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EBusConsoleUtils {
    private static final Logger logger = LoggerFactory.getLogger(EBusConsoleUtils.class);

    private EBusConsoleUtils() {
        throw new IllegalStateException("Utility class");
    }

    public static String bruteforceData(byte @Nullable [] data) throws EBusTypeException {
        EBusTypeRegistry typeRegistry = new EBusTypeRegistry();
        IEBusType typeD1C = typeRegistry.getType(EBusTypeData1c.TYPE_DATA1C);
        IEBusType typeBCD = typeRegistry.getType(EBusTypeBCD.TYPE_BCD);
        IEBusType typeWord = typeRegistry.getType(EBusTypeWord.TYPE_WORD);
        IEBusType typeInt = typeRegistry.getType(EBusTypeInteger.TYPE_INTEGER);
        IEBusType typeD2B = typeRegistry.getType(EBusTypeData2b.TYPE_DATA2B);
        IEBusType typeD2C = typeRegistry.getType(EBusTypeData2c.TYPE_DATA2C);
        String format = String.format("%-4s%-13s%-13s%-13s%-13s%-13s%-13s%-13s", "Pos", "WORD", "Int", "UInt8", "DATA2B", "DATA2C", "DATA1c", "BCD");
        logger.info("    " + format);
        logger.info("    -----------------------------------------------------------------------------------------------");
        if (data != null) {
            for (int i = 0; i < data.length; ++i) {
                try {
                    String word = i == data.length - 1 ? "---" : typeWord.decode(new byte[]{data[i + 1], data[i]});
                    String integer = i == data.length - 1 ? "---" : typeInt.decode(new byte[]{data[i + 1], data[i]});
                    String data2b = i == data.length - 1 ? "---" : typeD2B.decode(new byte[]{data[i + 1], data[i]});
                    String data2c = i == data.length - 1 ? "---" : typeD2C.decode(new byte[]{data[i + 1], data[i]});
                    Object data1c = typeD1C.decode(new byte[]{data[i]});
                    Object bcd = typeBCD.decode(new byte[]{data[i]});
                    int uint = data[i] & 0xFF;
                    format = String.format("%-4s%-13s%-13s%-13s%-13s%-13s%-13s%-13s", i + 6, word, integer, uint, data2b, data2c, data1c, bcd);
                    logger.info("    " + format);
                    continue;
                }
                catch (EBusTypeException e) {
                    logger.error("error!", (Throwable)e);
                }
            }
        }
        return "";
    }

    public static @NonNull String getMetricsInformation(@NonNull EBusMetricsService service) {
        Objects.requireNonNull(service, "service");
        StringBuilder sb = new StringBuilder();
        sb.append(String.format("%-25s | %-10s\n", "Successful received", service.getReceived()));
        sb.append(String.format("%-25s | %-10s\n", "Failed received", service.getFailed()));
        sb.append(String.format("%-25s | %-10s\n", "Successful/Failed ratio", service.getFailureRatio()));
        sb.append("\n");
        sb.append(String.format("%-25s | %-10s\n", "Resolved telegrams", service.getResolved()));
        sb.append(String.format("%-25s | %-10s\n", "Unresolved telegrams", service.getUnresolved()));
        sb.append(String.format("%-25s | %-10s\n", "Resolved/Unresolved ratio", service.getUnresolvedRatio()));
        return sb.toString();
    }

    public static String getDeviceTableInformation(@NonNull Collection<@NonNull IEBusCommandCollection> collections, @NonNull EBusDeviceTable deviceTable) {
        StringBuilder sb = new StringBuilder();
        HashMap<String, String> mapping = new HashMap<String, String>();
        for (IEBusCommandCollection collection : collections) {
            for (String identification : collection.getIdentification()) {
                mapping.put(identification, collection.getId());
            }
        }
        EBusDevice ownDevice = deviceTable.getOwnDevice();
        sb.append(String.format("%-2s | %-2s | %-14s | %-14s | %-25s | %-2s | %-10s | %-10s | %-20s\n", "MA", "SA", "Identifier", "Device", "Manufacture", "ID", "Firmware", "Hardware", "Last Activity"));
        sb.append(String.format("%-2s-+-%-2s-+-%-14s-+-%-14s-+-%-20s-+-%-2s-+-%-10s-+-%-10s-+-%-20s\n", StringUtils.repeat((String)"-", (int)2), StringUtils.repeat((String)"-", (int)2), StringUtils.repeat((String)"-", (int)14), StringUtils.repeat((String)"-", (int)14), StringUtils.repeat((String)"-", (int)20), StringUtils.repeat((String)"-", (int)2), StringUtils.repeat((String)"-", (int)10), StringUtils.repeat((String)"-", (int)10), StringUtils.repeat((String)"-", (int)20)));
        for (EBusDevice device : deviceTable.getDeviceTable()) {
            boolean isBridge = device.equals(ownDevice);
            String masterAddress = EBusUtils.toHexDumpString(device.getMasterAddress());
            String slaveAddress = EBusUtils.toHexDumpString(device.getSlaveAddress());
            String activity = device.getLastActivity() == 0L ? "---" : new Date(device.getLastActivity()).toString();
            String id = EBusUtils.toHexDumpString(device.getDeviceId()).toString();
            String deviceName = isBridge ? "<interface>" : mapping.getOrDefault(id, "---");
            String manufacture = isBridge ? "eBUS Library" : device.getManufacturerName();
            sb.append(String.format("%-2s | %-2s | %-14s | %-14s | %-25s | %-2s | %-10s | %-10s | %-20s\n", masterAddress, slaveAddress, id, deviceName, manufacture, EBusUtils.toHexDumpString(device.getManufacturer()), device.getSoftwareVersion(), device.getHardwareVersion(), activity));
        }
        sb.append(StringUtils.repeat((String)"-", (int)118) + "\n");
        sb.append("MA = Master Address / SA = Slave Address / ID = Manufacture ID\n");
        return sb.toString();
    }

    public static @NonNull String analyzeTelegram(@NonNull EBusCommandRegistry registry, byte @NonNull [] data) {
        Objects.requireNonNull(registry, "registry");
        Objects.requireNonNull(data, "data");
        StringBuilder sb = new StringBuilder();
        try {
            byte[] edata = null;
            sb.append("\n");
            try {
                edata = EBusCommandUtils.checkRawTelegram(data);
            }
            catch (EBusDataException e) {
                String msg = String.format("** Error on checking telegram: %s **", e.getMessage());
                int len = msg.length();
                sb.append("\n");
                sb.append(StringUtils.repeat((String)"*", (int)len) + "\n");
                sb.append(msg + "\n");
                msg = "**     !!! Warning: All following results are wrong and only displayed for information purpose !!!";
                msg = msg + StringUtils.repeat((String)" ", (int)(len - msg.length() - 2)) + "**";
                sb.append(msg + "\n");
                sb.append(StringUtils.repeat((String)"*", (int)len) + "\n");
                sb.append("\n");
                return sb.toString();
            }
            sb.append("\n");
            sb.append("Check and unescape telegram\n");
            sb.append("***************************\n");
            sb.append("\n");
            sb.append(String.format("Original data : %s\n", EBusUtils.toHexDumpString(data)));
            sb.append(String.format("Unescaped data: %s\n", EBusUtils.toHexDumpString(edata)));
            byte[] command = Arrays.copyOfRange(edata, 2, 4);
            boolean isMasterMaster = EBusUtils.isMasterAddress(edata[1]);
            boolean isBroadcast = edata[1] == -2;
            boolean isMasterSlave = !isMasterMaster && !isBroadcast;
            int masterDataLenPos = 4;
            byte masterDataLen = edata[masterDataLenPos];
            byte[] masterData = Arrays.copyOfRange(edata, 5, 5 + masterDataLen);
            int masterCrcPos = 5 + masterDataLen;
            int slaveACKPos = masterCrcPos + 1;
            String dataString = EBusUtils.toHexDumpString(edata).toString();
            int dataLen = dataString.length();
            sb.append("\n");
            sb.append("Analyse the telegram\n");
            sb.append("********************\n");
            sb.append("\n");
            sb.append(dataString + "\n");
            String FORMAT = "%-20s | %-20s | %s";
            sb.append(EBusConsoleUtils.createTelegramResoverRow(0, 1, dataLen, String.format("%-20s | %-20s | %s", "Source address", "Type: " + EBusConsoleUtils.addressType(edata[0]), EBusConsoleUtils.hex(edata[0]))));
            sb.append(EBusConsoleUtils.createTelegramResoverRow(1, 1, dataLen, String.format("%-20s | %-20s | %s", "Destination address", "Type: " + EBusConsoleUtils.addressType(edata[1]), EBusConsoleUtils.hex(edata[1]))));
            sb.append(EBusConsoleUtils.createTelegramResoverRow(2, 2, dataLen, String.format("%-20s | %-20s | %s", "Command", "", EBusConsoleUtils.hex(command))));
            sb.append(EBusConsoleUtils.createTelegramResoverRow(4, 1, dataLen, String.format("%-20s | %-20s | %s", "Master Data Length", "Length: " + edata[4], EBusConsoleUtils.hex(edata[4]))));
            sb.append(EBusConsoleUtils.createTelegramResoverRow(5, masterDataLen, dataLen, String.format("%-20s | %-20s | %s", "Master Data", "", EBusConsoleUtils.hex(masterData))));
            sb.append(EBusConsoleUtils.createTelegramResoverRow(masterCrcPos, 1, dataLen, String.format("%-20s | %-20s | %s", "Master CRC", "", EBusConsoleUtils.hex(edata[masterCrcPos]))));
            if (isMasterMaster) {
                sb.append(EBusConsoleUtils.createTelegramResoverRow(slaveACKPos, 1, dataLen, EBusConsoleUtils.hex(edata[slaveACKPos]) + " Slave ACK"));
            } else if (!isBroadcast && isMasterSlave) {
                int slaveDataLenPos = slaveACKPos + 1;
                byte slaveDataLen = edata[slaveDataLenPos];
                int slaveDataPos = slaveDataLenPos + 1;
                int slaveCRCPos = slaveDataPos + slaveDataLen;
                int masterACKPos = slaveCRCPos + 1;
                byte[] slaveData = Arrays.copyOfRange(edata, slaveDataPos, slaveDataPos + slaveDataLen);
                sb.append(EBusConsoleUtils.createTelegramResoverRow(slaveACKPos, 1, dataLen, String.format("%-20s | %-20s | %s", "Slave ACK", "", EBusConsoleUtils.hex(edata[slaveACKPos]))));
                sb.append(EBusConsoleUtils.createTelegramResoverRow(slaveDataLenPos, 1, dataLen, String.format("%-20s | %-20s | %s", "Slave Data Length", "Length: " + edata[slaveDataLenPos], EBusConsoleUtils.hex(edata[slaveDataLenPos]))));
                sb.append(EBusConsoleUtils.createTelegramResoverRow(slaveDataPos, slaveDataLen, dataLen, String.format("%-20s | %-20s | %s", "Slave Data", "", EBusConsoleUtils.hex(slaveData))));
                sb.append(EBusConsoleUtils.createTelegramResoverRow(slaveCRCPos, 1, dataLen, String.format("%-20s | %-20s | %s", "Slave CRC", "", EBusConsoleUtils.hex(edata[slaveCRCPos]))));
                sb.append(EBusConsoleUtils.createTelegramResoverRow(masterACKPos, 1, dataLen, String.format("%-20s | %-20s | %s", "Master ACK", "", EBusConsoleUtils.hex(edata[masterACKPos]))));
            }
            List<IEBusCommandMethod> methods = registry.find(edata);
            sb.append("\n");
            sb.append("Resolve the telegram\n");
            sb.append("********************\n");
            sb.append("\n");
            sb.append(String.format("Found %s command method(s) for this telegram.\n", methods.size()));
            sb.append("\n");
            for (IEBusCommandMethod method : methods) {
                try {
                    if (method == null) continue;
                    Map<String, Object> result = EBusCommandUtils.decodeTelegram(method, data);
                    sb.append(String.format("Values from command '%s' with method '%s' from collection '%s'\n", new Object[]{method.getParent().getId(), method.getMethod(), method.getParent().getParentCollection().getId()}));
                    for (Map.Entry<String, Object> entry : result.entrySet()) {
                        Object value = entry.getValue();
                        if (value instanceof byte[]) {
                            value = EBusUtils.toHexDumpString((byte[])value);
                        }
                        sb.append(String.format("  %-20s = %s\n", entry.getKey(), value != null ? value.toString() : "NULL"));
                    }
                }
                catch (EBusTypeException e) {
                    logger.error("error!", (Throwable)e);
                }
            }
            sb.append("\n");
        }
        catch (Exception e) {
            logger.error("error!", (Throwable)e);
        }
        return sb.toString();
    }

    private static @NonNull String addressType(byte b) {
        if (EBusUtils.isMasterAddress(b)) {
            return "Master";
        }
        if (b == -2) {
            return "Broadcast";
        }
        return "Slave";
    }

    private static @NonNull String hex(byte[] b) {
        return EBusUtils.toHexDumpString(b).toString();
    }

    private static @NonNull String hex(byte b) {
        return EBusUtils.toHexDumpString(b);
    }

    private static @NonNull String createTelegramResoverRow(int pos, int length, int textStart, String text) {
        StringBuilder sb = new StringBuilder();
        String repeat = StringUtils.repeat((String)"^^ ", (int)length);
        if (repeat.length() > 0) {
            repeat = repeat.substring(0, repeat.length() - 1);
        }
        sb.append(StringUtils.repeat((String)" ", (int)(pos * 3)));
        sb.append(repeat);
        sb.append(StringUtils.repeat((String)"-", (int)(textStart - sb.length())));
        sb.append(" ");
        sb.append(text);
        sb.append("\n");
        return sb.toString();
    }
}

