/*
 * Decompiled with CFR 0.152.
 */
package io.github.slimjar.downloader;

import io.github.slimjar.downloader.DependencyDownloader;
import io.github.slimjar.downloader.output.OutputWriter;
import io.github.slimjar.downloader.output.OutputWriterFactory;
import io.github.slimjar.downloader.verify.DependencyVerifier;
import io.github.slimjar.exceptions.DownloaderException;
import io.github.slimjar.exceptions.UnresolvedDependencyException;
import io.github.slimjar.libs.annotations.Contract;
import io.github.slimjar.libs.annotations.NotNull;
import io.github.slimjar.logging.LocationAwareProcessLogger;
import io.github.slimjar.logging.LogDispatcher;
import io.github.slimjar.logging.ProcessLogger;
import io.github.slimjar.resolver.DependencyResolver;
import io.github.slimjar.resolver.ResolutionResult;
import io.github.slimjar.resolver.data.Dependency;
import io.github.slimjar.util.Connections;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.Optional;

public final class URLDependencyDownloader
implements DependencyDownloader {
    @NotNull
    private static final ProcessLogger LOGGER = LocationAwareProcessLogger.generic();
    private static final byte[] BOM_BYTES = "bom-file".getBytes();
    @NotNull
    private final OutputWriterFactory outputWriterProducer;
    @NotNull
    private final DependencyResolver dependencyResolver;
    @NotNull
    private final DependencyVerifier verifier;

    @Contract(pure=true)
    public URLDependencyDownloader(@NotNull OutputWriterFactory outputWriterProducer, @NotNull DependencyResolver dependencyResolver, @NotNull DependencyVerifier verifier) {
        this.outputWriterProducer = outputWriterProducer;
        this.dependencyResolver = dependencyResolver;
        this.verifier = verifier;
    }

    @Override
    @NotNull
    public Optional<File> download(@NotNull Dependency dependency) throws DownloaderException {
        InputStream inputStream;
        URLConnection connection;
        File expectedOutputFile = this.outputWriterProducer.getStrategy().selectFileFor(dependency);
        if (this.existingBOM(expectedOutputFile.toPath())) {
            LOGGER.debug("Skipping download of %s because it is a bom.", dependency);
            return Optional.empty();
        }
        if (this.verifier.verify(expectedOutputFile, dependency)) {
            LOGGER.debug("Skipping download of %s because it is already downloaded.", dependency);
            return Optional.of(expectedOutputFile);
        }
        ResolutionResult result = this.dependencyResolver.resolve(dependency).orElseThrow(() -> new UnresolvedDependencyException(dependency));
        if (result.aggregator()) {
            this.writeBOM(expectedOutputFile.toPath());
            return Optional.empty();
        }
        this.cleanupExisting(expectedOutputFile, dependency);
        URL url = result.dependencyURL();
        LogDispatcher.getMediatingLogger().info("Downloading %s", url);
        try {
            connection = Connections.createDownloadConnection(url);
            inputStream = connection.getInputStream();
        }
        catch (IOException err) {
            throw new DownloaderException("Failed to connect to " + String.valueOf(url), err);
        }
        LOGGER.debug("Connection successful! Downloading %s", String.valueOf(dependency) + "...");
        OutputWriter outputWriter = this.outputWriterProducer.create(dependency);
        LOGGER.debug("%s.Size = %s", dependency, connection.getContentLength());
        File downloadResult = outputWriter.writeFrom(inputStream, connection.getContentLength());
        Connections.tryDisconnect(connection);
        this.verifier.verify(downloadResult, dependency);
        LOGGER.debug("Artifact %s downloaded successfully from %s!", dependency, url);
        return Optional.of(downloadResult);
    }

    private boolean existingBOM(@NotNull Path expectedOutputPath) throws DownloaderException {
        try {
            if (expectedOutputPath.toFile().exists() && expectedOutputPath.toFile().length() == (long)BOM_BYTES.length && Arrays.equals(Files.readAllBytes(expectedOutputPath), BOM_BYTES)) {
                return true;
            }
        }
        catch (IOException err) {
            throw new DownloaderException("Failed to read existing BOM file", err);
        }
        return false;
    }

    private void writeBOM(@NotNull Path expectedOutputPath) throws DownloaderException {
        try {
            Files.createDirectories(expectedOutputPath.getParent(), new FileAttribute[0]);
            Files.createFile(expectedOutputPath, new FileAttribute[0]);
            Files.write(expectedOutputPath, BOM_BYTES, new OpenOption[0]);
        }
        catch (IOException err) {
            throw new DownloaderException("Failed to write to BOM file.", err);
        }
    }

    private void cleanupExisting(@NotNull File expectedOutputFile, @NotNull Dependency dependency) throws DownloaderException {
        try {
            Optional<Path> checksumFile;
            if (expectedOutputFile.exists()) {
                Files.delete(expectedOutputFile.toPath());
            }
            if ((checksumFile = this.verifier.getChecksumFile(dependency).filter(File::exists).map(File::toPath)).isEmpty()) {
                return;
            }
            Files.delete(checksumFile.get());
        }
        catch (IOException err) {
            throw new DownloaderException("Failed to cleanup existing files.", err);
        }
    }
}

