/*
 * Decompiled with CFR 0.152.
 */
package io.github.slimjar.downloader.verify;

import io.github.slimjar.downloader.output.OutputWriter;
import io.github.slimjar.downloader.output.OutputWriterFactory;
import io.github.slimjar.downloader.verify.ChecksumCalculator;
import io.github.slimjar.downloader.verify.DependencyVerifier;
import io.github.slimjar.exceptions.VerificationException;
import io.github.slimjar.libs.annotations.Contract;
import io.github.slimjar.libs.annotations.NotNull;
import io.github.slimjar.logging.LocationAwareProcessLogger;
import io.github.slimjar.logging.ProcessLogger;
import io.github.slimjar.resolver.DependencyResolver;
import io.github.slimjar.resolver.ResolutionResult;
import io.github.slimjar.resolver.data.Dependency;
import io.github.slimjar.util.Connections;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.Files;
import java.util.Optional;

public final class ChecksumDependencyVerifier
implements DependencyVerifier {
    @NotNull
    private static final ProcessLogger LOGGER = LocationAwareProcessLogger.generic();
    @NotNull
    private final DependencyResolver resolver;
    @NotNull
    private final OutputWriterFactory outputWriterFactory;
    @NotNull
    private final DependencyVerifier fallbackVerifier;
    @NotNull
    private final ChecksumCalculator checksumCalculator;

    @Contract(pure=true)
    public ChecksumDependencyVerifier(@NotNull DependencyResolver resolver, @NotNull OutputWriterFactory outputWriterFactory, @NotNull DependencyVerifier fallbackVerifier, @NotNull ChecksumCalculator checksumCalculator) {
        this.resolver = resolver;
        this.outputWriterFactory = outputWriterFactory;
        this.fallbackVerifier = fallbackVerifier;
        this.checksumCalculator = checksumCalculator;
    }

    @Override
    public boolean verify(@NotNull File file, @NotNull Dependency dependency) throws VerificationException {
        String expectedChecksum;
        if (!file.exists()) {
            return false;
        }
        LOGGER.info("Verifying checksum for %s", dependency.artifactId());
        File checksumFile = this.outputWriterFactory.getStrategy().selectFileFor(dependency);
        checksumFile.getParentFile().mkdirs();
        if (!checksumFile.exists() && !this.prepareChecksumFile(checksumFile, dependency)) {
            LOGGER.debug("Unable to resolve checksum for %s, falling back to fallbackVerifier!", dependency.artifactId());
            return this.fallbackVerifier.verify(file, dependency);
        }
        if (checksumFile.length() == 0L) {
            LOGGER.debug("Required checksum not found for %s, using fallbackVerifier!", dependency.artifactId());
            return this.fallbackVerifier.verify(file, dependency);
        }
        String actualChecksum = this.checksumCalculator.calculate(file);
        try {
            expectedChecksum = new String(Files.readAllBytes(checksumFile.toPath())).trim();
        }
        catch (IOException e) {
            throw new VerificationException("Unable to read bytes from checksum file (%s)".formatted(checksumFile), e);
        }
        LOGGER.debug("%s -> Actual checksum: %s;", dependency.artifactId(), actualChecksum);
        LOGGER.debug("%s -> Expected checksum: %s;", dependency.artifactId(), expectedChecksum);
        if (!actualChecksum.equals(expectedChecksum)) {
            LOGGER.error("Checksum mismatch for %s, expected %s, got %s", dependency.artifactId(), expectedChecksum, actualChecksum);
            return false;
        }
        LOGGER.debug("Checksum matched for %s.", dependency.artifactId());
        return true;
    }

    @Override
    @NotNull
    public Optional<File> getChecksumFile(@NotNull Dependency dependency) {
        File checksumFile = this.outputWriterFactory.getStrategy().selectFileFor(dependency);
        checksumFile.getParentFile().mkdirs();
        return Optional.of(checksumFile);
    }

    private boolean prepareChecksumFile(@NotNull File checksumFile, @NotNull Dependency dependency) throws VerificationException {
        Optional<ResolutionResult> result = this.resolver.resolve(dependency);
        if (result.isEmpty()) {
            return false;
        }
        URL checkSumUrl = result.get().checksumURL();
        LOGGER.info("Resolved checksum URL for %s as %s", dependency.artifactId(), checkSumUrl);
        try {
            if (checkSumUrl == null) {
                checksumFile.createNewFile();
                return false;
            }
            URLConnection connection = Connections.createDownloadConnection(checkSumUrl);
            InputStream inputStream = connection.getInputStream();
            OutputWriter outputWriter = this.outputWriterFactory.create(dependency);
            outputWriter.writeFrom(inputStream, connection.getContentLength());
            Connections.tryDisconnect(connection);
        }
        catch (IOException err) {
            throw new VerificationException("Unable to get checksum for %s".formatted(dependency.toString()), err);
        }
        LOGGER.info("Downloaded checksum for %s", dependency.artifactId());
        return true;
    }
}

