package de.comhix.ktorAnnotatedRouting.routeCreator

import de.comhix.ktorAnnotatedRouting.annotations.*
import de.comhix.ktorAnnotatedRouting.routeCreator.ParameterType.*
import io.ktor.http.HttpStatusCode
import io.ktor.server.application.ApplicationCall
import io.ktor.server.application.call
import io.ktor.server.auth.authenticate
import io.ktor.server.request.receive
import io.ktor.server.request.receiveParameters
import io.ktor.server.response.respond
import io.ktor.server.routing.*
import io.ktor.util.pipeline.PipelineInterceptor
import io.ktor.util.reflect.TypeInfo
import io.ktor.util.reflect.platformType
import kotlinx.serialization.json.Json
import kotlinx.serialization.serializer
import java.lang.reflect.InvocationTargetException
import kotlin.reflect.*
import kotlin.reflect.full.*
import de.comhix.ktorAnnotatedRouting.annotations.HttpMethod.Delete as DELETE_METHOD
import de.comhix.ktorAnnotatedRouting.annotations.HttpMethod.Get as GET_METHOD
import de.comhix.ktorAnnotatedRouting.annotations.HttpMethod.Post as POST_METHOD
import de.comhix.ktorAnnotatedRouting.annotations.HttpMethod.Put as PUT_METHOD

/**
 * @author Benjamin Beeker
 */
@Suppress("unused")
fun Routing.createRoutes(
    vararg api: Any,
    guestRole: String = "Guest",
    permissionForRole: (String) -> Set<String> = { emptySet() },
    requiredRoleCheck: ApplicationCall.(Set<String>) -> Boolean
) {
    api.forEach {
        createRoutes(it, guestRole, permissionForRole, requiredRoleCheck)
    }
}

fun Routing.createRoutes(
    api: Any,
    guestRole: String = "Guest",
    permissionForRole: (String) -> Set<String> = { emptySet() },
    requiredPermissionCheck: ApplicationCall.(Set<String>) -> Boolean
) {
    info { "load routes for class ${api::class}" }

    val classes = listOf(api::class) + api::class.allSuperclasses

    // base config of whole class
    val baseAnnotation: Path = classes.find { it.hasAnnotation<Path>() }?.findAnnotation() ?: Path()
    val baseRoles: Set<String> = classes.find { it.hasAnnotation<RequiredRole>() }
        ?.findAnnotation<RequiredRole>()
        ?.roles
        ?.toSet() ?: setOf(guestRole)
    val basePermissions: Set<String> = classes.find { it.hasAnnotation<RequiredPermissions>() }
        ?.findAnnotation<RequiredPermissions>()
        ?.permissions
        ?.toSet() ?: permissionForRole(guestRole)
    val baseAuthenticator: Authenticator? = classes.find { it.hasAnnotation<Authenticator>() }?.findAnnotation()

    info { "basePath: ${baseAnnotation.path}" }

    classes.forEach { klass ->
        klass.memberFunctions
            .mapNotNull {
                info { "check function ${it.name}" }
                val functionAnnotation: Path = it.findAnnotation() ?: return@mapNotNull null
                functionAnnotation to it
            }
            .toMap()
            .forEach { apiFunction ->
                info { "creating routing for ${apiFunction.key.method} ${apiFunction.key.path}" }

                val interfaceFunction = apiFunction.value

                val endpointPath = listOf(baseAnnotation.path, apiFunction.key.path).mapNotNull { it.ifEmpty { null } }.joinToString(separator = "/")

                val functionParameters = interfaceFunction.valueParameters
                val returnType = interfaceFunction.returnType
                val parameterTypes = functionParameters.map { it.type }

                val parameterEntries = functionParameters.map { it.toParameterEntry() }
                check(parameterEntries.filter { it.parameterType == BODY }.size <= 1) { "more then one body parameter in function ${apiFunction.value.name}" }

                val functions = api::class.functions
                val actualFunction = functions.find {
                    it.name == interfaceFunction.name &&
                            it.valueParameters.map(KParameter::type) == parameterTypes
                }!!

                val (authenticator, neededPermissions) = getAuthInformation(
                    baseAuthenticator,
                    baseRoles,
                    basePermissions,
                    permissionForRole,
                    interfaceFunction,
                    actualFunction
                )

                val httpFunction: Route.(String, PipelineInterceptor<Unit, ApplicationCall>) -> Route = when (apiFunction.key.method) {
                    GET_METHOD    -> { path, body ->
                        get(path, body)
                    }

                    POST_METHOD   -> { path, body ->
                        post(path, body)
                    }

                    PUT_METHOD    -> { path, body ->
                        put(path, body)
                    }

                    DELETE_METHOD -> { path, body ->
                        delete(path, body)
                    }
                }

                val httpCall: Route.() -> Unit = {
                    httpFunction(endpointPath) {
                        if (!call.requiredPermissionCheck(neededPermissions)) {
                            throw HttpException(HttpStatusCode.Forbidden)
                        }

                        val parameters = parameterEntries.map { it.toParameter(call) }.toTypedArray()

                        val message: Any? = try {
                            actualFunction.callSuspend(api, *parameters)
                        }
                        catch (exception: InvocationTargetException) {
                            throw exception.targetException
                        }

                        if (message != null) {
                            val typeInfo = TypeInfo(returnType.classifier as KClass<*>, returnType.platformType, returnType)
                            call.respond(message, typeInfo)
                        }
                        else {
                            call.response.status(HttpStatusCode.NoContent)
                        }
                    }
                }

                val authLayer = if (authenticator != null) {
                    {
                        authenticate(authenticator.name, optional = authenticator.optional, build = httpCall)
                    }
                }
                else {
                    {
                        httpCall()
                    }
                }

                authLayer()
            }
    }
}

private fun getAuthInformation(
    baseAuthenticator: Authenticator?,
    defaultRoles: Set<String>,
    defaultPermissions: Set<String>,
    permissionForRole: (String) -> Set<String>,
    interfaceFunction: KFunction<*>,
    actualFunction: KFunction<*>
): Pair<Authenticator?, Set<String>> {
    val authenticator = actualFunction.findAnnotation() ?: interfaceFunction.findAnnotation() ?: baseAuthenticator
    val requiredRoles: RequiredRole? = actualFunction.findAnnotation() ?: interfaceFunction.findAnnotation()
    val requiredPermissions: RequiredPermissions? = actualFunction.findAnnotation() ?: interfaceFunction.findAnnotation()

    val neededRoles = (requiredRoles?.roles?.toSet() ?: defaultRoles).flatMap(permissionForRole) +
            (requiredPermissions?.permissions?.toSet() ?: defaultPermissions)

    return Pair(authenticator, neededRoles.toSet())
}

private fun KParameter.toParameterEntry(): ParameterEntry {
    return if (this.type == typeOf<ApplicationCall>()) {
        ParameterEntry(CALL, this.type)
    }
    else if (this.hasAnnotation<PostParam>()) {
        ParameterEntry(POST, this.type, this.findAnnotation<PostParam>()!!.name)
    }
    else if (this.hasAnnotation<PathParam>()) {
        ParameterEntry(PATH, this.type, this.findAnnotation<PathParam>()!!.name)
    }
    else if (this.hasAnnotation<QueryParam>()) {
        ParameterEntry(QUERY, this.type, this.findAnnotation<QueryParam>()!!.name)
    }
    else {
        ParameterEntry(BODY, this.type)
    }
}

private data class ParameterEntry(val parameterType: ParameterType, val type: KType, val name: String? = null) {
    suspend fun toParameter(call: ApplicationCall): Any? {
        return when (this.parameterType) {
            BODY  -> call.receive(type.toTypeInfo())
            POST  -> call.receiveParameters()[name!!]?.toType(type)
            PATH  -> call.parameters[this.name!!]?.toType(type)
            QUERY -> call.request.queryParameters[this.name!!]?.toType(type)
            CALL  -> call
        }
    }
}

private fun String.toType(type: KType): Any? {
    return Json.decodeFromString(serializer(type), "\"$this\"")
}

private fun KType.toTypeInfo() = TypeInfo(this.classifier as KClass<*>, this.platformType, this)

private enum class ParameterType {
    BODY,
    POST,
    PATH,
    QUERY,
    CALL
}

private fun info(message: () -> String) = logFunction.invoke(message())

var logFunction: (String) -> Unit = {}