package de.comhix.database.dao

import com.google.inject.Inject
import de.comhix.database.dao.Query.Operation
import de.comhix.database.objects.UserDatabaseObject
import io.reactivex.Completable
import io.reactivex.Maybe
import io.reactivex.Single
import org.bson.types.ObjectId
import javax.inject.Named

/**
 * @author Benjamin Beeker
 */
class UserDao @Inject constructor(private val baseDao: BaseDao, @Named(USER_INJECT) private val user: String)
    : SimpleDao<UserDatabaseObject> {
    override fun <Type : UserDatabaseObject> get(id: String, typeClass: Class<Type>): Maybe<Type> {
        return baseDao.get(id, typeClass)
                .flatMap { `object`: UserDatabaseObject ->
                    if (`object`!!.user == user) {
                        return@flatMap Maybe.just(`object` as Type)
                    }
                    Maybe.empty()
                }
    }

    override fun <Type : UserDatabaseObject> save(instance: Type): Single<Type> {
        return (baseDao[instance!!.id, instance.javaClass] as Maybe<UserDatabaseObject>)
                .toSingle(instance)
                .flatMap { foundObject: UserDatabaseObject ->
                    if (foundObject!!.user != user) {
                        instance.id = ObjectId().toString()
                        instance.version = 0
                    }
                    instance.user = user
                    baseDao.save(instance)
                }
    }

    override fun <Type : UserDatabaseObject> delete(id: String, typeClass: Class<Type>): Completable {
        return baseDao.get(id, typeClass)
                .flatMapCompletable { `object`: UserDatabaseObject ->
                    if (`object`!!.user != user) {
                        return@flatMapCompletable Completable.complete()
                    }
                    baseDao.delete(id, typeClass)
                }
    }

    override fun <Type : UserDatabaseObject> query(typeClass: Class<Type>): Query<Type> {
        return baseDao.query(typeClass).with("user", Operation.EQ, user)!!
    }

    companion object {
        const val USER_INJECT = "User_Inject"
    }
}