package de.comhix.database.dao

import de.comhix.database.objects.DatabaseObject
import io.reactivex.Completable
import io.reactivex.Maybe
import io.reactivex.Single
import io.reactivex.schedulers.Schedulers
import org.mongodb.morphia.Datastore
import javax.inject.Inject

/**
 * @author Benjamin Beeker
 */
class BaseDao @Inject constructor(private val datastore: Datastore) : SimpleDao<DatabaseObject> {
    override fun <Type : DatabaseObject> get(id: String, typeClass: Class<Type>): Maybe<Type> {
        return Maybe.fromCallable { datastore[typeClass, id] }
                .subscribeOn(Schedulers.io())
    }

    override fun <Type : DatabaseObject> save(instance: Type): Single<Type> {
        return Single.fromCallable {
            val key = datastore.save(instance)
            datastore.getByKey(instance::class.java as Class<Type>?, key)
        }.subscribeOn(Schedulers.io())
    }

    override fun <Type : DatabaseObject> delete(id: String, typeClass: Class<Type>): Completable {
        return Completable.fromCallable { datastore.delete(typeClass, id) }
                .subscribeOn(Schedulers.io())
    }

    override fun <Type : DatabaseObject> query(typeClass: Class<Type>): Query<Type> {
        return Query(typeClass, datastore)
    }
}