package de.comhix.twitch.database.dao;

import de.comhix.twitch.database.objects.QueryResult;
import io.reactivex.Maybe;
import io.reactivex.Observable;
import io.reactivex.Single;
import io.reactivex.schedulers.Schedulers;
import org.mongodb.morphia.Datastore;
import org.mongodb.morphia.query.FindOptions;

/**
 * @author Benjamin Beeker
 */
public class Query<Type> {

    private final org.mongodb.morphia.query.Query<Type> query;
    private final FindOptions options = new FindOptions();

    /**
     * only for {@link de.comhix.twitch.database.dao.test.MockQuery}
     */
    protected Query() {
        query = null;
    }

    protected Query(Class<Type> typeClass, Datastore datastore) {
        query = datastore.createQuery(typeClass);
    }

    public <AllowedValueType> Query<Type> with(String field, Operation<AllowedValueType> operation, AllowedValueType value) {
        if (operation == Operation.EQ) {
            query.field(field).equal(value);
        }
        else if (operation == Operation.NEQ) {
            query.field(field).notEqual(value);
        }
        else if (operation == Operation.GT) {
            query.field(field).greaterThan(value);
        }
        else if (operation == Operation.GEQ) {
            query.field(field).greaterThanOrEq(value);
        }
        else if (operation == Operation.LT) {
            query.field(field).lessThan(value);
        }
        else if (operation == Operation.LEQ) {
            query.field(field).lessThanOrEq(value);
        }
        else if (operation == Operation.EXISTS) {
            if ((Boolean) value) {
                query.field(field).exists();
            }
            else {
                query.field(field).doesNotExist();
            }
        }
        else if (operation == Operation.IN) {
            query.field(field).in((Iterable<?>) value);
        }
        else if (operation == Operation.NOT_IN) {
            query.field(field).notIn((Iterable<?>) value);
        }
        else if (operation == Operation.HAS) {
            query.field(field).hasThisOne(value);
        }
        else if (operation == Operation.HAS_ANY) {
            query.field(field).hasAnyOf((Iterable<?>) value);
        }
        else if (operation == Operation.HAS_NONE) {
            query.field(field).hasNoneOf((Iterable<?>) value);
        }
        else if (operation == Operation.HAS_ALL) {
            query.field(field).hasAllOf((Iterable<?>) value);
        }

        return this;
    }

    public Query<Type> limit(int limit) {
        options.limit(limit);
        return this;
    }

    public Query<Type> skip(int skip) {
        options.skip(skip);
        return this;
    }

    public Query<Type> order(String field) {
        query.order(field);
        return this;
    }

    public Single<QueryResult<Type>> query() {
        return Observable.zip(Observable.fromCallable(() -> query.asList(options)),
                Observable.fromCallable(query::count),
                QueryResult::new)
                .firstOrError()
                .subscribeOn(Schedulers.io());
    }

    public Maybe<Type> find() {
        limit(1);
        return query().flatMapMaybe(results -> {
            if (results.isEmpty()) {
                return Maybe.empty();
            }
            else {
                return Maybe.just(results.get(0));
            }
        });
    }

    @SuppressWarnings("unused")
    public static class Operation<AllowedValueType> {
        public static final Operation<Object> EQ = new Operation<>();
        public static final Operation<Object> NEQ = new Operation<>();
        public static final Operation<Comparable> GT = new Operation<>();
        public static final Operation<Comparable> GEQ = new Operation<>();
        public static final Operation<Comparable> LT = new Operation<>();
        public static final Operation<Comparable> LEQ = new Operation<>();
        public static final Operation<Boolean> EXISTS = new Operation<>();
        public static final Operation<Iterable> IN = new Operation<>();
        public static final Operation<Iterable> NOT_IN = new Operation<>();
        public static final Operation<Object> HAS = new Operation<>();
        public static final Operation<Iterable> HAS_ANY = new Operation<>();
        public static final Operation<Iterable> HAS_NONE = new Operation<>();
        public static final Operation<Iterable> HAS_ALL = new Operation<>();

        private Operation() {
        }
    }
}
