package de.comhix.twitch.database.dao;

import com.google.inject.Inject;
import de.comhix.twitch.database.objects.UserDatabaseObject;
import io.reactivex.Completable;
import io.reactivex.Maybe;
import io.reactivex.Single;
import org.bson.types.ObjectId;

import javax.inject.Named;
import java.util.Objects;

import static de.comhix.twitch.database.dao.Query.Operation.EQ;

/**
 * @author Benjamin Beeker
 */
public class UserDao implements SimpleDao<UserDatabaseObject> {

    public static final String USER_INJECT = "User_Inject";
    private final BaseDao baseDao;
    private final String user;

    @Inject
    public UserDao(BaseDao baseDao, @Named(USER_INJECT) String user) {
        this.baseDao = baseDao;
        this.user = user;
    }

    @Override
    public <Type extends UserDatabaseObject> Maybe<Type> get(String id, Class<Type> typeClass) {
        return baseDao.get(id, typeClass)
                .flatMap(object -> {
                    if (Objects.equals(object.user, user)) {
                        return Maybe.just(object);
            }
                    return Maybe.empty();
        });
    }

    @Override
    public <Type extends UserDatabaseObject> Single<Type> save(Type object) {
        //noinspection unchecked
        return ((Maybe<Type>) baseDao.get(object.id, object.getClass()))
                .toSingle(object)
                .flatMap(foundObject -> {
                    if (!Objects.equals(foundObject.user, user)) {
                        object.id = new ObjectId().toString();
                        object.version = 0;
                    }
                    object.user = user;
                    return baseDao.save(object);
                });
    }

    @Override
    public <Type extends UserDatabaseObject> Completable delete(String id, Class<Type> typeClass) {
        return baseDao.get(id, typeClass)
                .flatMapCompletable(object -> {
                    if (!Objects.equals(object.user, user)) {
                        return Completable.complete();
                    }
                    return baseDao.delete(id, typeClass);
                });
    }

    @Override
    public <Type extends UserDatabaseObject> Query<Type> query(Class<Type> typeClass) {
        return baseDao.query(typeClass).with("user", EQ, user);
    }
}
