package de.comhix.twitch.database.dao;

import de.comhix.twitch.database.objects.DatabaseObject;
import io.reactivex.Completable;
import io.reactivex.Maybe;
import io.reactivex.Single;
import io.reactivex.schedulers.Schedulers;
import org.mongodb.morphia.Datastore;
import org.mongodb.morphia.Key;

import javax.inject.Inject;

/**
 * @author Benjamin Beeker
 */
public class BaseDao implements SimpleDao<DatabaseObject> {
    private final Datastore datastore;

    @Inject
    public BaseDao(Datastore datastore) {
        this.datastore = datastore;
    }

    public <Type extends DatabaseObject> Maybe<Type> get(String id, Class<Type> typeClass) {
        return Maybe.fromCallable(() -> datastore.get(typeClass, id))
                .subscribeOn(Schedulers.io());
    }

    public <Type extends DatabaseObject> Single<Type> save(Type object) {
        return Single.fromCallable(() -> {
            Key<Type> key = datastore.save(object);
            //noinspection unchecked
            return datastore.getByKey((Class<Type>) object.getClass(), key);
        }).subscribeOn(Schedulers.io());
    }

    public <Type extends DatabaseObject> Completable delete(String id, Class<Type> typeClass) {
        return Completable.fromCallable(() -> datastore.delete(typeClass, id))
                .subscribeOn(Schedulers.io());
    }

    public <Type extends DatabaseObject> Query<Type> query(Class<Type> typeClass) {
        return new Query<>(typeClass, datastore);
    }
}
