package de.comhix.twitch.database.dao;

import com.google.inject.Inject;
import de.comhix.twitch.database.objects.UserDatabaseObject;
import io.reactivex.Completable;
import io.reactivex.Observable;
import org.bson.types.ObjectId;

import javax.inject.Named;
import java.util.Objects;
import java.util.Optional;

/**
 * @author Benjamin Beeker
 */
public class UserDao implements SimpleDao<UserDatabaseObject> {

    public static final String USER_INJECT = "User_Inject";
    private final BaseDao baseDao;
    private final String user;

    @Inject
    public UserDao(BaseDao baseDao, @Named(USER_INJECT) String user) {
        this.baseDao = baseDao;
        this.user = user;
    }

    @Override
    public <Type extends UserDatabaseObject> Observable<Optional<Type>> get(String id, Class<Type> typeClass) {
        return baseDao.get(id, typeClass).map(object -> {
            if (object.isPresent() && Objects.equals(object.get().user, user)) {
                return object;
            }
            return Optional.empty();
        });
    }

    @Override
    public <Type extends UserDatabaseObject> Observable<Type> save(Type object) {
        return baseDao.get(object.id, object.getClass())
                .flatMap(foundObject -> {
                    if (foundObject.isPresent() && !Objects.equals(foundObject.get().user, user)) {
                        object.id = new ObjectId().toString();
                    }
                    return baseDao.save(object);
                });
    }

    @Override
    public <Type extends UserDatabaseObject> Completable delete(String id, Class<Type> typeClass) {
        return baseDao.get(id, typeClass).flatMapCompletable(object -> {
            if (object.isPresent()) {
                if (Objects.equals(object.get().user, user)) {
                    throw new IllegalAccessException();
                }
                return baseDao.delete(id, typeClass);
            }
            return Completable.complete();
        });
    }

    @Override
    public <Type extends UserDatabaseObject> Query<Type> query(Class<Type> typeClass) {
        return baseDao.query(typeClass).with("user", Query.Operation.EQ, user);
    }
}
