package de.comhix.twitch.database.dao;

import de.comhix.twitch.database.objects.DatabaseObject;
import io.reactivex.Completable;
import io.reactivex.Observable;
import io.reactivex.schedulers.Schedulers;
import org.mongodb.morphia.Datastore;
import org.mongodb.morphia.Key;

import javax.inject.Inject;
import java.util.Optional;

/**
 * @author Benjamin Beeker
 */
public class BaseDao implements SimpleDao<DatabaseObject> {
    private final Datastore datastore;

    @Inject
    public BaseDao(Datastore datastore) {
        this.datastore = datastore;
    }

    public <Type extends DatabaseObject> Observable<Optional<Type>> get(String id, Class<Type> typeClass) {
        return Observable.fromCallable(() -> Optional.ofNullable(datastore.get(typeClass, id))).
                subscribeOn(Schedulers.io());
    }

    public <Type extends DatabaseObject> Observable<Type> save(Type object) {
        return Observable.fromCallable(() -> {
            Key<Type> key = datastore.save(object);
            //noinspection unchecked
            return datastore.getByKey((Class<Type>) object.getClass(), key);
        }).subscribeOn(Schedulers.io());
    }

    public <Type extends DatabaseObject> Completable delete(String id, Class<Type> typeClass) {
        return Completable.fromCallable(() -> datastore.delete(typeClass, id))
                .subscribeOn(Schedulers.io());
    }

    public <Type extends DatabaseObject> Query<Type> query(Class<Type> typeClass) {
        return new Query<>(typeClass, datastore);
    }
}
