package de.comhix.commons.logging

import com.rollbar.api.payload.data.Level
import com.rollbar.api.payload.data.Level.*
import com.rollbar.notifier.Rollbar
import com.rollbar.notifier.config.ConfigBuilder
import org.slf4j.MDC
import java.util.*
import java.util.logging.Formatter
import java.util.logging.Handler
import java.util.logging.LogManager
import java.util.logging.LogRecord
import kotlin.collections.HashMap

/**
 * @author Benjamin Beeker
 */
class RollbarHandler : Handler() {
    private var apiKey: String? = null
    private var environment: String? = null
    private var rollbar: Rollbar? = null
    private fun getRollbar(): Rollbar? {
        if (rollbar == null) {
            configure()
            val config = ConfigBuilder.withAccessToken(apiKey).environment(environment).build()
            rollbar = Rollbar(config)
        }
        return rollbar
    }

    private fun configure() {
        val prexif = this.javaClass.name
        if (apiKey == null) {
            apiKey = getProperty("$prexif.apiKey")
        }
        if (environment == null) {
            environment = getProperty("$prexif.environment")
        }
        formatter = object : Formatter() {
            override fun format(record: LogRecord): String {
                return formatMessage(record)
            }
        }
    }

    private fun getProperty(key: String): String {
        return Optional.ofNullable(System.getProperty(key)).orElse(LogManager.getLogManager().getProperty(key))
    }

    override fun publish(record: LogRecord) {
        if (!isLoggable(record)) {
            return
        }
        val level: Level = when {
            record.level === java.util.logging.Level.SEVERE  -> ERROR
            record.level === java.util.logging.Level.WARNING -> WARNING
            record.level === java.util.logging.Level.INFO    -> INFO
            else                                             -> DEBUG
        }
        val customData: MutableMap<String, Any> = HashMap()
        customData["source"] = record.sourceClassName + "::" + record.sourceMethodName
        val mdcMap = MDC.getCopyOfContextMap()
        if (mdcMap != null) {
            customData.putAll(mdcMap)
        }
        getRollbar()!!.log(record.thrown, customData, formatter.formatMessage(record), level)
    }

    override fun flush() {}

    @Throws(SecurityException::class)
    override fun close() {
        try {
            getRollbar()!!.close(true)
        }
        catch (e: Exception) {
            throw RuntimeException(e)
        }
    }
}