package de.comhix.commons.logging;

import com.rollbar.api.payload.data.Level;
import com.rollbar.notifier.Rollbar;
import com.rollbar.notifier.config.Config;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Handler;
import java.util.logging.LogRecord;

import static com.rollbar.notifier.config.ConfigBuilder.withAccessToken;

public class RollbarAppender extends Handler {

    private String apiKey;
    private String environment;
    private Rollbar rollbar;

    private Rollbar getRollbar() {
        if (this.rollbar == null) {
            Config config = withAccessToken(apiKey).environment(environment).build();
            this.rollbar = new Rollbar(config);
        }
        return this.rollbar;
    }

    public void setApiKey(String apiKey) {
        this.apiKey = apiKey;
    }

    public void setEnvironment(String environment) {
        this.environment = environment.toLowerCase();
    }

    @Override
    public void publish(LogRecord record) {
        Level level;
        if (record.getLevel() == java.util.logging.Level.SEVERE) {
            level = Level.ERROR;
        }
        else if (record.getLevel() == java.util.logging.Level.WARNING) {
            level = Level.WARNING;
        }
        else if (record.getLevel() == java.util.logging.Level.INFO) {
            level = Level.INFO;
        }
        else {
            level = Level.DEBUG;
        }

        Map<String, Object> customData = new HashMap<>();
        customData.put("source", record.getSourceClassName() + "#" + record.getSourceMethodName());

        getRollbar().log(record.getThrown(), customData, formatMessage(record), level);
    }

    @Override
    public void flush() {
    }

    @Override
    public void close() throws SecurityException {
        try {
            getRollbar().close(true);
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private synchronized String formatMessage(LogRecord record) {
        String format = record.getMessage();
        java.util.ResourceBundle catalog = record.getResourceBundle();
        if (catalog != null) {
            try {
                format = catalog.getString(record.getMessage());
            }
            catch (java.util.MissingResourceException ex) {
                // Drop through.  Use record message as format
                format = record.getMessage();
            }
        }
        // Do the formatting.
        try {
            Object parameters[] = record.getParameters();
            if (parameters == null || parameters.length == 0) {
                // No parameters.  Just return format string.
                return format;
            }
            // Is it a java.text style format?
            // Ideally we could match with
            // Pattern.compile("\\{\\d").matcher(format).find())
            // However the cost is 14% higher, so we cheaply check for
            // 1 of the first 4 parameters
            if (format.indexOf("{0") >= 0 || format.indexOf("{1") >= 0 ||
                    format.indexOf("{2") >= 0 || format.indexOf("{3") >= 0) {
                return java.text.MessageFormat.format(format, parameters);
            }
            return format;
        }
        catch (Exception ex) {
            // Formatting failed: use localized format string.
            return format;
        }
    }
}