package de.codecentric.cxf.autodetection;

import de.codecentric.cxf.common.BootStarterCxfException;
import de.codecentric.cxf.autodetection.diagnostics.SeiImplClassNotFoundException;
import de.codecentric.cxf.autodetection.diagnostics.SeiNotFoundException;
import de.codecentric.cxf.autodetection.diagnostics.WebServiceClientNotFoundException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.jws.WebService;
import javax.xml.ws.Service;
import javax.xml.ws.WebServiceClient;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;


@Component
public class WebServiceAutoDetector {

    private static final Logger LOG = LoggerFactory.getLogger(WebServiceAutoDetector.class);
    protected static final String NO_CLASS_FOUND = "No class found";
    private final WebServiceScanner webServiceScanner;

    public static final Class<WebService> SEI_ANNOTATION = WebService.class;
    public static final Class<WebServiceClient> WEB_SERVICE_CLIENT_ANNOTATION = WebServiceClient.class;

    private final String seiAndWebServiceClientPackageName;
    private final String seiImplementationPackageName;

    public WebServiceAutoDetector(WebServiceScanner webServiceScanner) throws BootStarterCxfException {
        this.webServiceScanner = webServiceScanner;
        seiAndWebServiceClientPackageName = PackageNameReader.build().readSeiAndWebServiceClientPackageNameFromCxfSpringBootMavenProperties();
        seiImplementationPackageName = PackageNameReader.build().readSeiImplementationPackageNameFromCxfSpringBootMavenProperties();
    }

    /**
     * Detects & instantiates the SEI-Implementation. Therefore it detects the SEI itself first.
     *
     * @param <T>
     * @return
     * @throws BootStarterCxfException
     */
    public <T> T searchAndInstantiateSeiImplementation() throws BootStarterCxfException {
        return searchAndInstantiateSeiImplementation(searchServiceEndpointInterface());
    }

    @SuppressWarnings("unchecked")
    protected  <T> T searchAndInstantiateSeiImplementation(Class seiName) throws BootStarterCxfException {
        Class<T> implementingClass = null;
        try {
            implementingClass = webServiceScanner.scanForClassWhichImplementsAndPickFirst(seiName, seiImplementationPackageName);
            LOG.info("Found SEI implementing class: '{}'", implementingClass.getName());
        } catch (BootStarterCxfException exception) {
            throw SeiImplClassNotFoundException.build().setNotFoundClassName(seiName.getName()).setScannedBasePackage(seiImplementationPackageName);
        }
        return instantiateFromClass(implementingClass);
    }

    public Class searchServiceEndpointInterface() throws BootStarterCxfException {
        try{
            Class sei = webServiceScanner.scanForClassWithAnnotationAndIsAnInterface(SEI_ANNOTATION, seiAndWebServiceClientPackageName);
            LOG.info("Found Service Endpoint Interface (SEI): '{}'", sei.getName());
            return sei;
        } catch (BootStarterCxfException exception) {
            throw new SeiNotFoundException();
        }
    }

    @SuppressWarnings("unchecked")
    public Service searchAndInstantiateWebServiceClient() throws BootStarterCxfException {
        try{
            Class<Service> webServiceClientClass = webServiceScanner.scanForClassWithAnnotationAndPickTheFirstOneFound(WEB_SERVICE_CLIENT_ANNOTATION, seiAndWebServiceClientPackageName);
            LOG.info("Found WebServiceClient class: '{}'", webServiceClientClass.getName());
            return instantiateFromClass(webServiceClientClass);
        } catch (BootStarterCxfException exception) {
            throw new WebServiceClientNotFoundException();
        }
    }

    private <T> T instantiateFromClass(Class<T> clazz) throws BootStarterCxfException {
        try {
            Constructor<T> constructor = clazz.getConstructor();
            return constructor.newInstance();

        } catch (NoSuchMethodException |
                IllegalAccessException |
                InstantiationException |
                InvocationTargetException exception) {
            throw new BootStarterCxfException("Class couldn´t be instantiated", exception);
        }
    }

}
