package de.codecamp.maven.delayment;


import de.codecamp.maven.delayment.model.RecordedArtifacts;
import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import javax.inject.Inject;
import org.apache.maven.RepositoryUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProjectHelper;
import org.eclipse.aether.RepositorySystemSession;


@Mojo(name = ReattachArtifactsMojo.MOJO_NAME, defaultPhase = LifecyclePhase.VERIFY)
public class ReattachArtifactsMojo
  extends
    AbstractDelaymentMojo
{

  public static final String MOJO_NAME = "reattachArtifacts";

  public static final String CONF_USE_LOCAL_REPOSITORY = "useLocalRepository";


  @Inject
  private MavenProjectHelper helper;

  @Parameter(defaultValue = "${repositorySystemSession}", readonly = true, required = true)
  private RepositorySystemSession repositorySystemSession;

  @Parameter(
      property = DelaymentLifecycleParticipant.PROPERTY_USE_LOCAL_REPOSITORY,
      defaultValue = "false")
  private Boolean useLocalRepository;


  @Override
  public void execute()
    throws MojoExecutionException
  {
    Path artifactsFilePath = getArtifactsFilePath();

    getLog().info(
        "Reattaching artifacts for delayed deployment from file '" + artifactsFilePath + "':");

    if (!Files.isRegularFile(artifactsFilePath))
    {
      throw new MojoExecutionException("'" + artifactsFilePath.getFileName().toString()
          + "' missing. Recorded artifacts not found.");
    }

    RecordedArtifacts recordedArtifacts;
    try (Reader in = Files.newBufferedReader(artifactsFilePath, StandardCharsets.UTF_8))
    {
      recordedArtifacts = getGson().fromJson(in, RecordedArtifacts.class);
    }
    catch (IOException | RuntimeException ex)
    {
      throw new MojoExecutionException(
          "Failed to restore artifacts from '" + artifactsFilePath + "'.", ex);
    }

    if (recordedArtifacts.getMainFile() == null && recordedArtifacts.getArtifacts().isEmpty())
    {
      getLog().info("Nothing to reattach.");
      return;
    }


    if (recordedArtifacts.getMainFile() != null)
    {
      File mainArtifactFile;
      if (useLocalRepository)
      {
        File basedir = repositorySystemSession.getLocalRepository().getBasedir();
        mainArtifactFile = new File(basedir, repositorySystemSession.getLocalRepositoryManager()
            .getPathForLocalArtifact(RepositoryUtils.toArtifact(getProject().getArtifact())));
      }
      else
      {
        mainArtifactFile = recordedArtifacts.getMainFile();
      }
      getLog().info("  main - " + mainArtifactFile);
      getProject().getArtifact().setFile(mainArtifactFile);
    }

    for (RecordedArtifacts.Artifact recordedArtifact : recordedArtifacts.getArtifacts())
    {
      File attachedArtifactFile = recordedArtifact.getFile();

      helper.attachArtifact(getProject(), recordedArtifact.getType(),
          recordedArtifact.getClassifier(), attachedArtifactFile);

      if (useLocalRepository)
      {
        Artifact artifact =
            getProject().getAttachedArtifacts().get(getProject().getAttachedArtifacts().size() - 1);

        File basedir = repositorySystemSession.getLocalRepository().getBasedir();
        attachedArtifactFile = new File(basedir, repositorySystemSession.getLocalRepositoryManager()
            .getPathForLocalArtifact(RepositoryUtils.toArtifact(artifact)));
        artifact.setFile(attachedArtifactFile);
      }

      getLog().info(String.format("  %s:%s - %s", recordedArtifact.getType(),
          recordedArtifact.getClassifier(), attachedArtifactFile));
    }
  }

}
