package de.codecamp.maven.delayment;


import de.codecamp.maven.delayment.model.CleanLifecycle;
import de.codecamp.maven.delayment.model.DefaultLifecycle;
import java.util.ArrayList;
import java.util.Map;
import java.util.Optional;
import javax.inject.Named;
import javax.inject.Singleton;
import org.apache.maven.AbstractMavenLifecycleParticipant;
import org.apache.maven.MavenExecutionException;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.model.Extension;
import org.apache.maven.model.Plugin;
import org.apache.maven.model.PluginExecution;
import org.apache.maven.project.MavenProject;
import org.apache.maven.shared.utils.logging.MessageUtils;
import org.codehaus.plexus.logging.LogEnabled;
import org.codehaus.plexus.logging.Logger;
import org.codehaus.plexus.util.xml.Xpp3Dom;
import org.eclipse.sisu.Priority;


@Singleton
@Named("DelaymentLifecycleParticipant")
@Priority(Integer.MIN_VALUE)
public class DelaymentLifecycleParticipant
  extends
    AbstractMavenLifecycleParticipant
  implements
    LogEnabled
{

  public static final String GROUP_ID = "de.codecamp.maven";

  public static final String ARTIFACT_ID = "delayment-maven-plugin";

  public static final String PROPERTY_USE_LOCAL_REPOSITORY =
      "delayment." + ReattachArtifactsMojo.CONF_USE_LOCAL_REPOSITORY;


  private Logger log;

  private String extensionVersion;


  @Override
  public void enableLogging(Logger logger)
  {
    this.log = logger;
  }

  @Override
  public void afterProjectsRead(MavenSession session)
    throws MavenExecutionException
  {
    boolean useLocalRepository = getExtensionProperty(session, PROPERTY_USE_LOCAL_REPOSITORY)
        .map(Boolean::parseBoolean).orElse(false);

    Mode mode;
    if (session.getGoals().size() == 1 && session.getGoals().contains(CleanLifecycle.CLEAN.phase()))
    {
      mode = Mode.NONE;
    }
    else if (session.getGoals().contains(CleanLifecycle.CLEAN.phase()))
    {
      mode = Mode.RECORD;
    }
    else if ((session.getGoals().contains(DefaultLifecycle.INSTALL.phase()) && !useLocalRepository)
        || session.getGoals().contains(DefaultLifecycle.DEPLOY.phase()))
    {
      mode = Mode.RESTORE;
    }
    else
    {
      mode = Mode.RECORD;
    }

    log.info(String.format("--- %s ---", MessageUtils.buffer().warning(ARTIFACT_ID)));

    switch (mode)
    {
      case RECORD ->
      {
        log.info("RECORD-mode of delayed deployment enabled.");

        for (MavenProject project : session.getProjects())
        {
          log.debug("");
          log.debug(String.format("* %s",
              MessageUtils.buffer().project(project.getGroupId() + ":" + project.getArtifactId())));

          addPlugin(session, project, DefaultLifecycle.VERIFY, RecordArtifactsMojo.MOJO_NAME, null);
        }
      }

      case RESTORE ->
      {
        log.info("RESTORE-mode of delayed deployment enabled.");

        if (useLocalRepository)
        {
          log.info(
              "Artifact files will be reattached from local repository (instead of target-directory).");
        }

        DefaultLifecycle skippedPhases =
            useLocalRepository ? DefaultLifecycle.INSTALL : DefaultLifecycle.VERIFY;

        for (MavenProject project : session.getProjects())
        {
          log.debug("");
          log.debug(String.format("* %s",
              MessageUtils.buffer().project(project.getGroupId() + ":" + project.getArtifactId())));

          /*
           * Remove everything from 'completedStages' and before.
           *
           * Also remove all executions that don't have an explicit phase. It would be better the
           * default phase of the mojo, but that doesn't seem feasible.
           */
          removeExecutionsUpTo(project, skippedPhases);

          addPlugin(session, project, DefaultLifecycle.INITIALIZE, ReattachArtifactsMojo.MOJO_NAME,
              useLocalRepository
                  ? Map.of(ReattachArtifactsMojo.CONF_USE_LOCAL_REPOSITORY, Boolean.TRUE.toString())
                  : null);
        }
      }

      case NONE ->
      {
        log.info("Delayed deployment NOT enabled.");
      }
    }
  }

  private void addPlugin(MavenSession session, MavenProject project, DefaultLifecycle phase,
      String goal, Map<String, String> configuration)
    throws MavenExecutionException
  {
    log.debug(String.format("  Adding %s-goal to %s-phase.", MessageUtils.buffer().mojo(goal),
        phase.phase()));

    if (extensionVersion == null)
    {
      extensionVersion = getClass().getPackage().getImplementationVersion();
    }

    if (extensionVersion == null)
    {
      projectLoop: for (MavenProject p : session.getProjects())
      {
        for (Extension extension : p.getBuildExtensions())
        {
          if (extension.getGroupId().equals(GROUP_ID)
              && extension.getArtifactId().equals(ARTIFACT_ID))
          {
            extensionVersion = extension.getVersion();
            break projectLoop;
          }
        }
      }
    }

    if (extensionVersion == null)
    {
      throw new MavenExecutionException(
          "Failed to determine the used version of the '" + ARTIFACT_ID + "' extension.",
          (Throwable) null);
    }

    Plugin plugin = new Plugin();
    plugin.setGroupId(GROUP_ID);
    plugin.setArtifactId(ARTIFACT_ID);
    plugin.setVersion(extensionVersion);

    PluginExecution execution = new PluginExecution();
    execution.setPhase(phase.phase());
    execution.getGoals().add(goal);
    if (configuration != null)
    {
      Xpp3Dom executionConfiguration = new Xpp3Dom("configuration");
      execution.setConfiguration(executionConfiguration);
      configuration.forEach((key, value) ->
      {
        Xpp3Dom pluginConfEntry = new Xpp3Dom(key);
        pluginConfEntry.setValue(value);
        executionConfiguration.addChild(pluginConfEntry);
      });
    }

    plugin.getExecutions().add(execution);

    project.getBuild().getPlugins().add(plugin);
  }

  private void removeExecutionsUpTo(MavenProject project, DefaultLifecycle phase)
  {
    for (Plugin plugin : project.getBuild().getPlugins())
    {
      for (PluginExecution execution : new ArrayList<>(plugin.getExecutions()))
      {
        DefaultLifecycle executionPhase =
            DefaultLifecycle.valueOfPhaseOptional(execution.getPhase()).orElse(null);

        if (execution.getPhase() == null //
            || (executionPhase != null && executionPhase.ordinal() <= phase.ordinal()))
        {
          log.debug(String.format("  Removing execution %s (%s).",
              MessageUtils.buffer().mojo(plugin.getArtifactId()), execution.getId()));

          plugin.getExecutions().remove(execution);
        }
      }
    }
  }

  private static Optional<String> getExtensionProperty(MavenSession session, String key)
  {
    for (MavenProject project : session.getProjects())
    {
      String value = project.getProperties().getProperty(key);
      if (value != null)
        return Optional.of(value);
    }

    return Optional.empty();
  }


  private enum Mode
  {
    RECORD,
    RESTORE,
    NONE
  }

}
