package de.codecamp.maven.delayment;


import de.codecamp.maven.delayment.model.CleanLifecycle;
import de.codecamp.maven.delayment.model.DefaultLifecycle;
import java.util.ArrayList;
import java.util.Map;
import java.util.Optional;
import javax.inject.Named;
import javax.inject.Singleton;
import org.apache.maven.AbstractMavenLifecycleParticipant;
import org.apache.maven.MavenExecutionException;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.model.Extension;
import org.apache.maven.model.Plugin;
import org.apache.maven.model.PluginExecution;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.util.xml.Xpp3Dom;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


@Singleton
@Named("DelaymentLifecycleParticipant")
public class DelaymentLifecycleParticipant
  extends
    AbstractMavenLifecycleParticipant
{

  public static final String GROUP_ID = "de.codecamp.maven";

  public static final String ARTIFACT_ID = "delayment-maven-plugin";

  public static final String PROPERTY_USE_LOCAL_REPOSITORY =
      "delayment." + ReattachArtifactsMojo.CONF_USE_LOCAL_REPOSITORY;


  private static final Logger LOG = LoggerFactory.getLogger(DelaymentLifecycleParticipant.class);


  private String extensionVersion;


  @Override
  public void afterProjectsRead(MavenSession session)
    throws MavenExecutionException
  {
    boolean useLocalRepository = getExtensionProperty(session, PROPERTY_USE_LOCAL_REPOSITORY)
        .map(Boolean::parseBoolean).orElse(false);

    boolean recordPhase;
    if (session.getGoals().contains(CleanLifecycle.CLEAN.phase()))
    {
      recordPhase = true;
    }
    else if ((session.getGoals().contains(DefaultLifecycle.INSTALL.phase()) && !useLocalRepository)
        || session.getGoals().contains(DefaultLifecycle.DEPLOY.phase()))
    {
      recordPhase = false;
    }
    else
    {
      recordPhase = true;
    }

    LOG.info("**********");

    if (recordPhase)
    {
      LOG.info("Recording-phase of delayed deployment detected. Adding plugin to record attached"
          + " artifacts of all projects/modules.");
    }
    else
    {
      LOG.info("Reattachment-phase of delayed deployment detected. Removing all executions from"
          + " verify-phase or before and adding plugin to reattach recorded artifacts.");
    }

    for (MavenProject project : session.getProjects())
    {
      if (recordPhase)
      {
        LOG.info(String.format("Adding plugin to project %s:%s.", project.getGroupId(),
            project.getArtifactId()));
        addPlugin(session, project, RecordArtifactsMojo.MOJO_NAME, null);
      }
      else
      {
        LOG.info(String.format("Adding plugin to project %s:%s.", project.getGroupId(),
            project.getArtifactId()));
        addPlugin(session, project, ReattachArtifactsMojo.MOJO_NAME,
            useLocalRepository
                ? Map.of(ReattachArtifactsMojo.CONF_USE_LOCAL_REPOSITORY, Boolean.TRUE.toString())
                : null);

        LOG.info(String.format("Removing other plugins from project %s:%s.", project.getGroupId(),
            project.getArtifactId()));
        DefaultLifecycle completedStages =
            useLocalRepository ? DefaultLifecycle.INSTALL : DefaultLifecycle.VERIFY;
        /*
         * Remove everything from 'completedStages' and before.
         *
         * Also remove all executions that don't have an explicit phase. It would be better the
         * default phase of the mojo, but that doesn't seem feasible.
         */
        for (Plugin plugin : project.getBuild().getPlugins())
        {
          for (PluginExecution execution : new ArrayList<>(plugin.getExecutions()))
          {
            DefaultLifecycle executionPhase =
                DefaultLifecycle.valueOfPhaseOptional(execution.getPhase()).orElse(null);

            if (execution.getPhase() == null //
                || (executionPhase != null
                    && executionPhase.ordinal() <= completedStages.ordinal()))
            {
              plugin.getExecutions().remove(execution);
            }
          }
        }
      }
    }

    LOG.info("**********");
  }

  private void addPlugin(MavenSession session, MavenProject project, String goal,
      Map<String, String> configuration)
    throws MavenExecutionException
  {
    if (extensionVersion == null)
    {
      extensionVersion = getClass().getPackage().getImplementationVersion();
    }

    if (extensionVersion == null)
    {
      projectLoop: for (MavenProject p : session.getProjects())
      {
        for (Extension extension : p.getBuildExtensions())
        {
          if (extension.getGroupId().equals(GROUP_ID)
              && extension.getArtifactId().equals(ARTIFACT_ID))
          {
            extensionVersion = extension.getVersion();
            break projectLoop;
          }
        }
      }
    }

    if (extensionVersion == null)
    {
      throw new MavenExecutionException(
          "Failed to determine the used version of the '" + ARTIFACT_ID + "' extension.",
          (Throwable) null);
    }

    Plugin plugin = new Plugin();
    plugin.setGroupId(GROUP_ID);
    plugin.setArtifactId(ARTIFACT_ID);
    plugin.setVersion(extensionVersion);

    PluginExecution execution = new PluginExecution();
    execution.setPhase(DefaultLifecycle.VERIFY.phase());
    execution.getGoals().add(goal);
    if (configuration != null)
    {
      Xpp3Dom executionConfiguration = new Xpp3Dom("configuration");
      execution.setConfiguration(executionConfiguration);
      configuration.forEach((key, value) ->
      {
        Xpp3Dom pluginConfEntry = new Xpp3Dom(key);
        pluginConfEntry.setValue(value);
        executionConfiguration.addChild(pluginConfEntry);
      });
    }

    plugin.getExecutions().add(execution);

    project.getBuild().getPlugins().add(plugin);
  }

  private static Optional<String> getExtensionProperty(MavenSession session, String key)
  {
    for (MavenProject project : session.getProjects())
    {
      String value = project.getProperties().getProperty(key);
      if (value != null)
        return Optional.of(value);
    }

    return Optional.empty();
  }

}
