/*
 * Decompiled with CFR 0.152.
 */
package de.codesourcery.versiontracker.common.server;

import de.codesourcery.versiontracker.common.Artifact;
import de.codesourcery.versiontracker.common.IVersionProvider;
import de.codesourcery.versiontracker.common.IVersionStorage;
import de.codesourcery.versiontracker.common.VersionInfo;
import de.codesourcery.versiontracker.common.server.IVersionTracker;
import de.codesourcery.versiontracker.common.server.SharedLockCache;
import java.time.ZonedDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiPredicate;
import org.apache.commons.lang3.Validate;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class VersionTracker
implements IVersionTracker {
    private static final Logger LOG = LogManager.getLogger(VersionTracker.class);
    private final IVersionStorage versionStorage;
    private final IVersionProvider versionProvider;
    private final Object THREAD_POOL_LOCK = new Object();
    private int maxConcurrentThreads = 1;
    private ThreadPoolExecutor threadPool;
    private final SharedLockCache lockCache;
    private final ThreadFactory threadFactory = new ThreadFactory(){
        private final ThreadGroup threadGroup = new ThreadGroup(Thread.currentThread().getThreadGroup(), "versiontracker-request-threads");
        private final AtomicInteger threadId = new AtomicInteger(0);

        @Override
        public Thread newThread(Runnable r) {
            Thread t = new Thread(this.threadGroup, r);
            t.setDaemon(true);
            t.setName("versiontracker-request-thread-" + this.threadId.incrementAndGet());
            return t;
        }
    };

    public VersionTracker(IVersionStorage versionStorage, IVersionProvider versionProvider, SharedLockCache lockCache) {
        Validate.notNull((Object)versionProvider, (String)"versionProvider must not be NULL", (Object[])new Object[0]);
        Validate.notNull((Object)versionStorage, (String)"versionStorage must not be NULL", (Object[])new Object[0]);
        Validate.notNull((Object)lockCache, (String)"lockCache must not be NULL", (Object[])new Object[0]);
        this.versionProvider = versionProvider;
        this.versionStorage = versionStorage;
        this.lockCache = lockCache;
    }

    @Override
    public Map<Artifact, VersionInfo> getVersionInfo(List<Artifact> artifacts, BiPredicate<VersionInfo, Artifact> requiresUpdate) throws InterruptedException {
        HashMap<Artifact, VersionInfo> resultMap = new HashMap<Artifact, VersionInfo>();
        ZonedDateTime now = ZonedDateTime.now();
        DynamicLatch stopLatch = new DynamicLatch();
        for (Artifact artifact : artifacts) {
            try {
                this.lockCache.doWhileLocked(artifact, () -> {
                    Optional<VersionInfo> result;
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("getVersionInfo(): Looking for " + artifact + " in version storage");
                    }
                    if ((result = this.versionStorage.getVersionInfo(artifact)).isEmpty() || requiresUpdate.test(result.get(), artifact)) {
                        LOG.debug("getVersionInfo(): Got " + (result.isPresent() ? "outdated" : "no") + " metadata for " + artifact + " yet,fetching it");
                        this.updateArtifactFromServer(artifact, result.orElse(null), resultMap, stopLatch, now);
                    } else {
                        LOG.debug("getVersionInfo(): [from storage] " + result.get());
                        Map map = resultMap;
                        synchronized (map) {
                            resultMap.put(artifact, result.get().copy());
                        }
                        this.versionStorage.updateLastRequestDate(artifact, now);
                    }
                });
            }
            catch (InterruptedException e) {
                LOG.error("getVersionInfo(): Caught unexpected exception " + e.getMessage() + " while handling " + artifact, (Throwable)e);
                throw e;
            }
            catch (Exception e) {
                LOG.error("getVersionInfo(): Caught unexpected exception " + e.getMessage() + " while handling " + artifact, (Throwable)e);
                throw new RuntimeException("Uncaught exception " + e.getMessage() + " while handling " + artifact, e);
            }
        }
        stopLatch.await();
        return resultMap;
    }

    @Override
    public VersionInfo forceUpdate(String groupId, String artifactId) throws InterruptedException {
        Validate.notBlank((CharSequence)groupId, (String)"groupId must not be null or blank", (Object[])new Object[0]);
        Validate.notBlank((CharSequence)artifactId, (String)"artifactId must not be null or blank", (Object[])new Object[0]);
        Artifact a = new Artifact();
        a.groupId = groupId;
        a.artifactId = artifactId;
        LOG.info("forceUpdate(): Forced update of " + a);
        DynamicLatch updateFinished = new DynamicLatch();
        HashMap<Artifact, VersionInfo> map = new HashMap<Artifact, VersionInfo>();
        this.updateArtifactFromServer(a, null, map, updateFinished, ZonedDateTime.now());
        updateFinished.await();
        return (VersionInfo)map.get(a);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateArtifactFromServer(Artifact artifact, VersionInfo existing, Map<Artifact, VersionInfo> resultMap, DynamicLatch stopLatch, ZonedDateTime now) {
        stopLatch.inc();
        boolean submitted = false;
        try {
            if (LOG.isDebugEnabled()) {
                LOG.debug("updateArtifact(): About to submit task for " + artifact);
            }
            this.submit(() -> {
                try {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("updateArtifact(): Waiting to lock " + artifact);
                    }
                    this.lockCache.doWhileLocked(artifact, () -> {
                        VersionInfo newInfo;
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("updateArtifact(): Got lock for " + artifact);
                        }
                        if (existing != null) {
                            if (LOG.isDebugEnabled()) {
                                LOG.debug("updateArtifact(): [outdated] Trying to update metadata for " + artifact);
                            }
                            newInfo = existing;
                        } else {
                            if (LOG.isDebugEnabled()) {
                                LOG.debug("updateArtifact(): [missing] Trying to update metadata for " + artifact);
                            }
                            newInfo = new VersionInfo();
                            newInfo.creationDate = now;
                            newInfo.artifact = artifact;
                        }
                        newInfo.lastRequestDate = now;
                        Map map = resultMap;
                        synchronized (map) {
                            resultMap.put(artifact, newInfo);
                        }
                        try {
                            this.versionProvider.update(newInfo, true);
                        }
                        catch (Exception e) {
                            if (LOG.isDebugEnabled()) {
                                LOG.error("updateArtifact(): Caught " + e.getMessage() + " while updating " + artifact, (Throwable)e);
                            } else {
                                LOG.error("updateArtifact(): Caught " + e.getMessage() + " while updating " + artifact + ": " + e.getMessage());
                            }
                        }
                        finally {
                            this.versionStorage.saveOrUpdate(newInfo);
                        }
                    });
                }
                catch (Throwable e) {
                    LOG.error("updateArtifact(): Caught " + e.getMessage() + " while updating " + artifact, e);
                    if (e instanceof Error) {
                        Error t = (Error)e;
                        throw t;
                    }
                }
                finally {
                    stopLatch.dec();
                }
            });
            if (LOG.isDebugEnabled()) {
                LOG.debug("updateArtifact(): Submitted task for " + artifact);
            }
            submitted = true;
        }
        finally {
            if (!submitted) {
                stopLatch.dec();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void submit(Runnable r) {
        Object object = this.THREAD_POOL_LOCK;
        synchronized (object) {
            if (this.threadPool == null) {
                LOG.info("setMaxConcurrentThreads(): Using " + this.maxConcurrentThreads + " threads to retrieve artifact metadata.");
                ArrayBlockingQueue<Runnable> workingQueue = new ArrayBlockingQueue<Runnable>(200);
                this.threadPool = new ThreadPoolExecutor(this.maxConcurrentThreads, this.maxConcurrentThreads, 60L, TimeUnit.SECONDS, workingQueue, this.threadFactory, new ThreadPoolExecutor.CallerRunsPolicy());
            }
            this.threadPool.submit(r);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void setMaxConcurrentThreads(int maxConcurrentThreads) {
        if (maxConcurrentThreads < 1) {
            throw new IllegalArgumentException("maxConcurrentThreads needs to be >= 1");
        }
        Object object = this.THREAD_POOL_LOCK;
        synchronized (object) {
            boolean poolChanged = this.maxConcurrentThreads != maxConcurrentThreads;
            this.maxConcurrentThreads = maxConcurrentThreads;
            if (poolChanged && this.threadPool != null) {
                this.threadPool.shutdown();
                this.threadPool = null;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public int getMaxConcurrentThreads() {
        Object object = this.THREAD_POOL_LOCK;
        synchronized (object) {
            return this.maxConcurrentThreads;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        Object object = this.THREAD_POOL_LOCK;
        synchronized (object) {
            if (this.threadPool != null) {
                LOG.debug("close(): Shutting down thread pool");
                this.threadPool.shutdown();
                this.threadPool = null;
            }
        }
    }

    @Override
    public IVersionStorage getStorage() {
        return this.versionStorage;
    }

    @Override
    public IVersionProvider getVersionProvider() {
        return this.versionProvider;
    }

    protected static final class DynamicLatch {
        private int count = 0;

        protected DynamicLatch() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void inc() {
            DynamicLatch dynamicLatch = this;
            synchronized (dynamicLatch) {
                ++this.count;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void dec() {
            DynamicLatch dynamicLatch = this;
            synchronized (dynamicLatch) {
                if (this.count == 0) {
                    LOG.error("dec(): Internal error, count < 0");
                    throw new IllegalStateException("count < 0 ?");
                }
                --this.count;
                this.notifyAll();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void await() throws InterruptedException {
            DynamicLatch dynamicLatch = this;
            synchronized (dynamicLatch) {
                while (this.count > 0) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("await(): Waiting for " + this.count + " threads to finish");
                    }
                    this.wait();
                }
            }
        }
    }
}

