package de.clued_up.voicecontrols.voiceform;

import de.clued_up.commons.data.VoiceForm;
import de.clued_up.commons.data.VoiceFormResult;
import de.clued_up.voicecontrols.services.VoiceFormService;

import android.app.ActivityManager;
import android.app.ActivityManager.RunningServiceInfo;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.util.Log;


/**
 * helper class to build and run a VoiceForm
 *
 */
public class VoiceFormBuilder
{
	protected static final String TAG = VoiceFormBuilder.class.toString();

	protected final Context m_Context;
    protected final VoiceFormListener m_VoiceFormListener;
    protected final VoiceForm m_VoiceForm;
	
	private BroadcastReceiver m_VoiceFormReceiver;

    protected Class<?> getServiceClass() {
        return VoiceFormService.class;
    }
	
	/**
	 * Helper Class to create and run a VoiceForm
	 * 
	 * @param context the current Android context (Activity or App)
	 * @param strKey an identifier that must be unique among any running Form running at the device at this time!
	 * @param voiceForm contains the content configuration of the voice form
	 * @param listener implement to get the entered data and work with it
	 */
	public VoiceFormBuilder(Context context, String strKey, VoiceForm voiceForm, VoiceFormListener listener)
	{
		m_Context = context;
		m_VoiceFormListener = listener;
        m_VoiceForm = voiceForm;
	}
	
	/** starts the service that takes over control, shows all the cards, receives the input and so on */
	public void start()
	{
		Log.d(TAG, "start voice form");
	
		//check whether the service is already running
		if (isServiceRunning(getServiceClass()))
		{
			m_Context.stopService(new Intent(m_Context, getServiceClass()));

		}
		
		//before we start, we need to register a receiver
		m_VoiceFormReceiver = new BroadcastReceiver()
		{
			@Override
			public void onReceive(Context context, Intent intent)
			{
				Log.d(TAG, "onReceive");
				
				final int iReturnCode = intent.getIntExtra(VoiceFormService.RETURNCODE, VoiceFormService.VFE_RECEIVEREXCEPTION);
				
				//check if the message indicates that the VoiceForm is done, ...
				if (VoiceFormService.isFormFinished(iReturnCode))
				{
					//unregister the broadcast receiver, since the form is done... 
					m_Context.unregisterReceiver(m_VoiceFormReceiver);
					
					Log.d(TAG, "stop service");
					m_Context.stopService(new Intent(m_Context,getServiceClass()));
					
					if (VoiceFormService.isCompletedSuccessfully(iReturnCode))
						m_VoiceFormListener.onVoiceFormCompleted(
                                VoiceFormResult.parseFromString(
                                        intent.getStringExtra(VoiceFormService.VOICEFORMRESULT)
                                )
                        );
					else
						m_VoiceFormListener.onVoiceFormCanceled(iReturnCode);
				}
				else
				{
					//TODO react on status updating broadcasts...
                    Log.d(TAG, "VoiceForm is NOT yet finished");
				}
				
				
				
			}
			
		};
		
		//setup the receiver to get notified of the form's status and its result 
		final IntentFilter filter = new IntentFilter();
		filter.addAction(VoiceFormService.ACTION_VOICEFORM);
		m_Context.registerReceiver(m_VoiceFormReceiver, filter);
				
		//start the service that controls the form
		final Intent intent = new Intent(m_Context, getServiceClass());
		
		intent.putExtra(VoiceFormService.VOICEFORM, m_VoiceForm.parseToString());
		m_Context.startService(intent);
	}
	
	protected boolean isServiceRunning(Class<?> serviceClass)
	{
	    ActivityManager manager = (ActivityManager) m_Context.getSystemService(Context.ACTIVITY_SERVICE);
	    for (RunningServiceInfo service : manager.getRunningServices(Integer.MAX_VALUE)) {
	        if (serviceClass.getName().equals(service.service.getClassName())) {
	            return true;
	        }
	    }

	    return false;
	}
}
