package de.clued_up.voicecontrols.services;

import android.app.Notification;
import android.app.PendingIntent;
import android.app.Service;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.AsyncTask;
import android.os.IBinder;
import android.support.v4.app.NotificationManagerCompat;
import android.util.Log;

import android.support.v4.app.NotificationCompat;
import android.support.v4.app.RemoteInput;
import android.support.v4.app.NotificationCompat.WearableExtender;

import de.clued_up.commons.data.VoiceAttribute;
import de.clued_up.commons.data.VoiceAttributeResult;
import de.clued_up.commons.data.VoiceForm;
import de.clued_up.commons.data.VoiceFormResult;
import de.clued_up.voicecontrols.Configuration;
import de.clued_up.voicecontrols.async.AsyncTaskSupporter;
import de.clued_up.voicecontrols.async.AsyncTaskToastSupporter;
import de.clued_up.voicecontrols.async.AsyncTaskTracker;
import de.clued_up.voicecontrols.async.ProgressAnimator;
import de.clued_up.voicecontrols.async.TrackedAsyncTask;
import de.clued_up.voicecontrols.backend.apiVoiceForm.ApiVoiceForm;
import de.clued_up.voicecontrols.backend.apiVoiceForm.model.VoiceFormBean;
import de.clued_up.voicecontrols.voicereply.ReplyActivity;
import de.clued_up.voicecontrols.voicereply.VoiceReplyIntentReceiver;
import de.clued_up.voicecontrols.voicereply.VoiceReplyListener;
import com.google.api.client.extensions.android.http.AndroidHttp;
import com.google.api.client.extensions.android.json.AndroidJsonFactory;
import com.google.api.client.googleapis.services.AbstractGoogleClientRequest;
import com.google.api.client.googleapis.services.GoogleClientRequestInitializer;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * This service handles the VoiceForm process. Waits for input, and starts enquiries.
 * It communicates with the outside world via broadcasts
 *
 * TODO branded view when entering the text
 * --> http://stackoverflow.com/questions/23091584/android-custom-voice-recognition-gui-dialog
 *
 * TODO allow the user to specify everything in a xml-file
 * --> http://stackoverflow.com/questions/21706665/how-can-i-parse-an-android-layout-xml-file-to-create-a-replica-for-help-pages
 *
 */
public class VoiceFormService extends Service implements AsyncTaskToastSupporter, AsyncTaskSupporter
{
    /** ################################### Data Interchange to VoiceFormBuilder #################################################### */

    /** key of the voiceform bundle passed to the service */
    public static final String VOICEFORM = "com.cluedup.voicecontrols.voiceform.VoiceFormBuilder.VOICEFORM";

    /** key of the voiceform bundle passed to the service */
    public static final String VOICEFORMRESULT = "com.cluedup.voicecontrols.voiceform.VoiceFormBuilder.VOICEFORMRESULT";


    /** key of the attributeslist bundle passed to the service */
    public static final String ATTRIBUTESLISTBUNDLE = "com.cluedup.voicecontrols.voiceform.VoiceFormBuilder.ATTRIBUTESLISTBUNDLE";

    /** key for the broadcast receiver that listens to the voice form */
    public static final String ACTION_VOICEFORM = "com.cluedup.voicecontrols.voiceform.VoiceFormBuilder.ACTION_VOICEFORM";

    /** key of the commandaction */
    //public static final String COMMANDACTION = "com.cluedup.voicecontrols.voiceform.VoiceFormBuilder.ACTION_VOICEFORM";


    /** ################################### Data Interchange to VoiceFormBuilder #################################################### */

    /** key of the voiceformresult bundle passed back from the service */
    public static final String VOICEFORMRESULTBUNDLE = "com.cluedup.voicecontrols.voiceform.VoiceFormBuilder.VOICEFORMRESULTBUNDLE";

    /** ################################### ErrorCodes/ Returncodes #################################################### */

    /** key of the returncode, could be an errorcode */
    public static final String RETURNCODE = "com.cluedup.voicecontrols.voiceform.VoiceFormBuilder.RETURNCODE";

    /** voice form error: the user did not complete the speech input */
    public static final int VF_COMPLETED = 0;

    /** voice form error: the user did not complete the speech input */
    public static final int VFE_SPEECHCANCELED = -1;

    /** voice form error: calling the activity analysis failed */
    public static final int VFE_ACTIVITYANALYSIS = -2;

    /** voice form error: the thread was aborted */
    public static final int VFE_TECHNICALABORT = -3;

    /** voice form error: the thread could not wait */
    public static final int VFE_INTERRUPTEXCEPTION = -4;

    /** voice form error: is set by the voiceformBuilder when it does not receive a returncode */
    public static final int VFE_RECEIVEREXCEPTION = -5;

    /** voice form error: is set by the server: when the user exceed the limit of requests */
    public static final int VFE_LIMITEXCEEDED = -6;

    /** convenience function to check whether a returncode indicates that the form has finished or whether it continues*/
    public static boolean isFormFinished(int returncode){return returncode <= 0;}

    /** convenience function to check whether the form has finished successfully */
    public static boolean isCompletedSuccessfully(int returncode){return returncode == 0;}

    /** key for the task that runs the form */
    protected static final String KEY_RUNFORMTASK = "com.cluedup.voicecontrols.VoiceControlService.KEY_RUNFORMTASK";

    /** ####################################################################################### */

    private static final int NOTIFICATIONID = 27101981;

    protected static final String TAG = VoiceFormService.class.toString();
    
    private static ApiVoiceForm apiVoiceForm = null;

    private boolean m_bWaitingForInput = false;
    private String m_strReceivedInput = null;
    private NotificationManagerCompat notificationManager;

    private String m_strAppId;

    /** whenever the voice form requires input, a receives the answer using this broadcast receiver */
    protected VoiceReplyIntentReceiver m_VoiceReplyReceiver;

    /** required to run/track TrackedAsyncTasks */
    private AsyncTaskTracker m_AsyncTaskTracker;
    public AsyncTaskTracker getAsyncTaskTracker() {return m_AsyncTaskTracker;}


    /** the voice form must run in an async task, because it must wait for user input from the recognizer input */
    protected TrackedAsyncTask m_taskRunForm;

    /** counter for the input requests */
    protected int iCntInput = 0;




    /** the received data for configurating the voiceform*/
    private VoiceForm m_VoiceForm;



    @Override
    public IBinder onBind(Intent arg0) {return null;}
    
    @Override
    public void onCreate() {

        super.onCreate();

        Log.d(TAG, "onCreate");

        m_AsyncTaskTracker = new AsyncTaskTracker(this);

        m_taskRunForm = new TrackedAsyncTask(this, getAsyncTaskTracker(), KEY_RUNFORMTASK)
        {

            @Override
            public void abortTask()
            {
                //when we should abort the task,
                m_taskRunForm.cancel(true);
                onVoiceControlCanceled(VFE_TECHNICALABORT);
            }

            @Override
            protected Boolean doInBackground(Void... params)
            {
                Log.d(TAG, "doInBackground");

                //start the recursive process, we have no result and no enquiry at the beginning
                VoiceFormResult result = processStep(null,
                        m_VoiceForm.getEnquiry(),
                        m_VoiceForm.getEnquiryActionIconResource(),
                        m_VoiceForm.getEnquiryAction());

                //return the result
                if (isLimitExceeded(result))
                    onVoiceControlCanceled(VFE_LIMITEXCEEDED);
                else
                    onVoiceControlCompleted(result);

                return null;
            }


        };

        m_strAppId = getAppId();

        notificationManager = NotificationManagerCompat.from(this);

        //create the receiver and register it
        m_VoiceReplyReceiver = new VoiceReplyIntentReceiver()
        {
            @Override
            protected void stopService()
            {
                Log.d(TAG, "stop service");
                VoiceFormService.this.stopSelf();
            }
        }.register(this);

        Log.d(TAG, "onCreate");

    }

    private VoiceFormResult processStep(VoiceFormResult oldResult,
                                   String strOpenEnquiry,
                                   int iEnquiryActionIcon,
                                   String strEnquiryAction)
    {
        Log.d(TAG, "processStep, with enquiry: "+strOpenEnquiry);

        //ad the input to the form
        m_VoiceForm.setInput(
            getVoiceControlInput(

                //getIcon
                m_VoiceForm.getIconResource(),

                //get the label of the VoiceForm
                m_VoiceForm.getLabel(),

                //get the enquiry for collecting the voice input
                strOpenEnquiry,

                //the icon of the reply-action
                iEnquiryActionIcon,

                //the text of the reply-action
                strEnquiryAction,

                //get the background of the VoiceForm
                BitmapFactory.decodeResource(getResources(), m_VoiceForm.getBackgroundImageResource())
            )
        );

        //get a merged result
        VoiceFormResult mergedResult = mergeResults(oldResult,
                //get a result of the current step
                callBackend()
        );

        //OpenEnquiryObject newOpenEnquiryObject = getOpenEnquiryObject();

        //is the voice form completed with the result, or do we have an open enquiry that we need to answer?
        //VoiceAttribute attributeBean = setOpenContextAttribute(m_VoiceForm, mergedResult);

        //when we do not have a requested attribute, we do nothing but return null
        if (mergedResult == null || mergedResult.getRequestedAttribute() == null)
            return null;

        //when there is a requested attribute...
        Log.d(TAG, "requested attribute: "+mergedResult.getRequestedAttribute());

        VoiceAttribute attribute = mergedResult.getRequestedAttribute();
        //... set the context attribute in the voiceForm
        m_VoiceForm.setContextAttribute(attribute);



        if (attribute == null) {
            if (isCompleted(mergedResult))
                return mergedResult;
            else
                return processStep(null,
                        m_VoiceForm.getEnquiry(),
                        m_VoiceForm.getEnquiryActionIconResource(),
                        m_VoiceForm.getEnquiryAction());

        }
        else
            //otherwise we do the next step... with regard to the OpenEnquiryObject
            return processStep(mergedResult,
                    attribute.getEnquiry(),
                    attribute.getEnquiryActionIconResource(),
                    attribute.getEnquiryAction());

    }


    @Override
    public int onStartCommand(Intent intent, int flags, int startId)
    {
        Log.d(TAG, "Received start id " + startId + ": " + intent);

        m_VoiceForm = VoiceForm.parseFromString(intent.getStringExtra(VOICEFORM));
        //m_listAttributeBeans = createAttributeBeansListFromBundle(intent.getBundleExtra(VoiceControlService.ATTRIBUTESLISTBUNDLE));

        //start the voice form
        m_taskRunForm.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR);

        // We want this service to continue running until it is explicitly stopped, so return sticky.
        return START_STICKY;
    }

    protected void onVoiceControlCanceled(int iErrorCode)
    {
        Log.d(TAG, "onVoiceControlCanceled");

        Intent intent = new Intent();
        intent.setAction(ACTION_VOICEFORM);
        intent.putExtra(RETURNCODE, iErrorCode);
        sendBroadcast(intent);
    }

    protected void onVoiceControlCompleted(VoiceFormResult result)
    {
        Log.d(TAG, "onVoiceControlCompleted");
        Intent intent = new Intent();
        intent.setAction(ACTION_VOICEFORM);
        intent.putExtra(RETURNCODE, VF_COMPLETED);
        intent.putExtra(VOICEFORMRESULT, result.parseToString());
        sendBroadcast(intent);
    }


    public String getAppId() {
        PackageManager lPackageManager = getPackageManager();
        ApplicationInfo lApplicationInfo = null;
        try {
            lApplicationInfo = lPackageManager.getApplicationInfo(getApplicationInfo().packageName, 0);
        } catch (final PackageManager.NameNotFoundException e) {
        }

        return (String) (getApplicationContext().getPackageName() +
                lApplicationInfo != null ? lPackageManager.getApplicationLabel(lApplicationInfo) : "Unknown");
    }

    @Override
    public void onDestroy()
    {
    	Log.d(TAG, "onDestroy");

        //close all old notifications
        notificationManager.cancel(NOTIFICATIONID);

        // Do not forget to unregister the receiver!!!
        unregisterReceiver(m_VoiceReplyReceiver);


        super.onDestroy();
    }

    protected String getVoiceControlInput(int iIcon,
                                          String strLabel,
                                          String strEnquiry,
                                          int iEnquiryActionIcon,
                                          String strEnquiryAction,
                                          Bitmap bmpBackground)
    {
        Log.d(TAG, "getVoiceControlInput, with enquiry: "+strEnquiry);

        new VoiceReplyListener(m_VoiceReplyReceiver)
        {
            @Override
            public void onVoiceReplyReceived(String strReply)
            {
                Log.d(TAG, "received reply: "+strReply);

                //dismiss the notification
                notificationManager.cancel(NOTIFICATIONID);

                m_strReceivedInput = strReply;
                m_bWaitingForInput = false;

            }

            @Override
            public void onVoiceReplyFailed(){
                Log.d(TAG, "onVoiceReplyFailed");

                //not required since the service is stopped anyway
                //doing it would result in a further request that is not wanted
                //m_bWaitingForInput = false;
            }

        };

        m_bWaitingForInput = true;



        //setup the remote input
        RemoteInput remoteInput = new RemoteInput.Builder(VoiceReplyIntentReceiver.EXTRA_VOICE_REPLY)
                .setLabel(strEnquiryAction)
                .build();

        // Create an intent for the reply action
        Intent replyIntent = new Intent(this, ReplyActivity.class);
        PendingIntent replyPendingIntent = PendingIntent.getActivity(this, 0, replyIntent, PendingIntent.FLAG_UPDATE_CURRENT);


        // Create an intent for the dismissal
        Intent deleteIntent = new Intent();
        deleteIntent.setAction(VoiceReplyIntentReceiver.ACTION_VOICE_REPLY);
        deleteIntent.putExtra(VoiceReplyIntentReceiver.EXTRA_VOICE_CANCELLED, true);
        PendingIntent deletePendingIntent = PendingIntent.getBroadcast(this, 1, deleteIntent, PendingIntent.FLAG_UPDATE_CURRENT);

        // Create the reply action and add the remote input NotificationCompat.
        NotificationCompat.Action action = new NotificationCompat.Action.Builder(iEnquiryActionIcon, strEnquiryAction, replyPendingIntent)
                .addRemoteInput(remoteInput)
                .build();

        // Build the notification and add the action via WearableExtender
        Notification notification = new NotificationCompat.Builder(this)
                .setSmallIcon(iIcon)
                .setContentTitle(strLabel)
                .setContentText(strEnquiry)
                .setDeleteIntent(deletePendingIntent)
                .extend(new WearableExtender()
                    .addAction(action)
                    .setBackground(bmpBackground)
                )
                .build();

        // Issue the notification
        notificationManager.notify(NOTIFICATIONID, notification);

        Log.d(TAG, "notification should be issued!");

        //wait until we get a response
        while (m_bWaitingForInput)
        {
            try
            {
                Thread.sleep(300);
            }
            catch (InterruptedException e)
            {
                e.printStackTrace();
            }
        }



        return m_strReceivedInput;
    }

	protected VoiceFormResult callBackend(){

        Log.d(TAG, "call backend");
		iCntInput++;

        // Only do this once
        if(apiVoiceForm == null)
        {
            if (Configuration.doConnectToLocalHost()) {
                ApiVoiceForm.Builder builder = new ApiVoiceForm.Builder(AndroidHttp.newCompatibleTransport(),
                        new AndroidJsonFactory(), null)
                        .setRootUrl("http://10.0.2.2:8080/_ah/api/")
                        .setGoogleClientRequestInitializer(new GoogleClientRequestInitializer() {
                            @Override
                            public void initialize(AbstractGoogleClientRequest<?> abstractGoogleClientRequest) throws IOException {
                                abstractGoogleClientRequest.setDisableGZipContent(true);
                            }
                        });

                apiVoiceForm = builder.build();
            }
            else
            {
                Log.d(TAG, "build the online connection");
                ApiVoiceForm.Builder builder = new ApiVoiceForm.Builder(
                        AndroidHttp.newCompatibleTransport(),
                        new AndroidJsonFactory(),
                        null)
                    .setRootUrl("https://voicecontrols.appspot.com/_ah/api/");
                apiVoiceForm = builder.build();
            }

        }


        // ******************************* use when in Android Studio *************************//
        try {
            return VoiceFormResult.parseFromString(
                    apiVoiceForm.analyzeVoiceForm(
                            m_strAppId,
                            new VoiceFormBean().setVoiceFormString(m_VoiceForm.parseToString()))
                            .setStrAccountId("accountId")
                            .execute()
                            .getVoiceFormResult());
        } catch (IOException e) {
            Log.e(TAG, e.getMessage());
            return null;
        }

        
	}

	
	/** create a merged resultBean of the given */
	protected VoiceFormResult mergeResults(VoiceFormResult oldResult, VoiceFormResult currentResult)
	{
        Log.d(TAG, "mergeResults");
		//when we exceeded the limit, we do not need to do anything else but return that we exceeded the request limit
		if (currentResult != null && currentResult.getResultType() == VoiceFormResult.VFR_REQUESTLIMITEXCEEDED)
			return currentResult;
		
		//###################### create the lists of VoiceAttributeResultBundles ##############################################
		ArrayList<VoiceAttributeResult> listOldResults = new ArrayList<VoiceAttributeResult>();
		ArrayList<VoiceAttributeResult> listCurrentResults = new ArrayList<VoiceAttributeResult>();

        if (oldResult != null) {
            for (VoiceAttributeResult attrResult :oldResult.getAttributeResults()) {
                listOldResults.add(attrResult);
            }
        }

        if (currentResult != null) {
            final List<VoiceAttributeResult> listAttributeResults = currentResult.getAttributeResults();
            if (listAttributeResults != null) {
                for (VoiceAttributeResult attrResult :listAttributeResults)
                    listCurrentResults.add(attrResult);
            }
        }

        for (VoiceAttributeResult attribute: listCurrentResults) {
            for (Object strResult: attribute.getResults())
                Log.d(TAG, "attributeresult: "+(String)strResult);
        }

		ArrayList<VoiceAttributeResult> listNewResults = new ArrayList<VoiceAttributeResult>();
		
		//merge the attribute results
		while (!listOldResults.isEmpty())
		{
			VoiceAttributeResult oldAttributeResult = listOldResults.get(0);
			
			//find the corresponding current result
			VoiceAttributeResult currentAttributeResult = null;
			for (int i=0; i<listCurrentResults.size();i++)
			{
				VoiceAttributeResult current = listCurrentResults.get(i);
				if (oldAttributeResult.getName().compareTo(current.getName()) == 0 &&
						oldAttributeResult.getValueType() == current.getValueType())
				{
					currentAttributeResult = current;
					listCurrentResults.remove(i);
					break;
				}
			}
			
			//when we found a corresponding, ... combine both results
			if (currentAttributeResult != null)
			{
				//remove duplicates in the old list
				for (String strCurrentResult :currentAttributeResult.getResults())
				{
					for (int j=0;j<oldAttributeResult.getResults().size(); j++)
					{
                        String strOldResult = oldAttributeResult.getResults().get(j);
							
						//TODO when we have other types, we need to add them here
						//if (oldAttributeResult.getValueType() == VoiceAttribute.VALUETYPE_LIST ||
						//	oldAttributeResult.getValueType() == VoiceAttribute.VALUETYPE_FREETEXT)
						{
							if ((strCurrentResult).compareTo((strOldResult)) == 0)
							{
								oldAttributeResult.getResults().remove(j);
								break;
							}
						}
					}
				}
				
				//add the remaining old results to the current one
				for (String strOldResult: oldAttributeResult.getResults())
					currentAttributeResult.getResults().add(strOldResult);
				
				//put the current result into the new
				listNewResults.add(currentAttributeResult);
			}
			
			//when we did not find a corresponding result
			else
			{
				//just add the result to the new result
				listNewResults.add(oldAttributeResult);
			}
			
			
			listOldResults.remove(0);
		}
		
		//###################### check all existing attributes for (mandatory) completeness ##############################################
		
		boolean bAllMandatoryCompleted = true;
		boolean bAllCompleted = true;
		
		//loop through all attributes
		for (VoiceAttribute voiceAttribute :m_VoiceForm.getAttributes()){

			boolean foundResult = false;
			//search for results:
			for (VoiceAttributeResult result :listNewResults){
				if (voiceAttribute.getName().compareTo(result.getName()) == 0 && result.getResults().size() > 0){
					foundResult = true;
					break;
				}
			}
			
			//when we did not find a result, we're not completed
			if (!foundResult){
				bAllCompleted = false;
				
				//when the incomplete attribute is mandatory, we don't need to continue 
				if (voiceAttribute.getIsMandatory()){
					bAllMandatoryCompleted = false;
					break;
				}
			}
		}
		
		
		//###################### combine the core data of the VoiceFormResult ##############################################
		
		//take the requested attribute of the currentResultBean
		//currentResultBean.getRequestedAttribute();
		
		//get the entered activity
		if (currentResult != null && currentResult.getEnteredActivity() == null)
		{
            if (oldResult != null)
			    currentResult.setEnteredActivity(oldResult.getEnteredActivity());
			
			//when we take the entered activity of the oldResultBean, check whether we should also take number and unit
			if (oldResult != null &&
                    (oldResult.getRequestedNumber() == null || oldResult.getRequestedNumber() != 0))
				currentResult.setRequestedNumber(oldResult.getRequestedNumber());
			
			if (oldResult != null &&
                    (oldResult.getRequestedUnit() == null || oldResult.getRequestedUnit().length() == 0))
				currentResult.setRequestedUnit(oldResult.getRequestedUnit());
		}	

		//when we have no requested number in the current result, take the old one
		if (currentResult != null && oldResult != null &&
                (currentResult.getRequestedNumber() == null || currentResult.getRequestedNumber() <= 0))
			currentResult.setRequestedNumber(oldResult.getRequestedNumber());
		
		//when we have no requested unit in the current result, take the old one
		if (currentResult != null && oldResult != null &&
                (currentResult.getRequestedUnit() == null || currentResult.getRequestedUnit().length() == 0))
			currentResult.setRequestedUnit(oldResult.getRequestedUnit());
		
		//when the current result is not yet completed, we need to recalculate the result
		if (currentResult != null && currentResult.getResultType() != VoiceFormResult.VFR_COMPLETED)
		{
			//do we have an activity?
			if (currentResult.getEnteredActivity() != null)
			{
				//are all mandatory attributes filled?
				if (bAllMandatoryCompleted)
				{
					//are all fields are filled?
					if (bAllCompleted)
						currentResult.setResultType(VoiceFormResult.VFR_COMPLETED);
					else
						currentResult.setResultType(VoiceFormResult.VFR_MANDATORYCOMPLETED);
				}
				else
					//when we have open mandatory attributes
					currentResult.setResultType(VoiceFormResult.VFR_INCOMPLETE);
			}
			else
				//when there is no entered activity, then we have no command
				currentResult.setResultType(VoiceFormResult.VFR_INVALIDCOMMAND);
		}

		return currentResult;
	}



    
	/*@Override protected VoiceFormBean createBeanFromBundle(Bundle bundle){return VoiceFormConfiguration.createVoiceFormBeanFromBundle(bundle);}
	@Override protected Bundle createBundleFromTemplateResultBean(VoiceFormResultBean resultBean){
		return VoiceFormResultBundle.createBundleFromVoiceFormResultBean(resultBean);}
    @Override protected VoiceFormBean addInputToVoiceForm(String s, VoiceFormBean voiceFormBean) {return voiceFormBean.setInput(s);}
	@Override protected String getVoiceFormLabel(VoiceFormBean voiceForm) {return voiceForm.getLabel();}
	@Override protected String getVoiceFormEnquiry(VoiceFormBean voiceForm) {return voiceForm.getEnquiry();}
    @Override protected String getVoiceFormRepeatedEnquiry(VoiceFormBean voiceForm) {return voiceForm.getRepeatedEnquiry(); }

    @Override protected int getVoiceFormIcon(VoiceFormBean voiceForm) {return voiceForm.getIconResource();}
	@Override protected String getVoiceFormEnquiryAction(VoiceFormBean voiceForm){return voiceForm.getEnquiryAction();}
	@Override protected int getVoiceFormEnquiryActionIcon(VoiceFormBean voiceForm){return voiceForm.getEnquiryActionIconResource();}
    @Override protected Bitmap getVoiceFormBackground(VoiceFormBean voiceForm) {
        return BitmapFactory.decodeResource(getResources(), voiceForm.getBackgroundImageResource());}

    @Override protected String getAttributeEnquiryAction(VoiceAttributeConfiguration attributeConfiguration){return attributeConfiguration.getEnquiryAction();}
	@Override protected int getAttributeEnquiryActionIcon(VoiceAttributeConfiguration attributeConfiguration){return attributeConfiguration.getEnquiryActionIconResource();}
	@Override protected String getAttributeEnquiry(VoiceAttributeConfiguration attributeConfiguration){return attributeConfiguration.getEnquiry();}
	*/

    protected boolean isLimitExceeded(VoiceFormResult result){
        return result == null || result.getResultType() == VoiceFormResult.VFR_REQUESTLIMITEXCEEDED;
    }

    protected boolean isCompleted(VoiceFormResult result) {
        Log.d(TAG, "result: "+result.getResultType());
        return result.getResultType() <= VoiceFormResult.VFR_MANDATORYCOMPLETED;
    }

    @Override
    public void setCurrentProgressAnimator(ProgressAnimator animator){}

    @Override
    public void onAsyncTaskStarted(int newcount){}

    @Override
    public void onAsyncTaskStopped(int newcount){}

    @Override
    public void onAsyncTaskUpdated(String strMessage){}

    @Override
    public void showToast(String msg){}

    @Override
    public void showToast(int resourceId) {}
}
