package de.clued_up.voicecontrols.voiceform;

import de.clued_up.voicecontrols.services.CallVoiceControlBackendService;
import de.clued_up.voicecontrols.services.VoiceControlService;
import de.clued_up.voicecontrols.data.VoiceFormConfiguration;

import android.app.ActivityManager;
import android.app.ActivityManager.RunningServiceInfo;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.util.Log;


/**
 * helper class to build and run a VoiceForm
 *
 */
public class VoiceFormBuilder
{
	protected static final String TAG = VoiceFormBuilder.class.toString();

    /*********** MUST match VoiceAttribute values!!! ***************/
    /** any text is accepted as value*/
    public static final int VALUETYPE_FREETEXT = 0;

    /** accepts only values that are specified in the voiceform */
    public static final int VALUETYPE_LIST = 1;

    /** accepts only time values */
    public static final int VALUETYPE_TIME = 2;

	
	private final Context m_Context;
	private final VoiceFormListener m_VoiceFormListener;
	
	private final VoiceFormConfiguration m_VoiceFormBundle;
	
	private BroadcastReceiver m_VoiceFormReceiver;
	
	/**
	 * Helper Class to create and run a VoiceForm
	 * 
	 * @param context the current Android context (Activity or App)
	 * @param strKey an identifier that must be unique among any running Form running at the device at this time!
	 * @param voiceFormBundle contains the content configuration of the voice form
	 * @param listener implement to get the entered data and work with it
	 */
	public VoiceFormBuilder(Context context, String strKey, VoiceFormConfiguration voiceFormBundle, VoiceFormListener listener)
	{
		m_Context = context;
		m_VoiceFormListener = listener;
		m_VoiceFormBundle = voiceFormBundle;
	}
	
	/** starts the service that takes over control, shows all the cards, receives the input and so on */
	public void start()
	{
		Log.d(TAG, "start voice form");
	
		//check whether the service is already running
		if (isServiceRunning(CallVoiceControlBackendService.class))
		{
			m_Context.stopService(new Intent(m_Context, CallVoiceControlBackendService.class));
		}
		
		//before we start, we need to register a receiver
		m_VoiceFormReceiver = new BroadcastReceiver()
		{
			@Override
			public void onReceive(Context context, Intent intent)
			{
				Log.d(TAG, "onReceive");
				
				final int iReturnCode = intent.getIntExtra(VoiceControlService.RETURNCODE, VoiceControlService.VFE_RECEIVEREXCEPTION);
				
				//check if the message indicates that the VoiceForm is done, ...
				if (VoiceControlService.isFormFinished(iReturnCode))
				{
					//unregister the broadcast receiver, since the form is done... 
					m_Context.unregisterReceiver(m_VoiceFormReceiver);
					
					Log.d(TAG, "stop service");
					m_Context.stopService(new Intent(m_Context,CallVoiceControlBackendService.class));
					
					if (VoiceControlService.isCompletedSuccessfully(iReturnCode))
						m_VoiceFormListener.onVoiceFormCompleted(intent.getBundleExtra(VoiceControlService.VOICEFORMRESULTBUNDLE));
					else
						m_VoiceFormListener.onVoiceFormCanceled(iReturnCode);
				}
				else
				{
					//TODO react on status updating broadcasts... 
				}
				
				
				
			}
			
		};
		
		//setup the receiver to get notified of the form's status and its result 
		final IntentFilter filter = new IntentFilter();
		filter.addAction(VoiceControlService.ACTION_VOICEFORM);
		m_Context.registerReceiver(m_VoiceFormReceiver, filter);
				
		//start the service that controls the form
		final Intent intent = new Intent(m_Context, CallVoiceControlBackendService.class);
		
		intent.putExtra(VoiceControlService.VOICEFORMBUNDLE, m_VoiceFormBundle.toBundle());
		m_Context.startService(intent);
	}
	
	private boolean isServiceRunning(Class<?> serviceClass)
	{
	    ActivityManager manager = (ActivityManager) m_Context.getSystemService(Context.ACTIVITY_SERVICE);
	    for (RunningServiceInfo service : manager.getRunningServices(Integer.MAX_VALUE)) {
	        if (serviceClass.getName().equals(service.service.getClassName())) {
	            return true;
	        }
	    }
	    return false;
	}
}
