package de.clued_up.voicecontrols.services;

import de.clued_up.voicecontrols.async.AsyncTaskSupporter;
import de.clued_up.voicecontrols.async.AsyncTaskToastSupporter;
import de.clued_up.voicecontrols.async.AsyncTaskTracker;
import de.clued_up.voicecontrols.async.ProgressAnimator;
import de.clued_up.voicecontrols.async.TrackedAsyncTask;

import android.app.Notification;
import android.app.PendingIntent;
import android.app.Service;
import android.content.Intent;
import android.graphics.Bitmap;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.IBinder;
import android.support.v4.app.NotificationCompat;
import android.support.v4.app.NotificationManagerCompat;
import android.support.v4.app.RemoteInput;
import android.support.v4.app.NotificationCompat.WearableExtender;
import android.util.Log;

/**
 * This service handles the VoiceForm process. Waits for input, and starts enquiries.
 * It communicates with the outside world via broadcasts
 * 
 * TODO branded view when entering the text
 * --> http://stackoverflow.com/questions/23091584/android-custom-voice-recognition-gui-dialog
 * 
 * TODO allow the user to specify everything in a xml-file
 * --> http://stackoverflow.com/questions/21706665/how-can-i-parse-an-android-layout-xml-file-to-create-a-replica-for-help-pages
 *
 */
public abstract class VoiceControlService<Bean, ResultBean, AttributeBean>
        extends Service
        implements AsyncTaskToastSupporter, AsyncTaskSupporter
{
	protected static final String TAG = VoiceControlService.class.toString();
	
	/** ################################### Data Interchange to VoiceFormBuilder #################################################### */
	
	/** key of the voiceform bundle passed to the service */
	public static final String VOICEFORMBUNDLE = "com.cluedup.voicecontrols.voiceform.VoiceFormBuilder.VOICEFORMBUNDLE";
	
	/** key of the attributeslist bundle passed to the service */
	public static final String ATTRIBUTESLISTBUNDLE = "com.cluedup.voicecontrols.voiceform.VoiceFormBuilder.ATTRIBUTESLISTBUNDLE";
	
	/** key for the broadcast receiver that listens to the voice form */
	public static final String ACTION_VOICEFORM = "com.cluedup.voicecontrols.voiceform.VoiceFormBuilder.ACTION_VOICEFORM";
	
	/** key of the commandaction */
	//public static final String COMMANDACTION = "com.cluedup.voicecontrols.voiceform.VoiceFormBuilder.ACTION_VOICEFORM";
	
	
	/** ################################### Data Interchange to VoiceFormBuilder #################################################### */
	
	/** key of the voiceformresult bundle passed back from the service */
	public static final String VOICEFORMRESULTBUNDLE = "com.cluedup.voicecontrols.voiceform.VoiceFormBuilder.VOICEFORMRESULTBUNDLE";
	
	/** ################################### ErrorCodes/ Returncodes #################################################### */
	
	/** key of the returncode, could be an errorcode */
	public static final String RETURNCODE = "com.cluedup.voicecontrols.voiceform.VoiceFormBuilder.RETURNCODE";
	
	/** voice form error: the user did not complete the speech input */
	public static final int VF_COMPLETED = 0;
	
	/** voice form error: the user did not complete the speech input */
	public static final int VFE_SPEECHCANCELED = -1;
	
	/** voice form error: calling the activity analysis failed */
	public static final int VFE_ACTIVITYANALYSIS = -2;
	
	/** voice form error: the thread was aborted */
	public static final int VFE_TECHNICALABORT = -3;
	
	/** voice form error: the thread could not wait */
	public static final int VFE_INTERRUPTEXCEPTION = -4;
	
	/** voice form error: is set by the voiceformBuilder when it does not receive a returncode */
	public static final int VFE_RECEIVEREXCEPTION = -5;
	
	/** voice form error: is set by the server: when the user exceed the limit of requests */
	public static final int VFE_LIMITEXCEEDED = -6;
	
	/** convenience function to check whether a returncode indicates that the form has finished or whether it continues*/
	public static boolean isFormFinished(int returncode){return returncode <= 0;}
	
	/** convenience function to check whether the form has finished successfully */
	public static boolean isCompletedSuccessfully(int returncode){return returncode == 0;}
	
	/** ####################################################################################### */
	
	
	/** key for the activity analysis call*/
	protected static final String KEY_ACTIVITYANALYSIS = "com.cluedup.voicecontrols.VoiceControlActivity.KEY_ACTIVITYANALYSIS";
	
	/** key for the task that runs the form */
	protected static final String KEY_RUNFORMTASK = "com.cluedup.voicecontrols.VoiceControlService.KEY_RUNFORMTASK";
	
	/** ####################################################################################### */
	
	
	/** required to run/track TrackedAsyncTasks */
	private AsyncTaskTracker m_AsyncTaskTracker;
	public AsyncTaskTracker getAsyncTaskTracker() {return m_AsyncTaskTracker;}
	
	
	/** the voice form must run in an async task, because it must wait for user input from the recognizer input */
	protected TrackedAsyncTask m_taskRunForm;
	
	
	/** the received data for configurating the voiceform*/
	private Bean m_VoiceFormBean;
	protected abstract Bean createBeanFromBundle(Bundle bundle);
	
	/** get the list of attributes of the voiceform, so that we can separately put them into the request for the server 
    private ArrayList<OpenEnquiryObject> m_listAttributeBeans;
	protected abstract ArrayList<OpenEnquiryObject> createAttributeBeansListFromBundle(Bundle bundle);*/
    
	
	/** the data submitted when the form is completed */
	protected abstract Bundle createBundleFromTemplateResultBean(ResultBean result);
	
	/* implement with the call of the backend that analyzes ...
	 * @param strInput ...the user input ...
	 * @param bean ... into the specified VOiceForm ...
	 * @param listEnquiryObjects specifies the list of Attributes of the VoiceForm
	 * @param requestedOpenEnquiryObject ... with regard to the requested EnquiryObject (attribute) that was asked for when collecting the input*/
    protected abstract ResultBean callBackend(//String strInput,
    		Bean bean
    		//,ArrayList<OpenEnquiryObject> listEnquiryObjects, OpenEnquiryObject requestedOpenEnquiryObject
    		);
    
    
    /** implement with the code that takes user input, the openEnquiry is the Attribute */
    protected abstract String getVoiceControlInput(
            int iIcon,
            String strLabel,
            String strEnquiry,
            int iEnquiryActionIcon,
            String strEnquiryAction,
            Bitmap bmpBackground);
    
    
    /** get the enquiry string from an OpenEnquiryObject, i.e. get the Enquiry from an Attribute 
    protected abstract String getEnquiry(OpenEnquiryObject openEnquiry);*/
    
    /** identify an open enquiry object (unanswered attribute) in the VoiceForm that the result did not answer, i.e. form.getRequestedAttribute! 
    protected abstract OpenEnquiryObject getOpenEnquiryObject(ResultBean result);*/
    
    /** create a merged resultBean of the given  
     * @param voiceFormBean */
    protected abstract ResultBean mergeResults(Bean voiceFormBean, ResultBean oldResult, ResultBean currentResult);
    
    /** if there is one: sets the OpenContextAttribute in the Voice Form and returns the enquiry, if no openEnquiry exists, it returns null */
    protected abstract AttributeBean setOpenContextAttribute(Bean voiceFormBean, ResultBean mergedResult);
    
    /** put the input into the voiceformbean*/
    protected abstract Bean addInputToVoiceForm(String strInput, Bean voiceForm);
    
    /** get the label (title) of the voiceForm */
    protected abstract String getVoiceFormLabel(Bean voiceForm);
    
    /** get the very first enquiry of the voiceForm */
    protected abstract String getVoiceFormEnquiry(Bean voiceForm);

    /** get the repetition of the very first enquiry of the voiceForm */
    protected abstract String getVoiceFormRepeatedEnquiry(Bean voiceForm);
    
    /** get the icon of the voice form */
    protected abstract int getVoiceFormIcon(Bean voiceForm);
    
    /** get the icon shown as action in the first enquiry */
    protected abstract String getVoiceFormEnquiryAction(Bean m_VoiceFormBean);

    /** get the text shown as action in the first enquiry */
    protected abstract int getVoiceFormEnquiryActionIcon(Bean m_VoiceFormBean);

    /** get the bitmap to be shown in the background of the form */
    protected abstract Bitmap getVoiceFormBackground(Bean m_VoiceFormBean);

    /** get the text shown as action when asking for the attribute */
    protected abstract String getAttributeEnquiryAction(AttributeBean attributeBean);

    /** get the text shown as action when asking for the attribute  */
    protected abstract int getAttributeEnquiryActionIcon(AttributeBean attributeBean);

    /** get the text shown as action  when asking for the attribute  */
    protected abstract String getAttributeEnquiry(AttributeBean attributeBean);
    
    /** check whether the VoiceFormResult indicates that the user exceeded the limit  */
    protected abstract boolean isLimitExceeded(ResultBean result);

    /** check whether the VoiceFormResult indicates that it is completed */
    protected abstract boolean isCompleted(ResultBean result);

    @Override
    public IBinder onBind(Intent arg0) {return null;}
    
    @Override
    public void onCreate()
    {
        super.onCreate();
        
        Log.d(TAG, "onCreate");
        
        m_AsyncTaskTracker = new AsyncTaskTracker(this);
        
        m_taskRunForm = new TrackedAsyncTask(this, getAsyncTaskTracker(), KEY_RUNFORMTASK)
    	{

    		@Override
    		public void abortTask()
    		{
    			//when we should abort the task, 
    			m_taskRunForm.cancel(true);
    			onVoiceControlCanceled(VFE_TECHNICALABORT);
    		}

    		@Override
    		protected Boolean doInBackground(Void... params)
    		{
    			Log.d(TAG, "doInBackground");
    			
    			//start the recursive process, we have no result and no enquiry at the beginning 
    			ResultBean result = processStep(null,
    							getVoiceFormEnquiry(m_VoiceFormBean),
    							getVoiceFormEnquiryActionIcon(m_VoiceFormBean),
    							getVoiceFormEnquiryAction(m_VoiceFormBean));
    			
    			//return the result
    			if (isLimitExceeded(result))
    				onVoiceControlCanceled(VFE_LIMITEXCEEDED);
    			else
    				onVoiceControlCompleted(result);
    			
    			return null;
    		}

    		
    	};

    }
    
    private ResultBean processStep(ResultBean oldResult,
                                   String strOpenEnquiry,
                                   int iEnquiryActionIcon,
                                   String strEnquiryAction)
    {
    	Log.d(TAG, "processStep, with enquiry: "+strOpenEnquiry);
    	//get a merged result
    	ResultBean mergedResult = mergeResults(m_VoiceFormBean, oldResult,    			
    			//get a result of the current step 
    			callBackend(
    					
    					//ad the input to the form
    					addInputToVoiceForm(getVoiceControlInput(
    							
    							//getIcon 
    							getVoiceFormIcon(m_VoiceFormBean),
    							
    							//get the label of the VoiceForm
    							getVoiceFormLabel(m_VoiceFormBean),
    							
    							//get the enquiry for collecting the voice input
    							strOpenEnquiry,
    							
    							//the icon of the reply-action
    							iEnquiryActionIcon, 

    							//the text of the reply-action
    							strEnquiryAction,

                                //get the background of the VoiceForm
                                getVoiceFormBackground(m_VoiceFormBean)
    							
    					), m_VoiceFormBean)
    					
    			)
    	);
    	
    	//OpenEnquiryObject newOpenEnquiryObject = getOpenEnquiryObject();
    	
    	//is the voice form completed with the result, or do we have an open enquiry that we need to answer?
    	AttributeBean attributeBean = setOpenContextAttribute(m_VoiceFormBean, mergedResult);
    	
    	if (attributeBean == null) {
            if (isCompleted(mergedResult))
                return mergedResult;
            else
                return processStep(null,
                        getVoiceFormRepeatedEnquiry(m_VoiceFormBean),
                        getVoiceFormEnquiryActionIcon(m_VoiceFormBean),
                        getVoiceFormEnquiryAction(m_VoiceFormBean));

        }
    	else
    		//otherwise we do the next step... with regard to the OpenEnquiryObject
    		return processStep(mergedResult,
    				getAttributeEnquiry(attributeBean),
    				getAttributeEnquiryActionIcon(attributeBean),
    				getAttributeEnquiryAction(attributeBean));
    	    	
    }


	@Override
    public int onStartCommand(Intent intent, int flags, int startId)
    {
    	Log.d(TAG, "Received start id " + startId + ": " + intent);
    	
    	m_VoiceFormBean = createBeanFromBundle(intent.getBundleExtra(VoiceControlService.VOICEFORMBUNDLE));
    	//m_listAttributeBeans = createAttributeBeansListFromBundle(intent.getBundleExtra(VoiceControlService.ATTRIBUTESLISTBUNDLE));
    	
    	//start the voice form
    	m_taskRunForm.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR);
    	
    	// We want this service to continue running until it is explicitly stopped, so return sticky.
    	return START_STICKY; 
    }

	protected void onVoiceControlCanceled(int iErrorCode)
	{
		Log.d(TAG, "onVoiceControlCanceled");
		
		Intent intent = new Intent();
		intent.setAction(ACTION_VOICEFORM);
		intent.putExtra(RETURNCODE, iErrorCode);
		sendBroadcast(intent);
	}
	
	protected void onVoiceControlCompleted(ResultBean result)
	{
		Log.d(TAG, "onVoiceControlCompleted");
		Intent intent = new Intent();
		intent.setAction(ACTION_VOICEFORM);
		intent.putExtra(RETURNCODE, VF_COMPLETED);
		intent.putExtra(VOICEFORMRESULTBUNDLE, createBundleFromTemplateResultBean(result));
		sendBroadcast(intent);
	}
   
    @Override
    public void onDestroy()
    {
    	Log.d(TAG, "onDestroy");
        
    	
        
        super.onDestroy();
    }

    //TODO send broadcasts to the VoiceFormBuilder and let it notify the programmer, maybe he wants to to something with it
    @Override
	public void setCurrentProgressAnimator(ProgressAnimator animator){}

	@Override
	public void onAsyncTaskStarted(int newcount){}

	@Override
	public void onAsyncTaskStopped(int newcount){}

	@Override
	public void onAsyncTaskUpdated(String strMessage){}

	@Override
	public void showToast(String msg){}

	@Override
	public void showToast(int resourceId) {}

    
}
