package de.clued_up.voicecontrols.services;

import android.app.Notification;
import android.app.PendingIntent;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Bundle;
import android.support.v4.app.NotificationManagerCompat;
import android.util.Log;

import android.support.v4.app.NotificationCompat;
import android.support.v4.app.RemoteInput;
import android.support.v4.app.NotificationCompat.WearableExtender;

import de.clued_up.voicecontrols.Configuration;
import de.clued_up.voicecontrols.backend.apiVoiceForm.ApiVoiceForm;
import de.clued_up.voicecontrols.data.VoiceAttributeConfiguration;
import de.clued_up.voicecontrols.data.VoiceAttributeResultBundle;
import de.clued_up.voicecontrols.data.VoiceFormConfiguration;
import de.clued_up.voicecontrols.data.VoiceFormResultBundle;
import de.clued_up.voicecontrols.voicereply.ReplyActivity;
import de.clued_up.voicecontrols.voicereply.VoiceReplyIntentReceiver;
import de.clued_up.voicecontrols.voicereply.VoiceReplyListener;
import com.google.api.client.extensions.android.http.AndroidHttp;
import com.google.api.client.extensions.android.json.AndroidJsonFactory;
import com.google.api.client.googleapis.services.AbstractGoogleClientRequest;
import com.google.api.client.googleapis.services.GoogleClientRequestInitializer;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import de.clued_up.voicecontrols.backend.apiVoiceForm.model.VoiceFormBean;
import de.clued_up.voicecontrols.backend.apiVoiceForm.model.VoiceFormResultBean;

public class CallVoiceControlBackendService extends VoiceControlService<VoiceFormBean, VoiceFormResultBean, VoiceAttributeConfiguration>
{
    private static final int NOTIFICATIONID = 27101981;

    protected static final String TAG = CallVoiceControlBackendService.class.toString();
    
    private static ApiVoiceForm apiVoiceForm = null;

    private boolean m_bWaitingForInput = false;
    private String m_strReceivedInput = null;
    private NotificationManagerCompat notificationManager;

    private String m_strAppId;

    /** whenever the voice form requires input, a receives the answer using this broadcast receiver */
    protected VoiceReplyIntentReceiver m_VoiceReplyReceiver;


    /** counter for the input requests */
    protected int iCntInput = 0;

    
    @Override
    public void onCreate() {
        super.onCreate();

        m_strAppId = getAppId();

        notificationManager = NotificationManagerCompat.from(this);

        //create the receiver and register it
        m_VoiceReplyReceiver = new VoiceReplyIntentReceiver()
        {
            @Override
            protected void stopService()
            {
                Log.d(TAG, "stop service");
                CallVoiceControlBackendService.this.stopSelf();
            }
        }.register(this);

        Log.d(TAG, "onCreate");

    }

    public String getAppId() {
        PackageManager lPackageManager = getPackageManager();
        ApplicationInfo lApplicationInfo = null;
        try {
            lApplicationInfo = lPackageManager.getApplicationInfo(getApplicationInfo().packageName, 0);
        } catch (final PackageManager.NameNotFoundException e) {
        }

        return (String) (getApplicationContext().getPackageName() +
                lApplicationInfo != null ? lPackageManager.getApplicationLabel(lApplicationInfo) : "Unknown");
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        Log.d(TAG, "onStartCommand");

        return super.onStartCommand(intent, flags, startId);
    }
    
    @Override
    public void onDestroy()
    {
    	Log.d(TAG, "onDestroy");

        //close all old notifications
        notificationManager.cancel(NOTIFICATIONID);

        // Do not forget to unregister the receiver!!!
        unregisterReceiver(m_VoiceReplyReceiver);


        super.onDestroy();
    }

    @Override
    protected String getVoiceControlInput(int iIcon,
                                          String strLabel,
                                          String strEnquiry,
                                          int iEnquiryActionIcon,
                                          String strEnquiryAction,
                                          Bitmap bmpBackground)
    {
        Log.d(TAG, "getVoiceControlInput, with enquiry: "+strEnquiry);

        new VoiceReplyListener(m_VoiceReplyReceiver)
        {
            @Override
            public void onVoiceReplyReceived(String strReply)
            {
                Log.d(TAG, "received reply: "+strReply);

                //dismiss the notification
                notificationManager.cancel(NOTIFICATIONID);

                m_strReceivedInput = strReply;
                m_bWaitingForInput = false;

            }

            @Override
            public void onVoiceReplyFailed(){
                Log.d(TAG, "onVoiceReplyFailed");

                //not required since the service is stopped anyway
                //doing it would result in a further request that is not wanted
                //m_bWaitingForInput = false;
            }

        };

        m_bWaitingForInput = true;



        //setup the remote input
        RemoteInput remoteInput = new RemoteInput.Builder(VoiceReplyIntentReceiver.EXTRA_VOICE_REPLY)
                .setLabel(strEnquiryAction)
                .build();

        // Create an intent for the reply action
        Intent replyIntent = new Intent(this, ReplyActivity.class);
        PendingIntent replyPendingIntent = PendingIntent.getActivity(this, 0, replyIntent, PendingIntent.FLAG_UPDATE_CURRENT);


        // Create an intent for the dismissal
        Intent deleteIntent = new Intent();
        deleteIntent.setAction(VoiceReplyIntentReceiver.ACTION_VOICE_REPLY);
        deleteIntent.putExtra(VoiceReplyIntentReceiver.EXTRA_VOICE_CANCELLED, true);
        PendingIntent deletePendingIntent = PendingIntent.getBroadcast(this, 1, deleteIntent, PendingIntent.FLAG_UPDATE_CURRENT);

        // Create the reply action and add the remote input NotificationCompat.
        NotificationCompat.Action action = new NotificationCompat.Action.Builder(iEnquiryActionIcon, strEnquiryAction, replyPendingIntent)
                .addRemoteInput(remoteInput)
                .build();

        // Build the notification and add the action via WearableExtender
        Notification notification = new NotificationCompat.Builder(this)
                .setSmallIcon(iIcon)
                .setContentTitle(strLabel)
                .setContentText(strEnquiry)
                .setDeleteIntent(deletePendingIntent)
                .extend(new WearableExtender()
                    .addAction(action)
                    .setBackground(bmpBackground)
                )
                .build();

        // Issue the notification
        notificationManager.notify(NOTIFICATIONID, notification);

        //wait until we get a response
        while (m_bWaitingForInput)
        {
            try
            {
                Thread.sleep(300);
            }
            catch (InterruptedException e)
            {
                e.printStackTrace();
            }
        }



        return m_strReceivedInput;
    }

        /** implement with the call of the backend that analyzes ...
         * @param formBean ... into the specified VOiceForm ...
         */
	@Override
	protected VoiceFormResultBean callBackend(VoiceFormBean formBean){

        Log.d(TAG, "call backend");
		iCntInput++;

        // Only do this once
        if(apiVoiceForm == null)
        {
            if (Configuration.doConnectToLocalHost()) {
                ApiVoiceForm.Builder builder = new ApiVoiceForm.Builder(AndroidHttp.newCompatibleTransport(),
                        new AndroidJsonFactory(), null)
                        .setRootUrl("http://10.0.2.2:8080/_ah/api/")
                        .setGoogleClientRequestInitializer(new GoogleClientRequestInitializer() {
                            @Override
                            public void initialize(AbstractGoogleClientRequest<?> abstractGoogleClientRequest) throws IOException {
                                abstractGoogleClientRequest.setDisableGZipContent(true);
                            }
                        });

                apiVoiceForm = builder.build();
            }
            else
            {
                Log.d(TAG, "build the online connection");
                ApiVoiceForm.Builder builder = new ApiVoiceForm.Builder(
                        AndroidHttp.newCompatibleTransport(),
                        new AndroidJsonFactory(),
                        null)
                    .setRootUrl("https://voicecontrols.appspot.com/_ah/api/");
                apiVoiceForm = builder.build();
            }

        }


        // ******************************* use when in Android Studio *************************//
        try {
            return apiVoiceForm.analyzeVoiceForm(
                        m_strAppId,
                        formBean)
                    .setStrAccountId("accountId")
                    .execute();
        } catch (IOException e) {
            Log.e(TAG, e.getMessage());
            return null;
        }

        
	}

	
	/** create a merged resultBean of the given */
	@Override
	protected VoiceFormResultBean mergeResults(VoiceFormBean formBean, VoiceFormResultBean oldResultBean, VoiceFormResultBean currentResultBean)
	{
        Log.d(TAG, "mergeResults");
		//when we exceeded the limit, we do not need to do anything else but return that we exceeded the request limit
		if (currentResultBean != null &&
                currentResultBean.getResultType() == VoiceFormResultBundle.VFR_REQUESTLIMITEXCEEDED)
			return currentResultBean;
		
		//###################### create the lists of VoiceAttributeResultBundles ##############################################
		ArrayList<VoiceAttributeResultBundle> listOldResults = new ArrayList<VoiceAttributeResultBundle>();
		ArrayList<VoiceAttributeResultBundle> listCurrentResults = new ArrayList<VoiceAttributeResultBundle>();

        if (oldResultBean != null) {
            for (String strAttrResult :oldResultBean.getAttributeResults()) {
                Log.d(TAG, "add Attribute Result: "+strAttrResult);
                listOldResults.add(VoiceAttributeResultBundle.parseFromString(strAttrResult));
            }
        }

        if (currentResultBean != null) {

            Log.d(TAG, "Activity: "+currentResultBean.getEnteredActivity());

            final List<String> listAttributeResults = currentResultBean.getAttributeResults();
            if (listAttributeResults != null) {
                for (String strAttrResult :listAttributeResults)
                    listCurrentResults.add(VoiceAttributeResultBundle.parseFromString(strAttrResult));
            }
        }

        for (VoiceAttributeResultBundle attribute: listCurrentResults) {
            Log.d(TAG, "results of: "+attribute.getName());
            for (Object strResult: attribute.getResults())
                Log.d(TAG, "attributeresult: "+(String)strResult);
        }

		
		ArrayList<VoiceAttributeResultBundle> listNewResults = new ArrayList<VoiceAttributeResultBundle>();
		
		//merge the attribute results
		while (!listOldResults.isEmpty())
		{
			VoiceAttributeResultBundle oldAttributeResult = listOldResults.get(0);
			
			//find the corresponding current result
			VoiceAttributeResultBundle currentAttributeResult = null;
			for (int i=0; i<listCurrentResults.size();i++)
			{
				VoiceAttributeResultBundle current = listCurrentResults.get(i);
				if (oldAttributeResult.getName().compareTo(current.getName()) == 0 &&
						oldAttributeResult.getType() == current.getType())
				{
					currentAttributeResult = current;
					listCurrentResults.remove(i);
					break;
				}
			}
			
			//when we found a corresponding, ... combine both results
			if (currentAttributeResult != null)
			{
				//remove duplicates in the old list
				for (Object currentResult :currentAttributeResult.getResults())
				{
					for (int j=0;j<oldAttributeResult.getResults().size(); j++)
					{
						Object oldResult = oldAttributeResult.getResults().get(j);
							
						//TODO when we have other types, we need to add them here
						if (oldAttributeResult.getType() == VoiceAttributeConfiguration.VALUETYPE_LIST ||
								oldAttributeResult.getType() == VoiceAttributeConfiguration.VALUETYPE_FREETEXT)
						{
							if (((String) currentResult).compareTo(((String) oldResult)) == 0)
							{
								oldAttributeResult.getResults().remove(j);
								break;
							}
						}
					}
				}
				
				//add the remaining old results to the current one
				for (Object oldResult: oldAttributeResult.getResults())
					currentAttributeResult.getResults().add(oldResult);
				
				//put the current result into the new
				listNewResults.add(currentAttributeResult);
			}
			
			//when we did not find a corresponding result
			else
			{
				//just add the result to the new result
				listNewResults.add(oldAttributeResult);
			}
			
			
			listOldResults.remove(0);
		}
		
		//###################### check all existing attributes for (mandatory) completeness ##############################################
		
		boolean bAllMandatoryCompleted = true;
		boolean bAllCompleted = true;
		
		//loop through all attributes
		for (String strAttribute :formBean.getAttributes())
		{
			VoiceAttributeConfiguration voiceAttribute = VoiceAttributeConfiguration.parseFromString(strAttribute);
			
			boolean foundResult = false;
			//search for results:
			for (VoiceAttributeResultBundle result :listNewResults)
			{
				if (voiceAttribute.getName().compareTo(result.getName()) == 0 && result.getResults().size() > 0)
				{
					foundResult = true;
					break;
				}
				
			}
			
			//when we did not find a result, we're not completed
			if (!foundResult)
			{
				bAllCompleted = false;
				
				//when the incomplete attribute is mandatory, we don't need to continue 
				if (voiceAttribute.getIsMandatory())
				{
					bAllMandatoryCompleted = false;
					break;
				}
			}
		}
		
		
		//###################### combine the core data of the VoiceFormResult ##############################################
		
		//take the requested attribute of the currentResultBean
		//currentResultBean.getRequestedAttribute();
		
		//get the entered activity
		if (currentResultBean != null &&
                (currentResultBean.getEnteredActivity() == null || currentResultBean.getEnteredActivity().length() == 0))
		{
            if (oldResultBean != null)
			    currentResultBean.setEnteredActivity(oldResultBean.getEnteredActivity());
			
			//when we take the entered activity of the oldResultBean, check whether we should also take number and unit
			if (oldResultBean != null &&
                    (oldResultBean.getRequestedNumber() == null || oldResultBean.getRequestedNumber() != 0))
				currentResultBean.setRequestedNumber(oldResultBean.getRequestedNumber());
			
			if (oldResultBean != null &&
                    (oldResultBean.getRequestedUnit() == null || oldResultBean.getRequestedUnit().length() == 0))
				currentResultBean.setRequestedUnit(oldResultBean.getRequestedUnit());
		}	

		//when we have no requested number in the current result, take the old one
		if (currentResultBean != null && oldResultBean != null &&
                (currentResultBean.getRequestedNumber() == null || currentResultBean.getRequestedNumber() <= 0))
			currentResultBean.setRequestedNumber(oldResultBean.getRequestedNumber());
		
		//when we have no requested unit in the current result, take the old one
		if (currentResultBean != null && oldResultBean != null &&
                (currentResultBean.getRequestedUnit() == null || currentResultBean.getRequestedUnit().length() == 0))
			currentResultBean.setRequestedUnit(oldResultBean.getRequestedUnit());
		
		//when the current result is not yet completed, we need to recalculate the result
		if (currentResultBean != null && currentResultBean.getResultType() != VoiceFormResultBundle.VFR_COMPLETED)
		{
			//do we have an activity?
			if (currentResultBean.getEnteredActivity() != null && currentResultBean.getEnteredActivity().length() > 0)
			{
				//are all mandatory attributes filled?
				if (bAllMandatoryCompleted)
				{
					//are all fields are filled?
					if (bAllCompleted)
						currentResultBean.setResultType(VoiceFormResultBundle.VFR_COMPLETED);
					else
						currentResultBean.setResultType(VoiceFormResultBundle.VFR_MANDATORYCOMPLETED);
				}
				else
					//when we have open mandatory attributes
					currentResultBean.setResultType(VoiceFormResultBundle.VFR_INCOMPLETE);
			}
			else
				//when there is no entered activity, then we have no command
				currentResultBean.setResultType(VoiceFormResultBundle.VFR_INVALIDCOMMAND);
		}

		return currentResultBean;
	}

	@Override
	protected VoiceAttributeConfiguration setOpenContextAttribute(VoiceFormBean voiceFormBean, VoiceFormResultBean mergedResult)
	{
		//when we do not have a requested attribute, we do nothing but return null
		if (mergedResult == null || mergedResult.getRequestedAttribute() == null)
			return null;
		
		//when there is a requested attribute...
		Log.d(TAG, "requested attribute: "+mergedResult.getRequestedAttribute());
		
		
		//... set the context attribute in the voiceForm
		voiceFormBean.setContextAttribute(mergedResult.getRequestedAttribute());
		
		//... return the enquiry of the requested attribute
		return VoiceAttributeConfiguration.parseFromString(mergedResult.getRequestedAttribute());
		
	}

    
	@Override protected VoiceFormBean createBeanFromBundle(Bundle bundle){return VoiceFormConfiguration.createVoiceFormBeanFromBundle(bundle);}
	@Override protected Bundle createBundleFromTemplateResultBean(VoiceFormResultBean resultBean){
		return VoiceFormResultBundle.createBundleFromVoiceFormResultBean(resultBean);}
    @Override protected VoiceFormBean addInputToVoiceForm(String s, VoiceFormBean voiceFormBean) {return voiceFormBean.setInput(s);}
	@Override protected String getVoiceFormLabel(VoiceFormBean voiceForm) {return voiceForm.getLabel();}
	@Override protected String getVoiceFormEnquiry(VoiceFormBean voiceForm) {return voiceForm.getEnquiry();}
    @Override protected String getVoiceFormRepeatedEnquiry(VoiceFormBean voiceForm) {return voiceForm.getRepeatedEnquiry(); }

    @Override protected int getVoiceFormIcon(VoiceFormBean voiceForm) {return voiceForm.getIconResource();}
	@Override protected String getVoiceFormEnquiryAction(VoiceFormBean voiceForm){return voiceForm.getEnquiryAction();}
	@Override protected int getVoiceFormEnquiryActionIcon(VoiceFormBean voiceForm){return voiceForm.getEnquiryActionIconResource();}
    @Override protected Bitmap getVoiceFormBackground(VoiceFormBean voiceForm) {
        return BitmapFactory.decodeResource(getResources(), voiceForm.getBackgroundImageResource());}

    @Override protected String getAttributeEnquiryAction(VoiceAttributeConfiguration attributeConfiguration){return attributeConfiguration.getEnquiryAction();}
	@Override protected int getAttributeEnquiryActionIcon(VoiceAttributeConfiguration attributeConfiguration){return attributeConfiguration.getEnquiryActionIconResource();}
	@Override protected String getAttributeEnquiry(VoiceAttributeConfiguration attributeConfiguration){return attributeConfiguration.getEnquiry();}
	@Override protected boolean isLimitExceeded(VoiceFormResultBean resultBean){return resultBean == null || resultBean.getResultType() == VoiceFormResultBundle.VFR_REQUESTLIMITEXCEEDED;}
    @Override protected boolean isCompleted(VoiceFormResultBean result) {
        Log.d(TAG, "result: "+result.getResultType());
        return result.getResultType() <= VoiceFormResultBundle.VFR_MANDATORYCOMPLETED;
    }

}
