package de.chiflux.tesla;

import de.chiflux.tesla.api.FullVehicleData;
import de.chiflux.tesla.api.VehicleData;

import java.math.BigDecimal;
import java.math.RoundingMode;


/**
 * Basic common interface for vehicles
 */
public interface VehicleInfo {

    /**
     * Conversion factor to convert miles to kilometers
     */
    double MILE_TO_KM_FACTOR = 1.609344;

    /**
     * ID (id_s) information of the vehicle
     * @return ID (id_s) information of the vehicle
     */
    String id();

    /**
     * Constructs a user-friendly record of selected information of the vehicles API
     * @param vehicleData record of the vehicles API
     * @return record of selected information of the vehicles API
     */
    static BasicVehicleInfo ofVehicleData(VehicleData vehicleData) {
        String id = vehicleData.id_s();
        String state = vehicleData.state();
        String vehicleName = vehicleData.display_name();
        return new BasicVehicleInfo(id, vehicleName, state);
    }

    /**
     * Constructs a user-friendly record of selected information of the vehicleData API
     * @param fullVehicleData record of the vehicleData API
     * @return record of selected information of the vehicleData API
     */
    static ExtendedVehicleInfo ofFullVehicleData(FullVehicleData fullVehicleData) {
        String id = fullVehicleData.id_s();
        String vin = fullVehicleData.vin();
        Boolean locked = fullVehicleData.vehicle_state().locked();
        Integer carApiVersion = fullVehicleData.vehicle_state().api_version();
        String carVersion = fullVehicleData.vehicle_state().car_version();
        BigDecimal odometer = BigDecimal.valueOf(fullVehicleData.vehicle_state().odometer() * MILE_TO_KM_FACTOR)
                .setScale(1, RoundingMode.HALF_UP);
        Boolean isUserPresent = fullVehicleData.vehicle_state().is_user_present();

        String carSpecialType = fullVehicleData.vehicle_config().car_special_type();
        String carType = fullVehicleData.vehicle_config().car_type();
        String trimBadging = fullVehicleData.vehicle_config().trim_badging();
        Integer utcOffset = fullVehicleData.vehicle_config().utc_offset();

        String chargingState = fullVehicleData.charge_state().charging_state();
        Integer batteryLevel = fullVehicleData.charge_state().battery_level();
        Integer chargeLimitSoc = fullVehicleData.charge_state().charge_limit_soc();
        Double timeToFullCharge = fullVehicleData.charge_state().time_to_full_charge();
        Double batteryRangeVal = fullVehicleData.charge_state().battery_range();
        BigDecimal batteryRange = BigDecimal.valueOf((batteryRangeVal==null?0:batteryRangeVal) * MILE_TO_KM_FACTOR)
                .setScale(0, RoundingMode.HALF_UP);
        Double estimatedBatteryRangeVal = fullVehicleData.charge_state().est_battery_range();
        BigDecimal estimatedBatteryRange = BigDecimal.valueOf((estimatedBatteryRangeVal==null?0:estimatedBatteryRangeVal) * MILE_TO_KM_FACTOR)
                .setScale(0, RoundingMode.HALF_UP);
        Boolean chargePortDoorOpen = fullVehicleData.charge_state().charge_port_door_open();
        Integer usableBatteryLevel = fullVehicleData.charge_state().usable_battery_level();

        Integer heading = fullVehicleData.drive_state().heading();
        String shiftState = fullVehicleData.drive_state().shift_state();
        Double latitude = fullVehicleData.drive_state().latitude();
        Double longitude = fullVehicleData.drive_state().longitude();
        Integer power = fullVehicleData.drive_state().power();

        Double speedVal = fullVehicleData.drive_state().speed();
        BigDecimal speed = BigDecimal.valueOf((speedVal==null?0:speedVal) * MILE_TO_KM_FACTOR)
                .setScale(0, RoundingMode.HALF_UP);

        // battery size derive from trim badging
        String trimBadgingString = trimBadging.replaceAll("[^0-9]", "");
        Integer batterySize = Integer.parseInt(trimBadgingString.isBlank()?"50":trimBadgingString);

        // simple car version
        String simpleCarVersion = carVersion.substring(0, carVersion.indexOf(" "));

        return new ExtendedVehicleInfo(
                id,
                vin,locked,carApiVersion,carVersion,odometer,isUserPresent,
                carSpecialType,carType,trimBadging,utcOffset,chargingState,batteryLevel,chargeLimitSoc,timeToFullCharge,
                batteryRange, estimatedBatteryRange,
                chargePortDoorOpen,usableBatteryLevel,heading,shiftState,latitude,longitude,power,
                speed,batterySize,simpleCarVersion);
    }

}
