package de.chiflux.tesla;

import de.chiflux.tesla.api.FullVehicleData;
import de.chiflux.tesla.api.VehicleData;
import de.chiflux.tesla.api.VehiclesResponse;

import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Service Monitoring of all assigned vehicles of a tesla account.
 */
public class TeslaApiService implements TeslaApi {

    private static final Logger LOGGER = Logger.getLogger(TeslaLiveApi.class.getName());

    private static TeslaApiService INSTANCE = null;

    /**
     * Get the instance
     * @return the instance
     */
    public static synchronized TeslaApiService getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new TeslaApiService();
        }
        return INSTANCE;
    }

    private final Map<String, Instant> INACTIVE_MAP = new ConcurrentHashMap<>();

    /**
     * Will let a vehicle fall asleep
     * @param id the vehicle id
     */
    public void markVehicleInactive(String id) {
        if (!INACTIVE_MAP.containsKey(id)) {
            Instant now = Instant.now();
            INACTIVE_MAP.put(id, now);
            LOGGER.log(Level.FINE,"vehicle " + id + " marked INACTIVE instant=" + now);
            LOGGER.log(Level.FINE,"INACTIVE_MAP (2): " + INACTIVE_MAP);
        }
    }

    /**
     * Allow requesting the vehicle data API
     * @param id_s id of the vehicle
     */
    public void unmuteVehicle(String id_s) {
        if (id_s==null) {
            return;
        }
        INACTIVE_MAP.remove(id_s);
        LOGGER.info("UNMUTED vehicle " + id_s);
    }

    /**
     * Tests if a car is mute i.e., it will not query the vehicle data API to give the vehicle a chance to fall asleep.
     * @param id_s id of the vehicle
     * @return true if vehicle is mute
     */
    public boolean isMute(String id_s) {
        Instant instant = INACTIVE_MAP.get(id_s);
        if (instant != null) {
            Instant now = Instant.now();
            return now.isBefore(instant.plus(20, ChronoUnit.MINUTES))
                    && now.isAfter(instant.plus(3, ChronoUnit.MINUTES));
        }
        return false;
    }

    private final ConcurrentHashMap<String, Instant> LAST_BASIC_VEHICLE_INFO_TS = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, BasicVehicleInfo> LAST_BASIC_VEHICLE_INFO = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, Instant> LAST_EXTENDED_VEHICLE_INFO_TS = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, ExtendedVehicleInfo> LAST_EXTENDED_VEHICLE_INFO = new ConcurrentHashMap<>();

    private TeslaApiService() {
        teslaApi = TeslaApi.getLiveApi();
        Runnable task = () -> {
            try {
                if (!PAUSED.get()) {
                    Instant i1 = Instant.now();
                    VehiclesResponse vehicles = getVehicles();
                    Instant i2 = Instant.now();
                    if (LOGGER.isLoggable(Level.FINE)) {
                        LOGGER.log(Level.INFO, "getVehicles took: " + Duration.between(i1, i2) + ", data: " + vehicles);
                    }
                    List<VehicleData> vehicleDataList = vehicles.response();
                    for (VehicleData vehicleData : vehicleDataList) {
                        boolean stateChange = false;
                        BasicVehicleInfo basicVehicleInfo = VehicleInfo.ofVehicleData(vehicleData);
                        String id = basicVehicleInfo.id();
                        String state = basicVehicleInfo.state();
                        String vehicleName = basicVehicleInfo.vehicleName();
                        VehicleInfo oldBasicVehicleInfo = LAST_BASIC_VEHICLE_INFO.put(id, basicVehicleInfo);
                        LAST_BASIC_VEHICLE_INFO_TS.put(id, Instant.now());
                        if (!basicVehicleInfo.equals(oldBasicVehicleInfo)) {
                            stateChange = true;
                            unmuteVehicle(id);
                            LOGGER.log(Level.INFO, "basic state change detected");
                        }
                        if (state.equals(TeslaApiDefines.TESLA_STATE.online.name())) {
                            i1 = Instant.now();
                            FullVehicleData fullVehicleData = getFullVehicleData(id);
                            i2 = Instant.now();
                            if (LOGGER.isLoggable(Level.FINE)) {
                                LOGGER.log(Level.INFO, "getFullVehicleData from " + vehicleName + " took: " + Duration.between(i1, i2) + ", fullVehicleData: " + fullVehicleData);
                            }
                            handleSleepTry(vehicleData, fullVehicleData);
                            if (fullVehicleData==null) {
                                if (stateChange) {
                                    callStateChangeListeners(id);
                                }
                                return;
                            }
                            ExtendedVehicleInfo extendedVehicleInfo = VehicleInfo.ofFullVehicleData(fullVehicleData);
                            ExtendedVehicleInfo oldExtendedVehicleInfo = LAST_EXTENDED_VEHICLE_INFO.put(id, extendedVehicleInfo);
                            LAST_EXTENDED_VEHICLE_INFO_TS.put(id, Instant.now());
                            if (!extendedVehicleInfo.equals(oldExtendedVehicleInfo)) {
                                stateChange = true;
                                LOGGER.log(Level.INFO, "extended state change detected");
                            }
                        }
                        if (stateChange) {
                            callStateChangeListeners(id);
                        }
                    }
                }
            } catch (RuntimeException | InterruptedException e) {
                LOGGER.log(Level.INFO, e.getMessage());
                LOGGER.log(Level.FINE, e.getMessage(), e);
            }
        };
        executor.scheduleAtFixedRate(task, 2, 15, TimeUnit.SECONDS);
    }

    /**
     * Synchronous queue to safely exchange state change events over thread boundaries.
     */
    public final SynchronousQueue<StateChangeInfo> STATE_CHANGE_INFO_QUEUE = new SynchronousQueue<>();

    private void callStateChangeListeners(String id) throws InterruptedException {
        unmuteVehicle(id);
        BasicVehicleInfo basicVehicleInfo = LAST_BASIC_VEHICLE_INFO.get(id);
        Instant basicVehicleInfoTimestamp = LAST_BASIC_VEHICLE_INFO_TS.get(id);
        ExtendedVehicleInfo extendedVehicleInfo = LAST_EXTENDED_VEHICLE_INFO.get(id);
        Instant extendedVehicleInfoTimestamp = LAST_EXTENDED_VEHICLE_INFO_TS.get(id);
        if (basicVehicleInfo!=null || extendedVehicleInfo!=null) {
            StateChangeInfo stateChangeInfo = new StateChangeInfo(
                    basicVehicleInfoTimestamp, basicVehicleInfo,
                    extendedVehicleInfoTimestamp, extendedVehicleInfo);
            STATE_CHANGE_INFO_QUEUE.put(stateChangeInfo);
        }
    }

    private void handleSleepTry(VehicleData vehicleData, FullVehicleData fullVehicleData) {
        if (vehicleData==null || fullVehicleData==null) {
            return;
        }
        String state = vehicleData.state();
        String id = vehicleData.id_s();
        String vehicleName = vehicleData.display_name();
        boolean tryToSleep = false;
        if (state.equals(TeslaApiDefines.TESLA_STATE.online.name())) {
            Boolean locked = fullVehicleData.vehicle_state().locked();
            String shiftState = fullVehicleData.drive_state().shift_state();
            boolean shiftStateParked = shiftState==null || shiftState.equals("P");
            LOGGER.log(Level.FINE, "locked: " + locked);
            LOGGER.log(Level.FINE, "shiftStateParked: " + shiftStateParked);
            boolean charging = fullVehicleData.charge_state().charging_state().equalsIgnoreCase(TeslaApiDefines.CHARGE_STATE.Charging.name());
            if (shiftStateParked && locked && !charging) {
                tryToSleep = true;
                LOGGER.log(Level.FINE,"INACTIVE_MAP (1): " + INACTIVE_MAP);
                if (!INACTIVE_MAP.containsKey(id)) {
                    LOGGER.log(Level.INFO, "Vehicle is parked and locked => try to let car fall asleep: " + id + " (" + vehicleName + ") at " + Instant.now());
                    markVehicleInactive(id);
                }
            }
        }
        if (!tryToSleep) {
            LOGGER.log(Level.FINE,"removing car " + id + " (" + vehicleName + ") from try to fall asleep list");
            unmuteVehicle(id);
        }
    }

    private static final ScheduledExecutorService executor = Executors.newScheduledThreadPool(1);

    private static final AtomicBoolean PAUSED = new AtomicBoolean(true);

    /**
     * Starts the monitoring service
     */
    public void startService() {
        PAUSED.set(false);
    }

    /**
     * Pauses the monitoring service
     */
    public void pauseService() {
        PAUSED.set(true);
    }

    /**
     * Ends the monitoring service (cannot be restarted afterward)
     */
    public void endService() {
        executor.shutdown();
    }

    private final TeslaApi teslaApi;

    @Override
    public Boolean startCharge(String id_s) {
        return teslaApi.startCharge(id_s);
    }

    @Override
    public Boolean stopCharge(String id_s) {
        return teslaApi.stopCharge(id_s);
    }

    @Override
    public Boolean setChargeLimit(String id_s, int chargeLimit) {
        return teslaApi.setChargeLimit(id_s, chargeLimit);
    }

    @Override
    public Boolean wakeUp(String id_s) {
        unmuteVehicle(id_s);
        return teslaApi.wakeUp(id_s);
    }

    @Override
    public FullVehicleData getFullVehicleData(String id) {
        if (isMute(id)) {
            LOGGER.info("car " + id + " is muted!");
            return null;
        }
        return teslaApi.getFullVehicleData(id);
    }

    @Override
    public VehiclesResponse getVehicles() {
        return teslaApi.getVehicles();
    }

    @Override
    public void refreshToken() {
        teslaApi.refreshToken();
    }

}
