package de.chiflux.tesla;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import de.chiflux.tesla.api.ChargeLimitRequest;
import de.chiflux.tesla.api.CmdRespose;
import de.chiflux.tesla.api.FullVehicleData;
import de.chiflux.tesla.api.FullVehicleDataResponse;
import de.chiflux.tesla.api.TokenRefreshRequest;
import de.chiflux.tesla.api.TokenRefreshResponse;
import de.chiflux.tesla.api.VehicleData;
import de.chiflux.tesla.api.VehiclesResponse;
import de.chiflux.tesla.api.WakeUpResponse;
import de.chiflux.utils.ApiRateLimiter;
import okhttp3.Call;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * A thin wrapper around the Tesla API functionality supporting charging.
 * It makes use of these APIs:
 * /api/1/vehicles, /ID/vehicle_data, /ID/wake_up, /ID/command/set_charge_limit, /ID/command/charge_*
 */
public class TeslaLiveApi implements TeslaApi {

    private static final String TESLA_REFRESH_TOKEN = "TESLA_REFRESH_TOKEN";
    private final ApiRateLimiter rateLimiter = new ApiRateLimiter(1, Duration.ofSeconds(5), Duration.ofSeconds(2));

    private static final Logger LOGGER = Logger.getLogger(TeslaLiveApi.class.getName());
    private static final String REFRESH_URL = "https://auth.tesla.com/oauth2/v3/token";
    private static final String BASE_URL = "https://owner-api.teslamotors.com";
    private static final String VEHICLES_URL = BASE_URL + "/api/1/vehicles";
    private static final String GET_VEHICLE_DATA_URL = VEHICLES_URL + "/CARID/vehicle_data";
    private static final String POST_WAKE_UP_URL = VEHICLES_URL + "/CARID/wake_up";
    private static final String POST_CHARGE_LIMIT_URL = VEHICLES_URL + "/CARID/command/set_charge_limit";
    private static final String POST_CHARGE = VEHICLES_URL + "/CARID/command/charge_";

    private static final RequestBody EMPTY_JSON_REQUEST_BODY = RequestBody.create("{}".getBytes(StandardCharsets.UTF_8), MediaType.parse("application/json"));

    private final OkHttpClient client = new OkHttpClient();

    private final Path teslaSettings = Paths.get("tesla_settings.json");

    private final String teslaRefreshToken;

    private TeslaToken teslaToken = null;


    /**
     * Explicit constructor that takes the TESLA_REFRESH_TOKEN as parameter.
     * @param teslaRefreshToken The TESLA_REFRESH_TOKEN as parameter - must not be null
     */
    public TeslaLiveApi(String teslaRefreshToken) {
        if (teslaRefreshToken==null) {
            throw new IllegalArgumentException("teslaRefreshToken must not be null");
        }
        this.teslaRefreshToken = teslaRefreshToken;
    }

    /**
     * Default constructor which reads TESLA_REFRESH_TOKEN from Java Properties or ENV Variable.
     * A Java Property has precedence over an ENV variable.
     */
    public TeslaLiveApi() {
        String property = System.getProperty(TESLA_REFRESH_TOKEN);
        if (property==null) {
            property = System.getenv(TESLA_REFRESH_TOKEN);
        }
        if (property==null) {
            throw new IllegalStateException("TESLA_REFRESH_TOKEN must not be null");
        }
        this.teslaRefreshToken = property;
    }


    @Override
    public Boolean startCharge(String id_s) {
        return charge(id_s, "start");
    }

    @Override
    public Boolean stopCharge(String id_s) {
        return charge(id_s, "stop");
    }

    private Boolean charge(String id_s, String cmd) {
        try {
            refreshToken();
            String url = (POST_CHARGE + cmd).replace("CARID", id_s);
            ObjectMapper objectMapper = new ObjectMapper();
            Request.Builder builder = new Request.Builder();
            builder.addHeader("Authorization", "Bearer " + teslaToken.accessToken());
            Request request = builder.url(url).post(EMPTY_JSON_REQUEST_BODY).build();
            Call call = client.newCall(request);
            try (Response response = call.execute()) {
                int code = response.code();
                if (code<200 || code>=300) {
                    LOGGER.warning("Error charge vehicle status code: " + code);
                    throw new IllegalStateException("code=" + code);
                }
                return processCMDResponse(objectMapper, response);
            }
        } catch (Exception e) {
            LOGGER.log(Level.WARNING, e.getMessage(), e);
            return false;
        }
    }

    private Boolean processCMDResponse(ObjectMapper objectMapper, Response response) throws IOException {
        byte[] bytes = handleResponse(objectMapper, response);
        CmdRespose cmdResponse = objectMapper.readValue(bytes, CmdRespose.class);
        LOGGER.finer("CmdResponse: " + cmdResponse);
        return cmdResponse.response().result();
    }

    @Override
    public Boolean setChargeLimit(String id_s, int chargeLimit) {
        if (id_s==null || id_s.isBlank()) {
            LOGGER.warning("setChargeLimit id_s is not set");
            return false;
        }
        try {
            refreshToken();
            String url = POST_CHARGE_LIMIT_URL.replace("CARID", id_s);
            ObjectMapper objectMapper = new ObjectMapper();
            Request.Builder builder = new Request.Builder();
            builder.addHeader("Authorization", "Bearer " + teslaToken.accessToken());
            String json = objectMapper.writeValueAsString(new ChargeLimitRequest(chargeLimit));
            RequestBody requestBody = RequestBody.create(json.getBytes(StandardCharsets.UTF_8), MediaType.parse("application/json"));
            Request request = builder.url(url).post(requestBody).build();
            Call call = client.newCall(request);
            try (Response response = call.execute()) {
                int code = response.code();
                if (code<200 || code>=300) {
                    LOGGER.warning("Error setChargeLimit to " + chargeLimit + " status code: " + code);
                    throw new IllegalStateException("code=" + code);
                }
                return processCMDResponse(objectMapper, response);
            }
        } catch (Exception e) {
            LOGGER.log(Level.WARNING, e.getMessage(), e);
            return false;
        }
    }


    @Override
    public Boolean wakeUp(String id_s) {
        try {
            refreshToken();
            String url = POST_WAKE_UP_URL.replace("CARID", id_s);
            ObjectMapper objectMapper = new ObjectMapper();
            Request.Builder builder = new Request.Builder();
            builder.addHeader("Authorization", "Bearer " + teslaToken.accessToken());
            Request request = builder.url(url).post(EMPTY_JSON_REQUEST_BODY).build();
            Call call = client.newCall(request);
            try (Response response = call.execute()) {
                int code = response.code();
                if (code<200 || code>=300) {
                    LOGGER.warning("Error waking up vehicle status code: " + code);
                } else {
                    byte[] bytes = handleResponse(objectMapper, response);
                    WakeUpResponse wakeUpResponse = objectMapper.readValue(bytes, WakeUpResponse.class);
                    LOGGER.finer("WakeUpResponse: " + wakeUpResponse);
                    String state = wakeUpResponse.response().state();
                    if (!state.equals("asleep")) {
                        return true;
                    }
                }
            }
        } catch (Exception e) {
            LOGGER.log(Level.WARNING, e.getMessage(), e);
        }
        return false;
    }

    private void waitForWakeupState(String id_s, long timeout) {
        synchronized (this) {
            final long t1 = System.currentTimeMillis();
            VehicleData vehicleData;
            String state;
            do {
                final VehiclesResponse vehicles = getVehicles();
                vehicleData = vehicles.response().stream()
                        .filter(vd -> id_s.equals(vd.id_s())).findFirst().orElse(null);
                state = vehicleData != null ? vehicleData.state() : null;
            } while (System.currentTimeMillis() - t1 < timeout * 1_000 && state != null && !state.equals("asleep"));
        }
    }

    @Override
    public FullVehicleData getFullVehicleData(String id_s) {
        try {
            refreshToken();
            ObjectMapper objectMapper = new ObjectMapper();
            Request.Builder builder = new Request.Builder();
            builder.addHeader("Authorization", "Bearer " + teslaToken.accessToken());
            String url = GET_VEHICLE_DATA_URL.replace("CARID", id_s);
            LOGGER.fine("query url: " + url);
            Request request = builder.url(url).get().build();
            Call call = client.newCall(request);
            try (Response response = call.execute()) {
                int code = response.code();
                if (code<200 || code>=300) {
                    if (code==408) {
                        LOGGER.log(Level.WARNING, "status code 408");
                    }
                    if (code==429) {
                        int slowdownInPercent = 10;
                        LOGGER.log(Level.WARNING, "status code 429: slowing down request frequency by " + slowdownInPercent + "%");
                        rateLimiter.slowdown(slowdownInPercent);
                    }
                    return null;
                }
                byte[] bytes = handleResponse(objectMapper, response);
                FullVehicleDataResponse fullVehicleDataResponse = objectMapper.readValue(bytes, FullVehicleDataResponse.class);
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.log(Level.FINE, "unmarshalled json response: " + fullVehicleDataResponse);
                }
                return fullVehicleDataResponse.response();
            }
        } catch (Exception e) {
            LOGGER.log(Level.WARNING, e.getMessage(), e);
        }
        return null;
    }

    @Override
    public VehiclesResponse getVehicles() {
        try {
            refreshToken();
            ObjectMapper objectMapper = new ObjectMapper();
            Request.Builder builder = new Request.Builder();
            builder.addHeader("Authorization", "Bearer " + teslaToken.accessToken());
            Request request = builder.url(VEHICLES_URL).get().build();
            Call call = client.newCall(request);
            try (Response response = call.execute()) {
                int code = response.code();
                if (code<200 || code>=300) {
                    if (code==408) {
                        LOGGER.log(Level.WARNING, "status code 408");
                    }
                    if (code==429) {
                        int slowdownInPercent = 10;
                        LOGGER.log(Level.WARNING, "status code 429: slowing down request frequency by " + slowdownInPercent + "%");
                        rateLimiter.slowdown(slowdownInPercent);
                    }
                    throw new IllegalStateException("code=" + code);
                }
                byte[] bytes = handleResponse(objectMapper, response);
                VehiclesResponse vehiclesResponse = objectMapper.readValue(bytes, VehiclesResponse.class);
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.log(Level.FINE, "unmarshalled json response: " + vehiclesResponse);
                }
                return vehiclesResponse;
            }
        } catch (Exception e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }

    private byte[] handleResponse(ObjectMapper objectMapper, Response response) throws IOException {
        ResponseBody responseBody = response.body();
        if (responseBody == null) {
            throw new IllegalStateException("responseBody is null");
        }
        byte[] bytes = responseBody.bytes();
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.log(Level.FINE, "json response: " + new String(bytes, StandardCharsets.UTF_8));
        }
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        return bytes;
    }

    @Override
    public synchronized void refreshToken() {
        rateLimiter.acquireWait();
        try {
            if (teslaToken != null && teslaToken.isValid()) {
                return;
            }
            ObjectMapper objectMapper = new ObjectMapper();
            if (Files.exists(teslaSettings)) {
                try {
                    byte[] bytes = Files.readAllBytes(teslaSettings);
                    TeslaToken teslaToken = objectMapper.readValue(bytes, TeslaToken.class);
                    if (teslaToken.isValid()) {
                        this.teslaToken = teslaToken;
                        return;
                    }
                } catch (Exception e) {
                    LOGGER.log(Level.WARNING, e.getMessage(), e);
                }
            }
            rateLimiter.acquireWait();
            Request.Builder builder = new Request.Builder();
            String json = objectMapper.writeValueAsString(new TokenRefreshRequest(teslaRefreshToken));
            RequestBody requestBody = RequestBody.create(json.getBytes(StandardCharsets.UTF_8), MediaType.parse("application/json"));
            Request request = builder.url(REFRESH_URL).post(requestBody).build();
            Call call = client.newCall(request);
            try (Response response = call.execute()) {
                ResponseBody responseBody = response.body();
                if (responseBody == null) {
                    throw new IllegalStateException("responseBody is null");
                }
                byte[] bytes = responseBody.bytes();
                TokenRefreshResponse tokenRefreshResponse = objectMapper.readValue(bytes, TokenRefreshResponse.class);
                TeslaToken teslaToken = new TeslaToken(tokenRefreshResponse.access_token(), tokenRefreshResponse.expires());
                this.teslaToken = teslaToken;
                Files.writeString(teslaSettings, objectMapper.writeValueAsString(teslaToken));
            }
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }


}
