package de.chiflux.utils;

/**
 * Represents geographic coordinates along with distance calculation.
 * @param latitude latitude
 * @param longitude longitude
 */
public record GeoCoords2D(double latitude, double longitude) {

    private static final double EARTH_RADIUS = 6371_000; // Radius of the Earth in meters

    /**
     * Calculates the spatial distance in meter between the coordinates.
     * @param source source coordinate
     * @param target target coordinate
     * @return the distance in meter between the coordinates
     */
    public static double calculateDistance(GeoCoords2D source, GeoCoords2D target) {
        return calculateDistance(source.latitude, source.longitude, target.latitude, target.longitude);
    }

    /**
     * Calculates the spatial distance in meter between the coordinates.
     * @param source source coordinate
     * @param targetLatitude target latitude coordinate
     * @param targetLongitude target longitude coordinate
     * @return the distance in meter between the coordinates
     */
    public static double calculateDistance(GeoCoords2D source, double targetLatitude, double targetLongitude) {
        return calculateDistance(source.latitude, source.longitude, targetLatitude, targetLongitude);
    }

    /**
     * Calculates the spatial distance in meter between the coordinates.
     * @param sourceLatitude source latitude coordinate
     * @param sourceLongitude source longitude coordinate
     * @param targetLatitude target latitude coordinate
     * @param targetLongitude target longitude coordinate
     * @return the distance in meter between the coordinates
     */
    public static double calculateDistance(double sourceLatitude, double sourceLongitude, double targetLatitude, double targetLongitude) {
        // Convert latitude and longitude from degrees to radians
        double lat1Rad = Math.toRadians(sourceLatitude);
        double lon1Rad = Math.toRadians(sourceLongitude);
        double lat2Rad = Math.toRadians(targetLatitude);
        double lon2Rad = Math.toRadians(targetLongitude);

        // Calculate the differences between the two coordinates
        double deltaLat = lat2Rad - lat1Rad;
        double deltaLon = lon2Rad - lon1Rad;

        // Haversine formula
        double a = Math.pow(Math.sin(deltaLat / 2), 2) +
                Math.cos(lat1Rad) * Math.cos(lat2Rad) *
                        Math.pow(Math.sin(deltaLon / 2), 2);
        double c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));

        return EARTH_RADIUS * c;
    }

}
